package fr.ifremer.tutti.ui.swing.spatial;

/*
 * #%L
 * Tutti :: UI
 * $Id: DmsCoordinateEditorHandler.java 1316 2013-10-25 16:43:35Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-2.8/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/spatial/DmsCoordinateEditorHandler.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import fr.ifremer.tutti.TuttiTechnicalException;
import jaxx.runtime.swing.editor.bean.BeanUIUtil;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.swing.JFormattedTextField;
import javax.swing.text.DefaultFormatterFactory;
import javax.swing.text.MaskFormatter;
import java.awt.event.KeyEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.lang.reflect.Method;
import java.text.ParseException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Created on 10/16/13.
 *
 * @author Tony Chemit <chemit@codelutin.com>
 * @since 2.8
 */
public class DmsCoordinateEditorHandler {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(DmsCoordinateEditorHandler.class);

    protected static final Pattern VALUE_PATTERN =
            Pattern.compile("(.*)°(.*)'(.*)''");

    private final DmsCoordinateEditor ui;

    protected Method signMutator;

    protected Method degreMutator;

    protected Method minuteMutator;

    protected Method secondMutator;

    protected DefaultFormatterFactory unsignedFactory;

    protected DefaultFormatterFactory signedFactory;

    protected boolean valueIsAdjusting;

    protected boolean valueModelIsAdjusting;

    public DmsCoordinateEditorHandler(DmsCoordinateEditor ui) {
        this.ui = ui;
    }

    public void init() {

        final DmsCoordinateEditorModel model = ui.getModel();

        Preconditions.checkNotNull(model.getBean(), "could not find bean in " + ui);
        Preconditions.checkNotNull(model.getPropertySign(), "could not find propertySign in " + ui);
        Preconditions.checkNotNull(model.getPropertyDegree(), "could not find propertyDegree in " + ui);
        Preconditions.checkNotNull(model.getPropertyMinute(), "could not find propertyMinute in " + ui);
        Preconditions.checkNotNull(model.getPropertySecond(), "could not find propertySecond in " + ui);

        Object bean = model.getBean();
        signMutator = BeanUIUtil.getMutator(bean, model.getPropertySign());
        Preconditions.checkNotNull(signMutator, "could not find mutator for " + model.getPropertySign());

        degreMutator = BeanUIUtil.getMutator(bean, model.getPropertyDegree());
        Preconditions.checkNotNull(degreMutator, "could not find mutator for " + model.getPropertyDegree());

        minuteMutator = BeanUIUtil.getMutator(bean, model.getPropertyMinute());
        Preconditions.checkNotNull(minuteMutator, "could not find mutator for " + model.getPropertyMinute());

        secondMutator = BeanUIUtil.getMutator(bean, model.getPropertySecond());
        Preconditions.checkNotNull(secondMutator, "could not find mutator for " + model.getPropertySecond());

        MaskFormatter unsignedFormatter;
        try {

            String pattern = model.getMaskFormatterPattern();
            unsignedFormatter = new MaskFormatter(pattern);
            unsignedFormatter.setValidCharacters(" 01234567890");
        } catch (ParseException e) {
            // can't happen here
            throw new TuttiTechnicalException(e);
        }
        unsignedFactory = new DefaultFormatterFactory(unsignedFormatter);

        MaskFormatter signedFormatter;
        try {

            String pattern = model.getMaskFormatterPattern();
            signedFormatter = new MaskFormatter("-" + pattern);
            signedFormatter.setValidCharacters(" 01234567890");
        } catch (ParseException e) {
            // can't happen here
            throw new TuttiTechnicalException(e);
        }
        signedFactory = new DefaultFormatterFactory(signedFormatter);

        JFormattedTextField editor = ui.getEditor();
        editor.setFormatterFactory(unsignedFactory);
        editor.setFocusLostBehavior(JFormattedTextField.COMMIT);

        // When editor changes his value, propagate it to model
        editor.addPropertyChangeListener("value", new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                String newValue = (String) evt.getNewValue();
                if (log.isDebugEnabled()) {
                    log.debug("Value has changed: " + newValue);
                }
                DmsCoordinate value = null;
                if (newValue != null) {

                    Matcher matcher = VALUE_PATTERN.matcher(newValue);

                    if (matcher.matches()) {

                        String degresStr = matcher.group(1).replaceAll("\\s","");
                        String minutesStr = matcher.group(2).replaceAll("\\s", "");
                        String secondsStr = matcher.group(3).replaceAll("\\s", "");

                        Integer degre = degresStr.isEmpty() || "-".equals(degresStr) ? 0 : Integer.valueOf(degresStr);
                        Integer minutes = minutesStr.isEmpty() ? 0 : Integer.valueOf(minutesStr);
                        Integer seconds = secondsStr.isEmpty() ? 0 : Integer.valueOf(secondsStr);

                        boolean signed = degre < 0;
                        value =
                                DmsCoordinate.valueOf(signed,
                                                      Math.abs(degre),
                                                      minutes,
                                                      seconds);
                    }
                }
                model.setValue(value);
            }
        });

        // When model sign changed, let's push it back in bean
        model.addPropertyChangeListener(
                DmsCoordinateEditorModel.PROPERTY_SIGN,
                new ModelPropertyChangeListener(model, signMutator));

        // When model degre changed, let's push it back in bean
        model.addPropertyChangeListener(
                DmsCoordinateEditorModel.PROPERTY_DEGREE,
                new ModelPropertyChangeListener(model, degreMutator));

        // When model minute changed, let's push it back in bean
        model.addPropertyChangeListener(
                DmsCoordinateEditorModel.PROPERTY_MINUTE,
                new ModelPropertyChangeListener(model, minuteMutator));

        // When model second changed, let's push it back in bean
        model.addPropertyChangeListener(
                DmsCoordinateEditorModel.PROPERTY_SECOND,
                new ModelPropertyChangeListener(model, secondMutator));
    }

    public void setValue(DmsCoordinate value, boolean pushToModel) {

        if (valueModelIsAdjusting) {
            // avoid re-entrant code
            return;
        }

        String signStr = "";
        String degreeStr = "";
        String minuteStr = "";
        String secondStr = "";
        if (value != null) {

            boolean sign = value.isSign();
            signStr = sign ? "-" : "";

            Integer degree = value.getDegree();
            degreeStr = degree == null ? "" : degree.toString();

            Integer minute = value.getMinute();
            minuteStr = minute == null ? "" : minute.toString();

            Integer second = value.getSecond();
            secondStr = second == null ? "" : second.toString();
        }

        DmsCoordinateEditorModel model = ui.getModel();
        String stringPattern = model.getStringPattern();
        String valueStr = String.format(
                stringPattern,
                signStr,
                StringUtils.leftPad(degreeStr, model.isLongitudeEditor() ? 3 : 2, ' '),
                StringUtils.leftPad(minuteStr, 2, ' '),
                StringUtils.leftPad(secondStr, 2, ' '));

        valueIsAdjusting = !pushToModel;

        try {
            ui.getEditor().setValue(valueStr);
        } finally {
            valueIsAdjusting = false;
        }
    }

    public void resetEditor() {
        // set null value to model
        setValue(null, true);

        // use back unsigned format
        ui.getEditor().setFormatterFactory(unsignedFactory);
    }

    public void onKeyReleased(KeyEvent e) {

        JFormattedTextField source = (JFormattedTextField) e.getSource();

        char keyChar = e.getKeyChar();
        int caretPosition = source.getCaretPosition();
        if (log.isDebugEnabled()) {
            log.debug("Key pressed: " + keyChar + " (caret position: " + caretPosition + ")");
        }

        String newValue = null;
        DefaultFormatterFactory newFactory = null;
        if (keyChar == '-') {

            // try to switch unsigned to signed

            if (unsignedFactory == source.getFormatterFactory()) {

                // switch to signed
                if (log.isDebugEnabled()) {
                    log.debug("Switch to signed");
                }
                newFactory = signedFactory;

                try {
                    source.commitEdit();
                } catch (ParseException e1) {
                    // ignore it
                }

                newValue = "-" + source.getValue();

            } else {
                // switch to unsigned

                if (log.isDebugEnabled()) {
                    log.debug("Switch to unsigned");
                }
                newFactory = unsignedFactory;

                try {
                    source.commitEdit();
                } catch (ParseException e1) {
                    // ignore it
                }

                newValue = ((String) source.getValue()).substring(1);
            }
        } else {
            try {
                source.commitEdit();
            } catch (ParseException e1) {
                // ignore it
            }
            if (log.isDebugEnabled()) {
                log.debug("Key pressed: newValue " + source.getValue());
            }
        }

        if (newFactory != null) {

            // consume event (prevent any side-effects)
            e.consume();


            if (log.isDebugEnabled()) {
                log.debug("Key pressed: newValue " + newValue);
            }
            source.setFormatterFactory(newFactory);
            source.setValue(newValue);

            if (unsignedFactory == newFactory) {

                // remove a sign
                caretPosition--;
            } else {

                // add a sign
                caretPosition++;
            }

            source.setCaretPosition(caretPosition);
        }
    }

    private class ModelPropertyChangeListener implements PropertyChangeListener {

        private final DmsCoordinateEditorModel model;

        private final Method mutator;

        private ModelPropertyChangeListener(DmsCoordinateEditorModel model, Method mutator) {
            this.model = model;
            this.mutator = mutator;
        }

        @Override
        public void propertyChange(PropertyChangeEvent evt) {
            if (!valueIsAdjusting) {
                Object newValue = evt.getNewValue();

                try {

                    valueModelIsAdjusting = true;
                    try {
                        mutator.invoke(model.getBean(), newValue);
                    } finally {
                        valueModelIsAdjusting = false;
                    }

                } catch (Exception e) {
                    throw new RuntimeException(e);
                }
            }
        }
    }
}
