package fr.ifremer.tutti.ui.swing.content.cruise;

/*
 * #%L
 * Tutti :: UI
 * $Id: EditCruiseUIHandler.java 447 2013-02-18 15:17:03Z kmorin $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.0.1/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/cruise/EditCruiseUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Lists;
import fr.ifremer.tutti.persistence.entities.data.Cruise;
import fr.ifremer.tutti.persistence.entities.data.Program;
import fr.ifremer.tutti.persistence.entities.referential.Gear;
import fr.ifremer.tutti.persistence.entities.referential.Person;
import fr.ifremer.tutti.persistence.entities.referential.Vessel;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.ui.swing.AbstractTuttiUIHandler;
import fr.ifremer.tutti.ui.swing.TuttiScreen;
import fr.ifremer.tutti.ui.swing.TuttiUI;
import jaxx.runtime.JAXXUtil;
import jaxx.runtime.context.JAXXContextEntryDef;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.swing.JOptionPane;
import java.awt.event.ActionEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import static org.nuiton.i18n.I18n._;

/**
 * Handler of UI {@link EditCruiseUI}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class EditCruiseUIHandler extends AbstractTuttiUIHandler<EditCruiseUIModel, EditCruiseUI> {

    /** Logger. */
    private static final Log log = LogFactory.getLog(EditCruiseUIHandler.class);

    /**
     * Persistence service.
     *
     * @since 0.1
     */
    private final PersistenceService persistenceService;

    public EditCruiseUIHandler(TuttiUI parentUi, EditCruiseUI ui) {
        super(parentUi.getHandler().getContext(), ui);
        persistenceService = context.getService(PersistenceService.class);
    }

    @Override
    public void beforeInitUI() {

        EditCruiseUIModel model = new EditCruiseUIModel();

        // get vessels

        List<Vessel> fishingVessels = persistenceService.getAllFishingVessel();
        FISHING_VESSELS_CONTEXT_ENTRY.setContextValue(ui, fishingVessels);

        List<Vessel> scientificVessels = persistenceService.getAllScientificVessel();
        SCIENTIFIC_VESSELS_CONTEXT_ENTRY.setContextValue(ui, scientificVessels);

        // get gears

        List<Gear> scientificGears = persistenceService.getAllScientificGear();
        SCIENTIFIC_GEARS_CONTEXT_ENTRY.setContextValue(ui, scientificGears);

        List<Gear> fishingGears = persistenceService.getAllFishingGear();
        FISHING_GEARS_CONTEXT_ENTRY.setContextValue(ui, fishingGears);

        String cruiseId = context.getCruiseId();
        if (cruiseId == null) {

            // create new cruise
            Program program =
                    persistenceService.getProgram(context.getProgramId());
            model.setProgram(program);
            model.setMultirigNumber(1);

            if (log.isDebugEnabled()) {
                log.debug("Will create a new cruise from program: " + program);
            }
        } else {

            // load existing program
            Cruise cruise =
                    persistenceService.getCruise(cruiseId);

            model.fromBean(cruise);
            if (log.isDebugEnabled()) {
                log.debug("Will edit cruise: " + cruiseId);
            }
        }
        listModelIsModify(model);
        ui.setContextValue(model);
    }

    JAXXContextEntryDef<List<Vessel>> SCIENTIFIC_VESSELS_CONTEXT_ENTRY = JAXXUtil.newListContextEntryDef("scientificVessel");

    JAXXContextEntryDef<List<Vessel>> FISHING_VESSELS_CONTEXT_ENTRY = JAXXUtil.newListContextEntryDef("fishingVessel");

    JAXXContextEntryDef<List<Gear>> SCIENTIFIC_GEARS_CONTEXT_ENTRY = JAXXUtil.newListContextEntryDef("scientificGear");

    JAXXContextEntryDef<List<Gear>> FISHING_GEARS_CONTEXT_ENTRY = JAXXUtil.newListContextEntryDef("fishingGear");

    protected List<Vessel> getVesselList(VesselTypeEnum vesselType) {
        List<Vessel> result = Lists.newArrayList();
        switch (vesselType) {

            case ALL:
                result.addAll(FISHING_VESSELS_CONTEXT_ENTRY.getContextValue(ui));
                result.addAll(SCIENTIFIC_VESSELS_CONTEXT_ENTRY.getContextValue(ui));
                break;
            case SCIENTIFIC:
                result.addAll(SCIENTIFIC_VESSELS_CONTEXT_ENTRY.getContextValue(ui));
                break;
            case FISHING:
                result.addAll(FISHING_VESSELS_CONTEXT_ENTRY.getContextValue(ui));
                break;
        }
        return result;
    }

    protected List<Gear> getGearList(VesselTypeEnum vesselType) {
        List<Gear> result = Lists.newArrayList();
        switch (vesselType) {

            case ALL:
                result.addAll(SCIENTIFIC_GEARS_CONTEXT_ENTRY.getContextValue(ui));
                result.addAll(FISHING_GEARS_CONTEXT_ENTRY.getContextValue(ui));
                break;
            case SCIENTIFIC:
                result.addAll(SCIENTIFIC_GEARS_CONTEXT_ENTRY.getContextValue(ui));
                break;
            case FISHING:
                result.addAll(FISHING_GEARS_CONTEXT_ENTRY.getContextValue(ui));
                break;
        }
        return result;
    }

    @Override
    public void afterInitUI() {

        initUI(ui);

        List<Person> users = persistenceService.getAllPerson();

        EditCruiseUIModel model = getModel();

        model.setVesselType(model.isCreate() ? VesselTypeEnum.SCIENTIFIC : VesselTypeEnum.ALL);

        // Change vessel list and gear list when vessel type changes
        model.addPropertyChangeListener(EditCruiseUIModel.PROPERTY_VESSEL_TYPE, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {

                EditCruiseUIModel source = (EditCruiseUIModel) evt.getSource();

                List<Vessel> selectedVessel = source.getVessel();
                List<Gear> selectedGear = source.getGear();

                VesselTypeEnum vesselType = (VesselTypeEnum) evt.getNewValue();

                List<Vessel> newVesselList = getVesselList(vesselType);
                List<Gear> newGearList = getGearList(vesselType);

                // compute new selected vessel list
                List<Vessel> newSelectedVessel = Lists.newArrayList();
                if (selectedVessel != null) {
                    newSelectedVessel.addAll(selectedVessel);
                }
                newSelectedVessel.retainAll(newVesselList);

                // compute new selected gear list
                List<Gear> newSelectedGear = Lists.newArrayList();
                if (selectedGear != null) {
                    newSelectedGear.addAll(selectedGear);
                }
                newSelectedGear.retainAll(newGearList);

                // reinit vessel list
                ui.getVesselList().getModel().setUniverse(newVesselList);
                ui.getVesselList().getModel().setSelected(newSelectedVessel);

                // reinit gear list
                ui.getGearList().getModel().setUniverse(newGearList);
                ui.getGearList().getModel().setSelected(newSelectedGear);

            }
        });

        initBeanComboBox(ui.getSurveyComboBox(),
                         persistenceService.getAllProgram(),
                         model.getProgram());

        initBeanComboBox(ui.getCountryComboBox(),
                         persistenceService.getAllCountry(),
                         model.getCountry());

        initBeanList(ui.getVesselList(),
                     getVesselList(model.getVesselType()),
                     model.getVessel());

        initBeanList(ui.getGearList(),
                     getGearList(model.getVesselType()),
                     model.getGear());

        //TODO kmorin 20130218 remove this sort when the double lists will be sortable
        Collections.sort(users, new Comparator<Person>() {

            public int compare(Person o1, Person o2) {
                if (o1 == null) {
                    return -1;
                }
                if (o2 == null) {
                    return 1;
                }
                return o1.getFirstName().compareTo(o2.getFirstName());
            }
        });
        initBeanList(ui.getHeadOfMissionList(),
                     users,
                     model.getHeadOfMission());

        initBeanList(ui.getHeadOfSortRoomList(),
                     users,
                     model.getHeadOfSortRoom());

        SwingValidator validator = ui.getValidator();
        listenValidatorValid(validator, model);

        // if new fishingOperation can already cancel his creation
        model.setModify(model.isCreate());

        registerValidators(validator);

    }

    @Override
    public void onCloseUI() {
        if (log.isInfoEnabled()) {
            log.info("closing: " + ui);
        }
        clearValidators();
    }

    @Override
    public boolean canCloseUI(TuttiScreen nextScreen) {
        boolean result = true;
        if (getModel().isModify()) {
            int answer = askSaveBeforeLeaving(_("tutti.dialog.askSaveBeforeLeaving.saveCruise"));
            result = answer == JOptionPane.NO_OPTION;
            if (answer == JOptionPane.YES_OPTION) {
                ActionEvent event = new ActionEvent(nextScreen, nextScreen.ordinal(), null);
                doAction(ui.getSaveButton(), event);
            }
        }
        return result;
    }

    @Override
    public SwingValidator<EditCruiseUIModel> getValidator() {
        return ui.getValidator();
    }

    public void generateCampaignName() {

        EditCruiseUIModel model = getModel();
        String name = model.getGeneratedCampaignName();
        model.setName(name);
    }

}
