package fr.ifremer.tutti.ui.swing.content.protocol;

/*
 * #%L
 * Tutti :: UI
 * $Id: EditProtocolUIHandler.java 146 2013-01-02 18:51:13Z kmorin $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-0.2.5/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/protocol/EditProtocolUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.BiMap;
import com.google.common.collect.HashBiMap;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.data.SampleCategoryEnum;
import fr.ifremer.tutti.persistence.entities.protocol.SpeciesProtocol;
import fr.ifremer.tutti.persistence.entities.protocol.TuttiProtocol;
import fr.ifremer.tutti.persistence.entities.referential.Caracteristic;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.ui.swing.TuttiScreen;
import fr.ifremer.tutti.ui.swing.TuttiUIContext;
import fr.ifremer.tutti.ui.swing.util.HydrologicCaracteristicUtil;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import fr.ifremer.tutti.ui.swing.util.table.AbstractTuttiTableUIHandler;
import fr.ifremer.tutti.ui.swing.util.table.ColumnIdentifier;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.TableColumnModelEvent;
import javax.swing.event.TableColumnModelListener;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import jaxx.runtime.swing.editor.bean.BeanDoubleList;
import jaxx.runtime.validator.swing.SwingValidatorMessageTableRenderer;
import jaxx.runtime.validator.swing.SwingValidatorUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.table.DefaultTableColumnModelExt;


/**
 * TODO
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.3
 */
public class EditProtocolUIHandler extends AbstractTuttiTableUIHandler<EditProtocolSpeciesRowModel, EditProtocolUIModel> {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(EditProtocolUIHandler.class);

    /**
     * UI.
     *
     * @since 0.3
     */
    private final EditProtocolUI ui;

    protected Map<String, Species> allSpecies;

    protected Map<String, Caracteristic> allLengthStepPmfm;

    protected Multimap<String, String> hydroCaracteristics;

    /**
     * Mapping between columns and sample category
     * (used to define the {@link TuttiProtocol#sampleCategoryOrder}).
     *
     * @since 0.3
     */
    private final BiMap<ColumnIdentifier<EditProtocolSpeciesRowModel>, SampleCategoryEnum> columToSampleCategory;

    public EditProtocolUIHandler(TuttiUIContext context, EditProtocolUI ui) {
        super(context);
        this.ui = ui;
        columToSampleCategory = HashBiMap.create(SampleCategoryEnum.values().length);
        columToSampleCategory.put(EditProtocolSpeciesTableModel.SORTED_UNSORTED_ENABLED, SampleCategoryEnum.sortedUnsorted);
        columToSampleCategory.put(EditProtocolSpeciesTableModel.SIZE_ENABLED, SampleCategoryEnum.size);
        columToSampleCategory.put(EditProtocolSpeciesTableModel.SEX_ENABLED, SampleCategoryEnum.sex);
        columToSampleCategory.put(EditProtocolSpeciesTableModel.MATURITY_ENABLED, SampleCategoryEnum.maturity);
        columToSampleCategory.put(EditProtocolSpeciesTableModel.AGE_ENABLED, SampleCategoryEnum.age);
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiTableUIHandler methods                                --//
    //------------------------------------------------------------------------//

    @Override
    protected EditProtocolSpeciesTableModel getTableModel() {
        return (EditProtocolSpeciesTableModel) getTable().getModel();
    }

    @Override
    protected JXTable getTable() {
        return ui.getSpeciesTable();
    }

    @Override
    protected boolean isRowValid(EditProtocolSpeciesRowModel row) {
        boolean result = row.getLengthStepPmfm() != null;
        return result;
    }

    @Override
    protected void onRowModified(int rowIndex,
                                 EditProtocolSpeciesRowModel row,
                                 String propertyName,
                                 Object oldValue,
                                 Object newValue) {

        recomputeRowValidState(row);
    }


    @Override
    protected void onRowValidStateChanged(int rowIndex, EditProtocolSpeciesRowModel row, Boolean oldValue, Boolean newValue) {
        super.onRowValidStateChanged(rowIndex, row, oldValue, newValue);

        if (row != null && row.isModify() && newValue != null && newValue) {
            // row was modified and is valid, we can save protocol
            // even if after the row becomes again not valid, this is not a
            // problem since we will only save valid data!
            getModel().setModify(true);
        }
    }

    @Override
    protected void saveSelectedRowIfRequired(TuttiBeanMonitor<EditProtocolSpeciesRowModel> rowMonitor,
                                             EditProtocolSpeciesRowModel row) {

        // nothing to save when row changes, model is marked to be save
        // when a row has changed and is valid or other protocol properties are modified
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    protected EditProtocolUIModel getModel() {
        return ui.getModel();
    }

    @Override
    public void beforeInitUI() {

        EditProtocolUIModel model = new EditProtocolUIModel();

        // can't load directly model from database here, since we want to
        // fill only the model with rows (transformed from speciesProtocol)
        // As we still don't have the table model at this point, wait the
        // afterUI method to fill model

        listModelIsModify(model);
        ui.setContextValue(model);

        allSpecies = TuttiEntities.splitById(
                persistenceService.getAllSpecies());

        allLengthStepPmfm = TuttiEntities.splitById(
                persistenceService.getAllSpeciesLengthStepCaracteristic());

        hydroCaracteristics = ArrayListMultimap.create();
        for (Caracteristic caracteristic :
                persistenceService.getAllFishingOperationHydrologicCaracteristic()) {
            String name = HydrologicCaracteristicUtil.getGlobalName(caracteristic.getName());
            hydroCaracteristics.put(name, caracteristic.getId());
        }
    }

    @Override
    public void afterInitUI() {

        initUI(ui);

        EditProtocolUIModel model = getModel();
        
        // load protocol if existing

        String protocolId = context.getProtocolId();
        TuttiProtocol protocol = null;
        if (protocolId == null) {

            // create new protocol

            if (log.isDebugEnabled()) {
                log.debug("Will create a new protocol");
            }
        } else {

            // load existing protocol
            protocol = persistenceService.getProtocol(protocolId);

            model.fromBean(protocol);
        }

        SwingValidatorUtil.installUI(ui.getErrorTable(),
                                     new SwingValidatorMessageTableRenderer());

        listenValidatorValid(ui.getValidator(), model);

        // create table model

        JXTable table = getTable();

        DefaultTableColumnModelExt columnModel = new DefaultTableColumnModelExt();

        addColumnToModel(columnModel,
                         null,
                         newTableCellRender(Species.class),
                         EditProtocolSpeciesTableModel.SPECIES_ID);

        addComboDataColumnToModel(columnModel,
                                  EditProtocolSpeciesTableModel.SIZE_FRENQUENCY_PMFM_ID,
                                  getDecorator(Caracteristic.class, null),
                                  Lists.newArrayList(allLengthStepPmfm.values()));

        Map<SampleCategoryEnum, ColumnIdentifier<EditProtocolSpeciesRowModel>> sampleCategoryToColumn = columToSampleCategory.inverse();

        for (SampleCategoryEnum sampleOrder : model.getSampleCategoryOrder()) {

            ColumnIdentifier<EditProtocolSpeciesRowModel> columnIdentifier = sampleCategoryToColumn.get(sampleOrder);
            addBooleanColumnToModel(columnModel, columnIdentifier, table);
        }

        columnModel.addColumnModelListener(new TableColumnModelListener() {

            public void columnAdded(TableColumnModelEvent e) {
            }

            public void columnRemoved(TableColumnModelEvent e) {
            }

            public void columnMoved(TableColumnModelEvent e) {
                getModel().setModify(true);
            }

            public void columnMarginChanged(ChangeEvent e) {
            }

            public void columnSelectionChanged(ListSelectionEvent e) {
            }
        });

        EditProtocolSpeciesTableModel tableModel =
                new EditProtocolSpeciesTableModel(columnModel);
        table.setModel(tableModel);
        table.setColumnModel(columnModel);

        initTable(table);

        // Authorize to change column orders
        table.getTableHeader().setReorderingAllowed(true);

        List<Species> speciesList = Lists.newArrayList(allSpecies.values());

        List<EditProtocolSpeciesRowModel> rows = Lists.newArrayList();

        // build speciesProtocol rows
        if (protocol != null) {
            List<SpeciesProtocol> speciesProtocols = protocol.getSpecies();
            if (CollectionUtils.isNotEmpty(speciesProtocols)) {
                for (SpeciesProtocol speciesProtocol : speciesProtocols) {
                    Species species = allSpecies.get(speciesProtocol.getSpeciesId());
                    speciesList.remove(species);
                    EditProtocolSpeciesRowModel row = tableModel.createNewRow();
                    row.setSpecies(species);
                    row.setLengthStepPmfm(allLengthStepPmfm.get(speciesProtocol.getLengthStepPmfmId()));
                    row.fromBean(speciesProtocol);
                    rows.add(row);
                }
            }
            if (log.isDebugEnabled()) {
                log.debug("Will edit protocol: " + protocolId + " with " +
                          rows.size() + " species declared.");
            }
        }

        // set to model ( will propagate to tableModel)
        model.setRows(rows);

        // init species combo box
        initBeanComboBox(ui.getSpeciesComboBox(), speciesList, null);
        selectFirstInCombo(ui.getSpeciesComboBox());

        // init gear pmfm double list
        List<Caracteristic> gearPmfm =
                persistenceService.getAllFishingOperationGearCaracteristic();
        initDoubleList(ui.getGearList(), gearPmfm, model.getGearPmfmId());

        // init environement pmfm double list
        List<Caracteristic> environmentPmfm =
                persistenceService.getAllFishingOperationEnvironmentCaracteristic();
        initDoubleList(ui.getEnvironmentList(),
                       environmentPmfm,
                       model.getEnvironmentPmfmId());

        // init hydro pmfm double list
        List<Caracteristic> selection = Lists.newArrayList();
        Collection<String> availableCaracteristicNames = hydroCaracteristics.keySet();
        List<Caracteristic> availableCaracteristics = Lists.newArrayList();
        for (String name : availableCaracteristicNames) {
            Caracteristic caracteristic = HydrologicCaracteristicUtil.createGlobalCaracteristic(name, null);
            availableCaracteristics.add(caracteristic);
        }
        List<String> hydrologyPmfm = model.getHydrologyPmfmId();
        if (hydrologyPmfm != null) {
            for (String caracteristicName : availableCaracteristicNames) {
                Collection<String> ids = hydroCaracteristics.get(caracteristicName);
                for (String caracteristicId : hydrologyPmfm) {
                    if (ids.contains(caracteristicId)) {
                        Caracteristic caracteristic = HydrologicCaracteristicUtil.createGlobalCaracteristic(caracteristicName, null);
                        selection.add(caracteristic);
                        break;
                    }
                }
            }
        }
        initBeanList(ui.getHydrologyList(), availableCaracteristics, selection);

        // if new protocol can already cancel his creation
        model.setModify(model.isCreate());
    }

    @Override
    public void onCloseUI() {
        if (log.isInfoEnabled()) {
            log.info("closing: " + ui);
        }
    }
    
    @Override
    protected void onAfterSelectedRowChanged(int oldRowIndex,
                                             EditProtocolSpeciesRowModel oldRow,
                                             int newRowIndex,
                                             EditProtocolSpeciesRowModel newRow) {
        
        super.onAfterSelectedRowChanged(oldRowIndex, oldRow, newRowIndex, newRow);
        getModel().setRemoveSpeciesEnabled(newRow != null);
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    public void cancel() {

        context.setScreen(TuttiScreen.SELECT_CRUISE);
    }

    public void save() {

        EditProtocolUIModel model = getModel();

        TuttiProtocol bean = model.toBean();

        // get the species protocols from the table
        List<SpeciesProtocol> protocols = Lists.newArrayList();

        for (EditProtocolSpeciesRowModel row : getTableModel().getRows()) {
            if (row.isValid()) {
                SpeciesProtocol protocol = row.toBean();
                protocols.add(protocol);
            }
        }
        bean.setSpecies(protocols);

        // split the hydro grouped pmfm
        List<String> hydroPmfm = bean.getHydrologyPmfmId();
        List<String> allIds = Lists.newArrayList();
        if (hydroPmfm != null) {
            for (String id : hydroPmfm) {
                allIds.addAll(hydroCaracteristics.get(id));
            }
        }
        bean.setHydrologyPmfmId(allIds);

        List<SampleCategoryEnum> sampleOrder = Lists.newArrayList();
        TableColumnModel columnModel = getTable().getColumnModel();
        for (int i = 0; i < columnModel.getColumnCount(); i++) {
            TableColumn column = columnModel.getColumn(i);
            ColumnIdentifier identifier = (ColumnIdentifier) column.getIdentifier();
            SampleCategoryEnum sampleCategory = columToSampleCategory.get(identifier);
            if (sampleCategory != null) {
                sampleOrder.add(sampleCategory);
            }
        }
        bean.setSampleCategoryOrder(sampleOrder);

        TuttiProtocol saved;
        if (TuttiEntities.isNew(bean)) {

            saved = persistenceService.createProtocol(bean);
        } else {
            saved = persistenceService.saveProtocol(bean);
        }

        context.setProtocolId(saved.getId());

        context.setScreen(TuttiScreen.SELECT_CRUISE);
    }

    public void addRow() {
        Species species = (Species) ui.getSpeciesComboBox().getSelectedItem();
        EditProtocolSpeciesRowModel protocol = getTableModel().createNewRow();
        protocol.setSpecies(species);
        getTableModel().addNewRow(protocol);
        ui.getSpeciesComboBox().removeItem(species);
        selectFirstInCombo(ui.getSpeciesComboBox());
//        getModel().setModify(true);
    }
    
    /** Removes a species */
    public void removeSpecies() {
        int rowIndex = getTable().getSelectedRow();

        Preconditions.checkState(rowIndex != -1,
                                 "Cant remove species if no species selected");

        // remove the row from the model
        getModel().getRows().remove(rowIndex);

        // refresh all the table
        getTableModel().fireTableRowsDeleted(rowIndex, rowIndex);
    }

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected void initDoubleList(BeanDoubleList<Caracteristic> widget,
                                  List<Caracteristic> availableCaracteristics,
                                  List<String> selectedCaracteristics) {
        List<Caracteristic> selection = Lists.newArrayList();
        if (selectedCaracteristics != null) {
            for (Caracteristic caracteristic : availableCaracteristics) {
                if (selectedCaracteristics.contains(caracteristic.getId())) {
                    selection.add(caracteristic);
                }
            }
        }
        initBeanList(widget, availableCaracteristics, selection);
    }
}
