package fr.ifremer.tutti.ui.swing.config;

/*
 * #%L
 * Tutti :: Application
 * $Id: TuttiConfig.java 148 2013-01-03 17:52:22Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-0.2.5/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/config/TuttiConfig.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Throwables;
import fr.ifremer.tutti.service.TuttiServiceTechnicalException;
import fr.ifremer.tutti.service.config.TuttiServiceConfig;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.builder.ReflectionToStringBuilder;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.util.ArgumentsParserException;
import org.nuiton.util.Version;

import javax.swing.KeyStroke;
import java.awt.Color;
import java.io.File;
import java.io.IOException;
import java.net.URL;

/**
 * Tutti application config.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class TuttiConfig {

    /** Logger. */
    private static final Log log = LogFactory.getLog(TuttiConfig.class);

    /**
     * Delegate application config object containing configuration.
     *
     * @since 0.1
     */
    protected final ApplicationConfig applicationConfig;

    /**
     * Service configuration to pass to service context.
     *
     * @since 0.1
     */
    protected final TuttiServiceConfig serviceConfig;

    public TuttiConfig(String file, String... args) {

        applicationConfig = new ApplicationConfig();
        applicationConfig.setConfigFileName(file);

        log.info(this + " is initializing...");

        applicationConfig.loadDefaultOptions(TuttiConfigOption.values());

        serviceConfig = new TuttiServiceConfig(applicationConfig);

        try {
            applicationConfig.parse(args);

        } catch (ArgumentsParserException e) {
            throw new TuttiServiceTechnicalException(
                    "Could not parse configuration", e);
        }

        try {
            serviceConfig.prepareDirectories();
        } catch (IOException e) {
            throw Throwables.propagate(e);
        }
    }

    public ApplicationConfig getApplicationConfig() {
        return applicationConfig;
    }

    public TuttiServiceConfig getServiceConfig() {
        return serviceConfig;
    }

    public void save() {
        applicationConfig.saveForUser();
    }

    @Override
    public String toString() {
        return ReflectionToStringBuilder.toString(this);
    }

    /** @return {@link TuttiConfigOption#UI_CONFIG_FILE} value */
    public File getUIConfigFile() {
        return applicationConfig.getOptionAsFile(TuttiConfigOption.UI_CONFIG_FILE.getKey());
    }

    public boolean isAutoPopupNumberEditor() {
        return applicationConfig.getOptionAsBoolean(TuttiConfigOption.AUTO_POPUP_NUMBER_EDITOR.getKey());
    }

    public boolean isShowNumberEditorButton() {
        return applicationConfig.getOptionAsBoolean(TuttiConfigOption.SHOW_NUMBER_EDITOR_BUTTON.getKey());
    }

    public String getProgramId() {
        String result = applicationConfig.getOption(TuttiConfigOption.PROGRAM_ID.getKey());
        return StringUtils.isBlank(result) ? null : result;
    }

    public String getCruiseId() {
        String result = applicationConfig.getOption(TuttiConfigOption.CRUISE_ID.getKey());
        return StringUtils.isBlank(result) ? null : result;
    }

    public String getProtocolId() {
        String result = applicationConfig.getOption(TuttiConfigOption.PROTOCOL_ID.getKey());
        return StringUtils.isBlank(result) ? null : result;
    }

    public Color getColorRowInvalid() {
        return applicationConfig.getOptionAsColor(TuttiConfigOption.COLOR_ROW_INVALID.getKey());
    }

    public Color getColorRowReadOnly() {
        return applicationConfig.getOptionAsColor(TuttiConfigOption.COLOR_ROW_READ_ONLY.getKey());
    }

    public KeyStroke getShortcutClosePopup() {
        return applicationConfig.getOptionAsKeyStroke(TuttiConfigOption.SHORTCUT_CLOSE_POPUP.getKey());
    }

    public KeyStroke getShortCut(String actionName) {
        KeyStroke result = applicationConfig.getOptionAsKeyStroke(
                "tutti.ui." + actionName);
        return result;
    }

    public Version getVersion() {
        return serviceConfig.getVersion();
    }

    public int getInceptionYear() {
        return serviceConfig.getInceptionYear();
    }

    public URL getSiteUrl() {
        return serviceConfig.getSiteUrl();
    }

    public String getOrganizationName() {
        return serviceConfig.getOrganizationName();
    }

    public File getTmpDirectory() {
        return serviceConfig.getTmpDirectory();
    }

    public File getDataDirectory() {
        return serviceConfig.getDataDirectory();
    }

    public void setProgramId(String programId) {
        if (programId == null) {
            programId = "";
        }
        applicationConfig.setOption(TuttiConfigOption.PROGRAM_ID.getKey(), programId);
    }

    public void setCruiseId(String cruiseId) {
        if (cruiseId == null) {
            cruiseId = "";
        }
        applicationConfig.setOption(TuttiConfigOption.CRUISE_ID.getKey(), cruiseId);
    }

    public void setProtocolId(String protocolId) {
        if (protocolId == null) {
            protocolId = "";
        }
        applicationConfig.setOption(TuttiConfigOption.PROTOCOL_ID.getKey(), protocolId);
    }
}
