/*
 * #%L
 * IsisFish
 * 
 * $Id: RUtil.java 4135 2014-11-24 12:42:22Z kmorin $
 * $HeadURL: http://svn.codelutin.com/isis-fish/trunk/src/main/java/fr/ifremer/isisfish/util/RUtil.java $
 * %%
 * Copyright (C) 2006 - 2012 Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.util;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.util.List;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Utility class fo R integration with isis fish.
 * 
 * @author Eric Chatellier
 */
public class RUtil {

    private static final Log log = LogFactory.getLog(RUtil.class);

    /**
     * Try to call "R" process to get location where jri is installed.
     */
    public static void initJri() {

        // set Rtype in jni mode
        System.setProperty("R.type", "jni");

        try {
            ProcessBuilder pb = new ProcessBuilder("Rscript", "-e", "installed.packages()[which(installed.packages()[,1]=='rJava'),]['LibPath']");
            Process process = pb.start();
            int exitValue = process.waitFor();

            if (exitValue == 0) {
                // normal exit
                parseRscriptOutput(process.getInputStream());
            } else {
                // R trouvé, mais la commande R a échoué
                InputStream err = process.getErrorStream();
                if (log.isErrorEnabled()) {
                    log.error("Can't call Rscript : " + IOUtils.toString(err));
                }
            }
        } catch (IOException ex) {
            // Le processus n'a pas pu être lancé (R non trouvé)
            if (log.isDebugEnabled()) {
                log.debug("Can't find R", ex);
            }
        } catch (InterruptedException ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't find R", ex);
            }
        }
    }

    /**
     * Parse valid Rscript output to find if rJava is installed and where it is installed.
     * 
     * @param is process output
     * @throws IOException 
     */
    protected static void parseRscriptOutput(InputStream is) throws IOException {
        List<String> output = IOUtils.readLines(is);
        
        // 1 lignes = commande OK, mais rJava n'est pas installé
        // 2 lignes = commande OK et rJava est installé
        if (output.size() == 1) {
            //installRJava();
        } else if (output.size() == 2) {
            // get install location without ""
            String location = output.get(1).trim();
            location = StringUtils.removeStart(location, "\"");
            location = StringUtils.removeEnd(location, "\"");
            // build full library path
            location = location.replace('/', File.separatorChar);
            location += File.separator + "rJava" + File.separator + "jri";
            setJavaLibraryPath(location);
        } else if (log.isErrorEnabled()) {
            log.error("Can't analyze Rscript output. was: ");
            for (String line : output) {
                log.error(line);
            }
        }
    }

    /**
     * Set runtime java library path.
     * 
     * http://fahdshariff.blogspot.fr/2011/08/changing-java-library-path-at-runtime.html
     * 
     * @param newValue new value to set
     */
    protected static void setJavaLibraryPath(String newValue) {
        if (log.isInfoEnabled()) {
            log.info("Adding '" + newValue + "' to java.library.path");
        }
        System.setProperty("java.library.path", newValue);

        //set sys_paths to null
        try {
            final Field sysPathsField = ClassLoader.class.getDeclaredField("sys_paths");
            sysPathsField.setAccessible(true);
            sysPathsField.set(null, null);
        } catch (NoSuchFieldException | IllegalArgumentException | IllegalAccessException ex) {
            if (log.isErrorEnabled()) {
                log.error("Cant' reset ClassLoader#sys_paths value", ex);
            }
        }
    }
}
