/*
 * #%L
 * EchoBase :: UI
 * 
 * $Id: EchoBaseSession.java 420 2012-03-27 23:02:51Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/echobase/tags/echobase-0.6/echobase-ui/src/main/java/fr/ifremer/echobase/ui/EchoBaseSession.java $
 * %%
 * Copyright (C) 2011 Ifremer, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package fr.ifremer.echobase.ui;

import fr.ifremer.echobase.entities.EchoBaseUser;
import fr.ifremer.echobase.services.EmbeddedApplicationConfiguration;
import fr.ifremer.echobase.services.ExportDbConfiguration;
import fr.ifremer.echobase.services.ImportDbConfiguration;
import fr.ifremer.echobase.services.importdata.AbstractImportConfiguration;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.Map;
import java.util.TreeMap;

/**
 * The session object of EchoBase to put in servlet session.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class EchoBaseSession {

    /** Logger. */
    private static final Log log = LogFactory.getLog(EchoBaseSession.class);

    /** Key to set User connected in this session. */
    protected static final String PROPERTY_ECHO_BASE_USER = "echobaseUser";

    /** Key to keep import db configuration in this session. */
    protected static final String PROPERTY_IMPORT_DB_CONFIGURATION = "importDbConfiguration";

    /** Key to keep export db configuration in this session. */
    protected static final String PROPERTY_EXPORT_DB_CONFIGURATION = "exportDbConfiguration";

    /** Key to keep embedded application configuration in this session. */
    protected static final String PROPERTY_EMBEDDED_APPLICATION_CONFIGURATION = "embeddedApplicationConfiguration";

    /** To store all properties in this session. */
    protected Map<String, Object> store;

    /**
     * Gets the informations of user as soon as the user is loggued.
     *
     * @return the informations of loggued user or {@code null} if not in session
     */
    public EchoBaseUser getEchoBaseUser() {
        return get(PROPERTY_ECHO_BASE_USER, EchoBaseUser.class);
    }

    /**
     * Sets in this session the loggued user.
     *
     * @param echoBaseUser the user loggued to use in this session
     */
    public void setEchoBaseUser(EchoBaseUser echoBaseUser) {
        set(PROPERTY_ECHO_BASE_USER, echoBaseUser);
    }

    public ImportDbConfiguration getImportDbConfiguration() {
        return get(PROPERTY_IMPORT_DB_CONFIGURATION, ImportDbConfiguration.class);
    }

    public void setImportDbConfiguration(ImportDbConfiguration configuration) {
        set(PROPERTY_IMPORT_DB_CONFIGURATION, configuration);
    }

    public ExportDbConfiguration getExportDbConfiguration() {
        return get(PROPERTY_EXPORT_DB_CONFIGURATION, ExportDbConfiguration.class);
    }

    public void setExportDbConfiguration(ExportDbConfiguration configuration) {
        set(PROPERTY_EXPORT_DB_CONFIGURATION, configuration);
    }

    public EmbeddedApplicationConfiguration getEmbeddedApplicationConfiguration() {
        return get(PROPERTY_EMBEDDED_APPLICATION_CONFIGURATION, EmbeddedApplicationConfiguration.class);
    }

    public void setEmbeddedApplicationConfiguration(EmbeddedApplicationConfiguration configuration) {
        set(PROPERTY_EMBEDDED_APPLICATION_CONFIGURATION, configuration);
    }

    public <M extends AbstractImportConfiguration> M getImportDataConfiguration(Class<M> configurationType) {
        return get(configurationType.getSimpleName(), configurationType);
    }

    public <M extends AbstractImportConfiguration> void setImportDataConfiguration(M configuration) {
        set(configuration.getClass().getSimpleName(), configuration);
    }

    public <M extends AbstractImportConfiguration> void removeImportDataConfiguration(Class<M> configurationType) {
        set(configurationType.getSimpleName(), null);
    }

    /**
     * Remove form this session, the object from his given key and returns it.
     *
     * @param key the key of object to remove from this session
     * @return the removed object from session, or {@code null} if not found
     */
    public Object remove(String key) {
        Object remove = store.remove(key);
        if (log.isInfoEnabled()) {
            log.info("Remove from user session data [" + key + "] = " + remove);
        }
        return remove;
    }

    /**
     * Tests if for a given key, there is an associated object in this sessio!!!n.
     *
     * @param key the key to test in this session
     * @return {@code true} if an object was found in this session,
     *         {@code false} otherwise
     */
    public boolean contains(String key) {
        boolean result = store.containsKey(key);
        return result;
    }

    /** Protect session constructor. */
    public EchoBaseSession() {
        store = new TreeMap<String, Object>();
    }

    /**
     * Gets from this session an object given his key and his type.
     *
     * @param key  the key of object to obtain from this session
     * @param type the type of object to obtain from this session
     * @param <T>  the type of object to obtain from this session
     * @return the object found in this session, or {@code null} if not found
     */
    protected <T> T get(String key, Class<T> type) {
        Object o = store.get(key);
        if (o != null && !type.isInstance(o)) {
            throw new ClassCastException(
                    "parameter " + key + " should be of type " +
                    type.getName() + " but was " + o.getClass().getName());
        }
        return (T) o;
    }

    /**
     * Sets in this session the igven object using the given key.
     *
     * @param key   the key where to store the object in this session
     * @param value the object to store in this session
     */
    protected void set(String key, Object value) {
        if (value == null) {
            remove(key);
        } else {
            store.put(key, value);
            if (log.isInfoEnabled()) {
                log.info("Set in user session data [" + key + "] = " + value);
            }
        }
    }

    public boolean isAdmin() {
        EchoBaseUser user = getEchoBaseUser();
        return user != null && user.isAdmin();
    }
}
