/*
 * #%L
 * EchoBase :: UI
 * 
 * $Id: ManageExportQuery.java 343 2012-03-09 23:47:04Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/echobase/tags/echobase-0.3/echobase-ui/src/main/java/fr/ifremer/echobase/ui/actions/exportQuery/ManageExportQuery.java $
 * %%
 * Copyright (C) 2011 Ifremer, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package fr.ifremer.echobase.ui.actions.exportQuery;

import com.opensymphony.xwork2.Preparable;
import fr.ifremer.echobase.entities.EchoBaseUser;
import fr.ifremer.echobase.entities.ExportQuery;
import fr.ifremer.echobase.services.ExportQueryService;
import fr.ifremer.echobase.services.exceptions.ExportQueryNameAlreadyExistException;
import fr.ifremer.echobase.ui.actions.EchoBaseActionSupport;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.Collection;
import java.util.Map;

/**
 * To manage {@link ExportQuery}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class ManageExportQuery extends EchoBaseActionSupport implements Preparable {

    private static final long serialVersionUID = 1L;

    /** Logger. */
    private static final Log log = LogFactory.getLog(ManageExportQuery.class);

    /** All available queries from database. */
    protected Map<String, String> queries;

    /** Selected query loaded from database if his id is not empty. */
    protected ExportQuery query;

    protected boolean newQuery;

    protected boolean newLibreOfficeQuery;

    protected boolean queryExists;

    protected boolean canUpdateQuery;

    protected transient ExportQueryService service;

    public ExportQueryService getService() {
        if (service == null) {
            service = newService(ExportQueryService.class);
        }
        return service;
    }

    public ExportQuery getQuery() {
        if (query == null) {
            query = getService().newExportQuery();
        }
        return query;
    }

    public Map<String, String> getQueries() {
        return queries;
    }

    public Map<String, String> getTableNames() {
        return queries;
    }

    public boolean isNewQuery() {
        return newQuery;
    }

    public void setNewQuery(boolean newQuery) {
        this.newQuery = newQuery;
    }

    public boolean isNewLibreOfficeQuery() {
        return newLibreOfficeQuery;
    }

    public void setNewLibreOfficeQuery(boolean newLibreOfficeQuery) {
        this.newLibreOfficeQuery = newLibreOfficeQuery;
    }

    public boolean isCanUpdateQuery() {
        return canUpdateQuery;
    }

    public boolean isQueryExists() {
        return StringUtils.isNotEmpty(getQuery().getTopiaId());
    }

    public String saveQuery() throws Exception {

        String result = INPUT;

        boolean safeSql = checkQuery(getQuery().getSqlQuery());

        if (safeSql) {
            try {
                query = getService().createOrUpdate(getQuery(),
                                                    getEchoBaseSession().getEchoBaseUser()
                );

                result = SUCCESS;
            } catch (ExportQueryNameAlreadyExistException e) {
                addFieldError("query.name",
                              _("echobase.error.query.name.already.exists"));

            }
        }
        return result;
    }

    public String cloneQuery() throws Exception {

        getQuery().setTopiaId(null);
        getQuery().setName(getQuery().getName() + "-clone");
        query = getService().createOrUpdate(getQuery(),
                                            getEchoBaseSession().getEchoBaseUser()
        );
        return SUCCESS;
    }

    public String confirmDeleteQuery() throws Exception {

        return SUCCESS;
    }

    public String deleteQuery() throws Exception {

        //TODO : do validation
        service.delete(getQuery().getTopiaId());
        query = null;
        return SUCCESS;
    }

    @Override
    public String execute() {

        if (StringUtils.isNotEmpty(getQuery().getTopiaId())) {

            // test query
            checkQuery(getQuery().getSqlQuery());
        }
        return SUCCESS;
    }

    private boolean checkQuery(String sql) {
        boolean result;
        try {
            getService().testSql(sql);
            result = true;
        } catch (Exception e) {
            Throwable cause = e.getCause();
            if (log.isWarnEnabled()) {
                log.warn("Invalid sql ", cause);
            }
            addFieldError("query.sqlQuery",
                          _("echobase.error.invalid.sql", cause.getMessage()));
            result = false;
        }
        return result;
    }

    @Override
    public void prepare() throws Exception {

        Collection<ExportQuery> sqlQueries =
                getService().getEntities(ExportQuery.class);

        queries = sortAndDecorate(sqlQueries, null);

        if (sqlQueries.isEmpty()) {

            // no query saved
            addActionMessage(_("echobase.info.no.sqlQuery.saved"));
        } else {

            String selectedQueryId = getQuery().getTopiaId();

            if (!isQueryExists()) {

                if (isNewQuery() || isNewLibreOfficeQuery()) {

                    // new query in progress
                    addActionMessage(_("echobase.info.new.sqlQuery.inprogress"));
                } else {

                    // no query selected
                    addActionMessage(_("echobase.info.no.sqlQuery.selected"));
                }
            } else {

                // load query from database
                query = getService().getExportQuery(selectedQueryId);

                EchoBaseUser echoBaseUser =
                        getEchoBaseSession().getEchoBaseUser();
                canUpdateQuery = echoBaseUser.isAdmin() ||
                                 echoBaseUser.getEmail().equals(
                                         query.getLastModifiedUser());

                if (!canUpdateQuery) {

                    // this user can not update selected query
                    addActionMessage(
                            _("echobase.info.sqlQuery.not.modifiable"));
                }
            }
        }
    }
}