/*
 * #%L
 * Vradi :: Services
 * 
 * $Id: VradiQueryParser.java 1715 2010-10-27 19:21:28Z tchemit $
 * $HeadURL: svn+ssh://tchemit@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.3/vradi-services/src/main/java/com/jurismarches/vradi/services/search/VradiQueryParser.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package com.jurismarches.vradi.services.search;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.WhitespaceAnalyzer;
import org.apache.lucene.queryParser.ParseException;
import org.apache.lucene.queryParser.QueryParser;
import org.apache.lucene.search.Query;
import org.apache.lucene.util.Version;

/**
 * VradiQueryParser supports the analysis of textual queries.
 * 
 * Queries are analyzed by the lucene <code>QueryParser</code>, so they
 * must obey his syntax.
 * See at <a href="http://lucene.apache.org/java/3_0_2/queryparsersyntax.html">
 * queryparsersyntax.html</a> for a complete documentation.
 * 
 * Analyzer used to parse queries is <code>WhitespaceAnalyzer</code> and the default
 * parser operator is <code>QueryParser.Operator.AND</code>.
 * 
 * @author $Author: tchemit $
 * @version $Revision: 1715 $ $Date: 2010-10-27 21:21:28 +0200 (mer., 27 oct. 2010) $
 * @since 0.0.4
 */
public class VradiQueryParser {

    private static final Log log = LogFactory.getLog(VradiQueryParser.class);

    public static final String DEFAULT_FIELD = "text";

    /**
     * Parses the specified <code>expression</code> and returns a <code>FilterList</code>.
     * 
     * @param expression the expression
     * @return a <code>FilterList</code>
     * @throws UnsupportedQueryException
     */
    public static FilterList parse(String expression) throws UnsupportedQueryException {
        return parse(expression, DEFAULT_FIELD);
    }
    
    /**
     * Parses the specified <code>expression</code> and returns a <code>FilterList</code>.
     * 
     * @param expression the expression
     * @param defaultField the default search field
     * @return a <code>FilterList</code>
     * @throws UnsupportedQueryException
     */
    public static FilterList parse(String expression, String defaultField) throws UnsupportedQueryException {
        FilterListQueryHandler queryHandler = new FilterListQueryHandler();
        parse(queryHandler, expression, defaultField);
        
        FilterList filterList = queryHandler.getFilterList();
        return filterList;
    }
    
    static void parse(QueryHandler queryHandler, String expression, String defaultField) throws UnsupportedQueryException {
        if (log.isDebugEnabled()) {
            log.debug("expression: " + expression);
        }
        
        Analyzer analyzer = new WhitespaceAnalyzer();
        QueryParser parser = new QueryParser(Version.LUCENE_29, defaultField, analyzer);
        
        parser.setAllowLeadingWildcard(true);
        parser.setDefaultOperator(QueryParser.Operator.AND);
        
        if (expression == null || expression.trim().isEmpty()) {
            expression = defaultField + ":*";
        }

        try {
            Query query = parser.parse(expression.trim());
            
            LuceneQueryParser luceneQueryParser = new LuceneQueryParser(queryHandler);
            luceneQueryParser.parse(query);
            
        } catch (ParseException e) {
            throw new UnsupportedQueryException(expression, e);
        }
    }
}
