/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.tools.jib.cache;

import com.google.cloud.tools.jib.api.DescriptorDigest;
import com.google.cloud.tools.jib.api.ImageReference;
import com.google.cloud.tools.jib.blob.Blob;
import com.google.cloud.tools.jib.blob.BlobDescriptor;
import com.google.cloud.tools.jib.blob.Blobs;
import com.google.cloud.tools.jib.cache.CacheStorageFiles;
import com.google.cloud.tools.jib.cache.CachedLayer;
import com.google.cloud.tools.jib.cache.Retry;
import com.google.cloud.tools.jib.filesystem.LockFile;
import com.google.cloud.tools.jib.filesystem.TempDirectoryProvider;
import com.google.cloud.tools.jib.hash.CountingDigestOutputStream;
import com.google.cloud.tools.jib.hash.Digests;
import com.google.cloud.tools.jib.image.json.BuildableManifestTemplate;
import com.google.cloud.tools.jib.image.json.ContainerConfigurationTemplate;
import com.google.cloud.tools.jib.image.json.ImageMetadataTemplate;
import com.google.cloud.tools.jib.image.json.ManifestAndConfigTemplate;
import com.google.cloud.tools.jib.image.json.ManifestTemplate;
import com.google.cloud.tools.jib.image.json.V21ManifestTemplate;
import com.google.cloud.tools.jib.json.JsonTemplate;
import com.google.cloud.tools.jib.json.JsonTemplateMapper;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.AtomicMoveNotSupportedException;
import java.nio.file.CopyOption;
import java.nio.file.FileSystemException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import java.util.zip.GZIPOutputStream;
import javax.annotation.Nullable;
import org.apache.commons.compress.compressors.CompressorException;
import org.apache.commons.compress.compressors.CompressorInputStream;
import org.apache.commons.compress.compressors.CompressorStreamFactory;

class CacheStorageWriter {
    private final CacheStorageFiles cacheStorageFiles;

    private static void verifyImageMetadata(ImageMetadataTemplate metadata) {
        Predicate<ManifestAndConfigTemplate> isManifestNull = pair -> pair.getManifest() == null;
        Predicate<ManifestAndConfigTemplate> isConfigNull = pair -> pair.getConfig() == null;
        Predicate<ManifestAndConfigTemplate> isDigestNull = pair -> pair.getManifestDigest() == null;
        List<ManifestAndConfigTemplate> manifestsAndConfigs = metadata.getManifestsAndConfigs();
        Preconditions.checkArgument((!manifestsAndConfigs.isEmpty() ? 1 : 0) != 0, (Object)"no manifests given");
        Preconditions.checkArgument((boolean)manifestsAndConfigs.stream().noneMatch(isManifestNull), (Object)"null manifest(s)");
        Preconditions.checkArgument((metadata.getManifestList() != null || manifestsAndConfigs.size() == 1 ? 1 : 0) != 0, (Object)"manifest list missing while multiple manifests given");
        ManifestTemplate firstManifest = manifestsAndConfigs.get(0).getManifest();
        if (firstManifest instanceof V21ManifestTemplate) {
            Preconditions.checkArgument((metadata.getManifestList() == null ? 1 : 0) != 0, (Object)"manifest list given for schema 1");
            Preconditions.checkArgument((boolean)isConfigNull.test(manifestsAndConfigs.get(0)), (Object)"container config given for schema 1");
        } else if (firstManifest instanceof BuildableManifestTemplate) {
            Preconditions.checkArgument((boolean)manifestsAndConfigs.stream().noneMatch(isConfigNull), (Object)"null container config(s)");
            if (metadata.getManifestList() != null) {
                Preconditions.checkArgument((boolean)manifestsAndConfigs.stream().noneMatch(isDigestNull), (Object)"null manifest digest(s)");
            }
        } else {
            throw new IllegalArgumentException("Unknown manifest type: " + firstManifest);
        }
    }

    @VisibleForTesting
    static void moveIfDoesNotExist(Path source, Path destination) throws IOException {
        String errorMessage = String.format("unable to move: %s to %s; such failures are often caused by interference from antivirus (https://github.com/GoogleContainerTools/jib/issues/3127#issuecomment-796838294), or rarely if the operation is not supported by the file system (for example: special non-local file system)", source, destination);
        try {
            Retry.Action rename = () -> {
                if (Files.exists(destination, new LinkOption[0])) {
                    return true;
                }
                Files.move(source, destination, new CopyOption[0]);
                return Files.exists(destination, new LinkOption[0]);
            };
            if (!Retry.action(rename).maximumRetries(10).retryOnException(ex -> ex instanceof FileSystemException).sleep(15L, TimeUnit.MILLISECONDS).run()) {
                throw new IOException(errorMessage);
            }
        }
        catch (IOException ex2) {
            throw new IOException(errorMessage, ex2);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static DescriptorDigest getDiffIdByDecompressingFile(Path compressedFile) throws IOException {
        try (CompressorInputStream in = CompressorStreamFactory.getSingleton().createCompressorInputStream((InputStream)new BufferedInputStream(Files.newInputStream(compressedFile, new OpenOption[0])));){
            DescriptorDigest descriptorDigest = Digests.computeDigest((InputStream)in).getDigest();
            return descriptorDigest;
        }
        catch (CompressorException e) {
            throw new IOException(e);
        }
    }

    private static void writeJsonTemplate(JsonTemplate jsonTemplate, Path destination) throws IOException {
        Path temporaryFile = Files.createTempFile(destination.getParent(), null, null, new FileAttribute[0]);
        temporaryFile.toFile().deleteOnExit();
        try (OutputStream outputStream = Files.newOutputStream(temporaryFile, new OpenOption[0]);){
            JsonTemplateMapper.writeTo(jsonTemplate, outputStream);
        }
        try {
            Files.move(temporaryFile, destination, StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.REPLACE_EXISTING);
        }
        catch (AtomicMoveNotSupportedException ignored) {
            Files.move(temporaryFile, destination, StandardCopyOption.REPLACE_EXISTING);
        }
    }

    CacheStorageWriter(CacheStorageFiles cacheStorageFiles) {
        this.cacheStorageFiles = cacheStorageFiles;
    }

    CachedLayer writeCompressed(Blob compressedLayerBlob) throws IOException {
        Files.createDirectories(this.cacheStorageFiles.getLayersDirectory(), new FileAttribute[0]);
        Files.createDirectories(this.cacheStorageFiles.getTemporaryDirectory(), new FileAttribute[0]);
        try (TempDirectoryProvider tempDirectoryProvider = new TempDirectoryProvider();){
            Path temporaryLayerDirectory = tempDirectoryProvider.newDirectory(this.cacheStorageFiles.getTemporaryDirectory());
            WrittenLayer writtenLayer = this.writeCompressedLayerBlobToDirectory(compressedLayerBlob, temporaryLayerDirectory);
            CacheStorageWriter.moveIfDoesNotExist(temporaryLayerDirectory, this.cacheStorageFiles.getLayerDirectory(writtenLayer.layerDigest));
            Path layerFile = this.cacheStorageFiles.getLayerFile(writtenLayer.layerDigest, writtenLayer.layerDiffId);
            CachedLayer cachedLayer = CachedLayer.builder().setLayerDigest(writtenLayer.layerDigest).setLayerDiffId(writtenLayer.layerDiffId).setLayerSize(writtenLayer.layerSize).setLayerBlob(Blobs.from(layerFile)).build();
            return cachedLayer;
        }
    }

    CachedLayer writeUncompressed(Blob uncompressedLayerBlob, @Nullable DescriptorDigest selector) throws IOException {
        Files.createDirectories(this.cacheStorageFiles.getLayersDirectory(), new FileAttribute[0]);
        Files.createDirectories(this.cacheStorageFiles.getTemporaryDirectory(), new FileAttribute[0]);
        try (TempDirectoryProvider tempDirectoryProvider = new TempDirectoryProvider();){
            Path temporaryLayerDirectory = tempDirectoryProvider.newDirectory(this.cacheStorageFiles.getTemporaryDirectory());
            WrittenLayer writtenLayer = this.writeUncompressedLayerBlobToDirectory(uncompressedLayerBlob, temporaryLayerDirectory);
            CacheStorageWriter.moveIfDoesNotExist(temporaryLayerDirectory, this.cacheStorageFiles.getLayerDirectory(writtenLayer.layerDigest));
            Path layerFile = this.cacheStorageFiles.getLayerFile(writtenLayer.layerDigest, writtenLayer.layerDiffId);
            CachedLayer.Builder cachedLayerBuilder = CachedLayer.builder().setLayerDigest(writtenLayer.layerDigest).setLayerDiffId(writtenLayer.layerDiffId).setLayerSize(writtenLayer.layerSize).setLayerBlob(Blobs.from(layerFile));
            if (selector != null) {
                this.writeSelector(selector, writtenLayer.layerDigest);
            }
            CachedLayer cachedLayer = cachedLayerBuilder.build();
            return cachedLayer;
        }
    }

    CachedLayer writeTarLayer(DescriptorDigest diffId, Blob compressedBlob) throws IOException {
        Files.createDirectories(this.cacheStorageFiles.getLocalDirectory(), new FileAttribute[0]);
        Files.createDirectories(this.cacheStorageFiles.getTemporaryDirectory(), new FileAttribute[0]);
        try (TempDirectoryProvider tempDirectoryProvider = new TempDirectoryProvider();){
            BlobDescriptor layerBlobDescriptor;
            Path temporaryLayerDirectory = tempDirectoryProvider.newDirectory(this.cacheStorageFiles.getTemporaryDirectory());
            Path temporaryLayerFile = this.cacheStorageFiles.getTemporaryLayerFile(temporaryLayerDirectory);
            try (BufferedOutputStream fileOutputStream = new BufferedOutputStream(Files.newOutputStream(temporaryLayerFile, new OpenOption[0]));){
                layerBlobDescriptor = compressedBlob.writeTo(fileOutputStream);
            }
            String fileName = layerBlobDescriptor.getDigest().getHash();
            Path digestLayerFile = temporaryLayerDirectory.resolve(fileName);
            CacheStorageWriter.moveIfDoesNotExist(temporaryLayerFile, digestLayerFile);
            Path destination = this.cacheStorageFiles.getLocalDirectory().resolve(diffId.getHash());
            CacheStorageWriter.moveIfDoesNotExist(temporaryLayerDirectory, destination);
            CachedLayer cachedLayer = CachedLayer.builder().setLayerDigest(layerBlobDescriptor.getDigest()).setLayerDiffId(diffId).setLayerSize(layerBlobDescriptor.getSize()).setLayerBlob(Blobs.from(destination.resolve(fileName))).build();
            return cachedLayer;
        }
    }

    void writeMetadata(ImageReference imageReference, ImageMetadataTemplate metadata) throws IOException {
        CacheStorageWriter.verifyImageMetadata(metadata);
        Path imageDirectory = this.cacheStorageFiles.getImageDirectory(imageReference);
        Files.createDirectories(imageDirectory, new FileAttribute[0]);
        try (LockFile ignored = LockFile.lock(imageDirectory.resolve("lock"));){
            CacheStorageWriter.writeJsonTemplate(metadata, imageDirectory.resolve("manifests_configs.json"));
        }
    }

    void writeLocalConfig(DescriptorDigest imageId, ContainerConfigurationTemplate containerConfiguration) throws IOException {
        Path configDirectory = this.cacheStorageFiles.getLocalDirectory().resolve("config");
        Files.createDirectories(configDirectory, new FileAttribute[0]);
        CacheStorageWriter.writeJsonTemplate(containerConfiguration, configDirectory.resolve(imageId.getHash()));
    }

    private WrittenLayer writeCompressedLayerBlobToDirectory(Blob compressedLayerBlob, Path layerDirectory) throws IOException {
        BlobDescriptor layerBlobDescriptor;
        Path temporaryLayerFile = this.cacheStorageFiles.getTemporaryLayerFile(layerDirectory);
        try (BufferedOutputStream fileOutputStream = new BufferedOutputStream(Files.newOutputStream(temporaryLayerFile, new OpenOption[0]));){
            layerBlobDescriptor = compressedLayerBlob.writeTo(fileOutputStream);
        }
        DescriptorDigest layerDiffId = CacheStorageWriter.getDiffIdByDecompressingFile(temporaryLayerFile);
        Path layerFile = layerDirectory.resolve(this.cacheStorageFiles.getLayerFilename(layerDiffId));
        CacheStorageWriter.moveIfDoesNotExist(temporaryLayerFile, layerFile);
        return new WrittenLayer(layerBlobDescriptor.getDigest(), layerDiffId, layerBlobDescriptor.getSize());
    }

    private WrittenLayer writeUncompressedLayerBlobToDirectory(Blob uncompressedLayerBlob, Path layerDirectory) throws IOException {
        Path temporaryLayerFile = this.cacheStorageFiles.getTemporaryLayerFile(layerDirectory);
        try (CountingDigestOutputStream compressedDigestOutputStream = new CountingDigestOutputStream(new BufferedOutputStream(Files.newOutputStream(temporaryLayerFile, new OpenOption[0])));){
            GZIPOutputStream compressorStream = new GZIPOutputStream(compressedDigestOutputStream);
            DescriptorDigest layerDiffId = uncompressedLayerBlob.writeTo(compressorStream).getDigest();
            compressorStream.close();
            BlobDescriptor blobDescriptor = compressedDigestOutputStream.computeDigest();
            DescriptorDigest layerDigest = blobDescriptor.getDigest();
            long layerSize = blobDescriptor.getSize();
            Path layerFile = layerDirectory.resolve(this.cacheStorageFiles.getLayerFilename(layerDiffId));
            CacheStorageWriter.moveIfDoesNotExist(temporaryLayerFile, layerFile);
            WrittenLayer writtenLayer = new WrittenLayer(layerDigest, layerDiffId, layerSize);
            return writtenLayer;
        }
    }

    private void writeSelector(DescriptorDigest selector, DescriptorDigest layerDigest) throws IOException {
        Path selectorFile = this.cacheStorageFiles.getSelectorFile(selector);
        Files.createDirectories(selectorFile.getParent(), new FileAttribute[0]);
        Path temporarySelectorFile = Files.createTempFile(null, null, new FileAttribute[0]);
        temporarySelectorFile.toFile().deleteOnExit();
        try (OutputStream fileOut = Files.newOutputStream(temporarySelectorFile, new OpenOption[0]);){
            fileOut.write(layerDigest.getHash().getBytes(StandardCharsets.UTF_8));
        }
        try {
            Files.move(temporarySelectorFile, selectorFile, StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.REPLACE_EXISTING);
        }
        catch (AtomicMoveNotSupportedException ignored) {
            Files.move(temporarySelectorFile, selectorFile, StandardCopyOption.REPLACE_EXISTING);
        }
    }

    private static class WrittenLayer {
        private final DescriptorDigest layerDigest;
        private final DescriptorDigest layerDiffId;
        private final long layerSize;

        private WrittenLayer(DescriptorDigest layerDigest, DescriptorDigest layerDiffId, long layerSize) {
            this.layerDigest = layerDigest;
            this.layerDiffId = layerDiffId;
            this.layerSize = layerSize;
        }
    }
}

