/*
 * Decompiled with CFR 0.152.
 */
package smile.math.matrix;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.nio.DoubleBuffer;
import java.nio.IntBuffer;
import java.util.Arrays;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import smile.math.MathEx;
import smile.math.blas.BLAS;
import smile.math.blas.Diag;
import smile.math.blas.EVDJob;
import smile.math.blas.LAPACK;
import smile.math.blas.Layout;
import smile.math.blas.SVDJob;
import smile.math.blas.Side;
import smile.math.blas.Transpose;
import smile.math.blas.UPLO;
import smile.math.matrix.DMatrix;
import smile.sort.QuickSort;
import smile.stat.distribution.Distribution;
import smile.stat.distribution.GaussianDistribution;

public class Matrix
extends DMatrix {
    private static final long serialVersionUID = 2L;
    private static final Logger logger = LoggerFactory.getLogger(Matrix.class);
    transient DoubleBuffer A;
    transient int ld;
    int m;
    int n;
    UPLO uplo;
    Diag diag;

    public Matrix(int m, int n) {
        this(m, n, 0.0);
    }

    public Matrix(int m, int n, double a) {
        if (m <= 0 || n <= 0) {
            throw new IllegalArgumentException(String.format("Invalid matrix size: %d x %d", m, n));
        }
        this.m = m;
        this.n = n;
        this.ld = Matrix.ld(m);
        double[] array = new double[this.ld * n];
        if (a != 0.0) {
            Arrays.fill(array, a);
        }
        this.A = DoubleBuffer.wrap(array);
    }

    public Matrix(int m, int n, double[][] A) {
        this(m, n);
        for (int i = 0; i < m; ++i) {
            for (int j = 0; j < n; ++j) {
                this.set(i, j, A[i][j]);
            }
        }
    }

    public Matrix(double[][] A) {
        this(A.length, A[0].length, A);
    }

    public Matrix(double[] A) {
        this(A, 0, A.length);
    }

    public Matrix(double[] A, int offset, int length) {
        this.m = length;
        this.n = 1;
        this.ld = length;
        this.A = DoubleBuffer.wrap(A, offset, length);
    }

    public Matrix(int m, int n, int ld, DoubleBuffer A) {
        if (this.layout() == Layout.COL_MAJOR && ld < m) {
            throw new IllegalArgumentException(String.format("Invalid leading dimension for COL_MAJOR: %d < %d", ld, m));
        }
        if (this.layout() == Layout.ROW_MAJOR && ld < n) {
            throw new IllegalArgumentException(String.format("Invalid leading dimension for ROW_MAJOR: %d < %d", ld, n));
        }
        this.m = m;
        this.n = n;
        this.ld = ld;
        this.A = A;
    }

    public static Matrix of(Layout layout, int m, int n) {
        if (layout == Layout.COL_MAJOR) {
            int ld = Matrix.ld(m);
            int size = ld * n;
            return Matrix.of(layout, m, n, ld, DoubleBuffer.allocate(size));
        }
        int ld = Matrix.ld(n);
        int size = ld * m;
        return Matrix.of(layout, m, n, ld, DoubleBuffer.allocate(size));
    }

    public static Matrix of(Layout layout, int m, int n, int ld, DoubleBuffer A) {
        if (layout == Layout.COL_MAJOR && ld < m) {
            throw new IllegalArgumentException(String.format("Invalid leading dimension for COL_MAJOR: %d < %d", ld, m));
        }
        if (layout == Layout.ROW_MAJOR && ld < n) {
            throw new IllegalArgumentException(String.format("Invalid leading dimension for ROW_MAJOR: %d < %d", ld, n));
        }
        if (layout == Layout.COL_MAJOR) {
            return new Matrix(m, n, ld, A);
        }
        return new Matrix(m, n, ld, A){

            @Override
            public Layout layout() {
                return Layout.ROW_MAJOR;
            }

            @Override
            protected int index(int i, int j) {
                return i * this.ld + j + this.A.position();
            }

            @Override
            public Matrix transpose() {
                return new Matrix(this.n, this.m, this.ld, this.A);
            }
        };
    }

    public static Matrix eye(int n) {
        Matrix matrix = new Matrix(n, n);
        for (int i = 0; i < n; ++i) {
            matrix.set(i, i, 1.0);
        }
        return matrix;
    }

    public static Matrix eye(int m, int n) {
        Matrix matrix = new Matrix(m, n);
        int k = Math.min(m, n);
        for (int i = 0; i < k; ++i) {
            matrix.set(i, i, 1.0);
        }
        return matrix;
    }

    public static Matrix randn(int m, int n) {
        return Matrix.rand(m, n, GaussianDistribution.getInstance());
    }

    public static Matrix rand(int m, int n, Distribution distribution) {
        Matrix matrix = new Matrix(m, n);
        for (int j = 0; j < n; ++j) {
            for (int i = 0; i < m; ++i) {
                matrix.set(i, j, distribution.rand());
            }
        }
        return matrix;
    }

    public static Matrix rand(int m, int n, double lo, double hi) {
        Matrix matrix = new Matrix(m, n);
        for (int j = 0; j < n; ++j) {
            for (int i = 0; i < m; ++i) {
                matrix.set(i, j, MathEx.random(lo, hi));
            }
        }
        return matrix;
    }

    public static Matrix diag(double[] diag) {
        int n = diag.length;
        Matrix D = new Matrix(n, n);
        for (int i = 0; i < n; ++i) {
            D.set(i, i, diag[i]);
        }
        return D;
    }

    public static Matrix toeplitz(double[] a) {
        int n = a.length;
        Matrix toeplitz = new Matrix(n, n);
        toeplitz.uplo(UPLO.LOWER);
        for (int i = 0; i < n; ++i) {
            int j;
            for (j = 0; j < i; ++j) {
                toeplitz.set(i, j, a[i - j]);
            }
            for (j = i; j < n; ++j) {
                toeplitz.set(i, j, a[j - i]);
            }
        }
        return toeplitz;
    }

    public static Matrix toeplitz(double[] kl, double[] ku) {
        if (kl.length != ku.length - 1) {
            throw new IllegalArgumentException(String.format("Invalid subdiagonals and superdiagonals size: %d != %d - 1", kl.length, ku.length));
        }
        int n = kl.length;
        Matrix toeplitz = new Matrix(n, n);
        for (int i = 0; i < n; ++i) {
            int j;
            for (j = 0; j < i; ++j) {
                toeplitz.set(i, j, kl[i - j]);
            }
            for (j = i; j < n; ++j) {
                toeplitz.set(i, j, ku[j - i]);
            }
        }
        return toeplitz;
    }

    private static int ld(int n) {
        int elementSize = 4;
        if (n <= 256 / elementSize) {
            return n;
        }
        return ((n * elementSize + 511) / 512 * 512 + 64) / elementSize;
    }

    private void writeObject(ObjectOutputStream out) throws IOException {
        out.defaultWriteObject();
        if (this.layout() == Layout.COL_MAJOR) {
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    out.writeDouble(this.get(i, j));
                }
            }
        } else {
            for (int i = 0; i < this.m; ++i) {
                for (int j = 0; j < this.n; ++j) {
                    out.writeDouble(this.get(i, j));
                }
            }
        }
    }

    private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException {
        in.defaultReadObject();
        this.A = DoubleBuffer.wrap(new double[this.m * this.n]);
        if (this.layout() == Layout.COL_MAJOR) {
            this.ld = this.m;
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    this.set(i, j, in.readDouble());
                }
            }
        } else {
            this.ld = this.n;
            for (int i = 0; i < this.m; ++i) {
                for (int j = 0; j < this.n; ++j) {
                    this.set(i, j, in.readDouble());
                }
            }
        }
    }

    @Override
    public int nrows() {
        return this.m;
    }

    @Override
    public int ncols() {
        return this.n;
    }

    @Override
    public long size() {
        return this.m * this.n;
    }

    public Layout layout() {
        return Layout.COL_MAJOR;
    }

    public int ld() {
        return this.ld;
    }

    public boolean isSubmatrix() {
        return this.A.position() != 0 || this.A.limit() != this.A.capacity();
    }

    public boolean isSymmetric() {
        return this.uplo != null && this.diag == null;
    }

    public Matrix uplo(UPLO uplo) {
        if (this.m != this.n) {
            throw new IllegalArgumentException(String.format("The matrix is not square: %d x %d", this.m, this.n));
        }
        this.uplo = uplo;
        return this;
    }

    public UPLO uplo() {
        return this.uplo;
    }

    public Matrix triangular(Diag diag) {
        if (this.m != this.n) {
            throw new IllegalArgumentException(String.format("The matrix is not square: %d x %d", this.m, this.n));
        }
        this.diag = diag;
        return this;
    }

    public Diag triangular() {
        return this.diag;
    }

    public Matrix clone() {
        Matrix matrix = new Matrix(this.m, this.n);
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                matrix.set(i, j, this.get(i, j));
            }
        }
        if (this.m == this.n) {
            matrix.uplo(this.uplo);
            matrix.triangular(this.diag);
        }
        return matrix;
    }

    public double[][] toArray() {
        double[][] array = new double[this.m][this.n];
        for (int i = 0; i < this.m; ++i) {
            for (int j = 0; j < this.n; ++j) {
                array[i][j] = this.get(i, j);
            }
        }
        return array;
    }

    public Matrix submatrix(int i, int j, int k, int l) {
        if (i < 0 || i >= this.m || k < i || k >= this.m || j < 0 || j >= this.n || l < j || l >= this.n) {
            throw new IllegalArgumentException(String.format("Invalid submatrix range (%d:%d, %d:%d) of %d x %d", i, k, j, l, this.m, this.n));
        }
        int offset = this.index(i, j);
        int length = this.index(k, l) - offset + 1;
        DoubleBuffer B = DoubleBuffer.wrap(this.A.array(), offset, length);
        return Matrix.of(this.layout(), k - i + 1, l - j + 1, this.ld, B);
    }

    public void fill(double x) {
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.set(i, j, x);
            }
        }
    }

    public Matrix transpose() {
        return Matrix.of(Layout.ROW_MAJOR, this.n, this.m, this.ld, this.A);
    }

    public boolean equals(Object o) {
        if (o == null || !(o instanceof Matrix)) {
            return false;
        }
        return this.equals((Matrix)o, 1.0E-7f);
    }

    public boolean equals(Matrix o, double eps) {
        if (this.m != o.m || this.n != o.n) {
            return false;
        }
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                if (MathEx.isZero(this.get(i, j) - o.get(i, j), eps)) continue;
                return false;
            }
        }
        return true;
    }

    protected int index(int i, int j) {
        return j * this.ld + i + this.A.position();
    }

    @Override
    public double get(int i, int j) {
        return this.A.get(this.index(i, j));
    }

    @Override
    public Matrix set(int i, int j, double x) {
        this.A.put(this.index(i, j), x);
        return this;
    }

    public Matrix set(int i, int j, Matrix B) {
        for (int jj = 0; jj < B.n; ++jj) {
            for (int ii = 0; ii < B.m; ++ii) {
                this.set(i + ii, j + jj, B.get(ii, jj));
            }
        }
        return this;
    }

    public double add(int i, int j, double b) {
        int k = this.index(i, j);
        double y = this.A.get(k) + b;
        this.A.put(k, y);
        return y;
    }

    public double sub(int i, int j, double b) {
        int k = this.index(i, j);
        double y = this.A.get(k) - b;
        this.A.put(k, y);
        return y;
    }

    public double mul(int i, int j, double b) {
        int k = this.index(i, j);
        double y = this.A.get(k) * b;
        this.A.put(k, y);
        return y;
    }

    public double div(int i, int j, double b) {
        int k = this.index(i, j);
        double y = this.A.get(k) / b;
        this.A.put(k, y);
        return y;
    }

    public Matrix add(double b) {
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.add(i, j, b);
            }
        }
        return this;
    }

    public Matrix sub(double b) {
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.sub(i, j, b);
            }
        }
        return this;
    }

    public Matrix mul(double b) {
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.mul(i, j, b);
            }
        }
        return this;
    }

    public Matrix div(double b) {
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.div(i, j, b);
            }
        }
        return this;
    }

    public Matrix add(int i, int j, double alpha, Matrix B) {
        for (int jj = 0; jj < B.n; ++jj) {
            for (int ii = 0; ii < B.m; ++ii) {
                this.add(i + ii, j + jj, alpha * B.get(ii, jj));
            }
        }
        return this;
    }

    public Matrix sub(int i, int j, double alpha, Matrix B) {
        for (int jj = 0; jj < B.n; ++jj) {
            for (int ii = 0; ii < B.m; ++ii) {
                this.sub(i + ii, j + jj, alpha * B.get(ii, jj));
            }
        }
        return this;
    }

    public Matrix mul(int i, int j, double alpha, Matrix B) {
        for (int jj = 0; jj < B.n; ++jj) {
            for (int ii = 0; ii < B.m; ++ii) {
                this.mul(i + ii, j + jj, alpha * B.get(ii, jj));
            }
        }
        return this;
    }

    public Matrix div(int i, int j, double alpha, Matrix B) {
        for (int jj = 0; jj < B.n; ++jj) {
            for (int ii = 0; ii < B.m; ++ii) {
                this.div(i + ii, j + jj, alpha * B.get(ii, jj));
            }
        }
        return this;
    }

    public Matrix add(double alpha, Matrix B) {
        if (this.m != B.m || this.n != B.n) {
            throw new IllegalArgumentException("Matrix is not of same size.");
        }
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.add(i, j, alpha * B.get(i, j));
            }
        }
        return this;
    }

    public Matrix sub(double alpha, Matrix B) {
        if (this.m != B.m || this.n != B.n) {
            throw new IllegalArgumentException("Matrix is not of same size.");
        }
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.sub(i, j, alpha * B.get(i, j));
            }
        }
        return this;
    }

    public Matrix mul(double alpha, Matrix B) {
        if (this.m != B.m || this.n != B.n) {
            throw new IllegalArgumentException("Matrix is not of same size.");
        }
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.mul(i, j, alpha * B.get(i, j));
            }
        }
        return this;
    }

    public Matrix div(double alpha, Matrix B) {
        if (this.m != B.m || this.n != B.n) {
            throw new IllegalArgumentException("Matrix is not of same size.");
        }
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.div(i, j, alpha * B.get(i, j));
            }
        }
        return this;
    }

    public Matrix add(double alpha, Matrix A, double beta, Matrix B) {
        if (this.m != A.m || this.n != A.n) {
            throw new IllegalArgumentException("Matrix A is not of same size.");
        }
        if (this.m != B.m || this.n != B.n) {
            throw new IllegalArgumentException("Matrix B is not of same size.");
        }
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.set(i, j, alpha * A.get(i, j) + beta * B.get(i, j));
            }
        }
        return this;
    }

    public Matrix sub(double alpha, Matrix A, double beta, Matrix B) {
        return this.add(alpha, A, -beta, B);
    }

    public Matrix mul(double alpha, Matrix A, Matrix B) {
        if (this.m != A.m || this.n != A.n) {
            throw new IllegalArgumentException("Matrix A is not of same size.");
        }
        if (this.m != B.m || this.n != B.n) {
            throw new IllegalArgumentException("Matrix B is not of same size.");
        }
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.set(i, j, alpha * A.get(i, j) * B.get(i, j));
            }
        }
        return this;
    }

    public Matrix div(double alpha, Matrix A, Matrix B) {
        if (this.m != A.m || this.n != A.n) {
            throw new IllegalArgumentException("Matrix A is not of same size.");
        }
        if (this.m != B.m || this.n != B.n) {
            throw new IllegalArgumentException("Matrix B is not of same size.");
        }
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.set(i, j, alpha * A.get(i, j) / B.get(i, j));
            }
        }
        return this;
    }

    public double add(int i, int j, double alpha, double beta) {
        int k = this.index(i, j);
        double y = alpha * this.A.get(k) + beta;
        this.A.put(k, y);
        return y;
    }

    public Matrix add(int i, int j, double alpha, double beta, Matrix B) {
        for (int jj = 0; jj < B.n; ++jj) {
            for (int ii = 0; ii < B.m; ++ii) {
                this.add(i + ii, j + jj, alpha, beta * B.get(ii, jj));
            }
        }
        return this;
    }

    public Matrix add(double alpha, double beta, Matrix B) {
        if (this.m != B.m || this.n != B.n) {
            throw new IllegalArgumentException("Matrix B is not of same size.");
        }
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.set(i, j, alpha * this.get(i, j) + beta * B.get(i, j));
            }
        }
        return this;
    }

    public Matrix add(double alpha, double[] x, double[] y) {
        if (this.m != x.length || this.n != y.length) {
            throw new IllegalArgumentException("Matrix is not of same size.");
        }
        if (this.isSymmetric() && x == y) {
            BLAS.engine.syr(this.layout(), this.uplo, this.m, alpha, DoubleBuffer.wrap(x), 1, this.A, this.ld);
        } else {
            BLAS.engine.ger(this.layout(), this.m, this.n, alpha, DoubleBuffer.wrap(x), 1, DoubleBuffer.wrap(y), 1, this.A, this.ld);
        }
        return this;
    }

    public Matrix replaceNaN(double x) {
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                if (!Double.isNaN(this.get(i, j))) continue;
                this.set(i, j, x);
            }
        }
        return this;
    }

    public double sum() {
        double s = 0.0;
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                s += this.get(i, j);
            }
        }
        return s;
    }

    public double norm1() {
        double f = 0.0;
        for (int j = 0; j < this.n; ++j) {
            double s = 0.0;
            for (int i = 0; i < this.m; ++i) {
                s += Math.abs(this.get(i, j));
            }
            f = Math.max(f, s);
        }
        return f;
    }

    public double norm2() {
        return this.svd((boolean)false, (boolean)false).s[0];
    }

    public double norm() {
        return this.norm2();
    }

    public double normInf() {
        double[] f = new double[this.m];
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                int n = i;
                f[n] = f[n] + Math.abs(this.get(i, j));
            }
        }
        return MathEx.max(f);
    }

    public double normFro() {
        double f = 0.0;
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                f = Math.hypot(f, this.get(i, j));
            }
        }
        return f;
    }

    public double xAx(double[] x) {
        if (this.m != this.n) {
            throw new IllegalArgumentException(String.format("The matrix is not square: %d x %d", this.m, this.n));
        }
        if (this.n != x.length) {
            throw new IllegalArgumentException(String.format("Matrix: %d x %d, Vector: %d", this.m, this.n, x.length));
        }
        int n = x.length;
        double s = 0.0;
        for (int j = 0; j < n; ++j) {
            for (int i = 0; i < n; ++i) {
                s += this.get(i, j) * x[i] * x[j];
            }
        }
        return s;
    }

    public double[] rowSums() {
        double[] x = new double[this.m];
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                int n = i;
                x[n] = x[n] + this.get(i, j);
            }
        }
        return x;
    }

    public double[] rowMeans() {
        double[] x = new double[this.m];
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                int n = i;
                x[n] = x[n] + this.get(i, j);
            }
        }
        int i = 0;
        while (i < this.m) {
            int n = i++;
            x[n] = x[n] / (double)this.n;
        }
        return x;
    }

    public double[] rowSds() {
        double[] x = new double[this.m];
        double[] x2 = new double[this.m];
        for (int j = 0; j < this.n; ++j) {
            int i = 0;
            while (i < this.m) {
                double a = this.get(i, j);
                int n = i;
                x[n] = x[n] + a;
                int n2 = i++;
                x2[n2] = x2[n2] + a * a;
            }
        }
        for (int i = 0; i < this.m; ++i) {
            double mu = x[i] / (double)this.n;
            x[i] = Math.sqrt(x2[i] / (double)this.n - mu * mu);
        }
        return x;
    }

    public double[] colSums() {
        double[] x = new double[this.n];
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                int n = j;
                x[n] = x[n] + this.get(i, j);
            }
        }
        return x;
    }

    public double[] colMeans() {
        double[] x = new double[this.n];
        int j = 0;
        while (j < this.n) {
            for (int i = 0; i < this.m; ++i) {
                int n = j;
                x[n] = x[n] + this.get(i, j);
            }
            int n = j++;
            x[n] = x[n] / (double)this.m;
        }
        return x;
    }

    public double[] colSds() {
        double[] x = new double[this.n];
        for (int j = 0; j < this.n; ++j) {
            double mu = 0.0;
            double sumsq = 0.0;
            for (int i = 0; i < this.m; ++i) {
                double a = this.get(i, j);
                mu += a;
                sumsq += a * a;
            }
            x[j] = Math.sqrt(sumsq / (double)this.m - (mu /= (double)this.m) * mu);
        }
        return x;
    }

    public Matrix scale() {
        double[] center = this.colMeans();
        double[] scale = this.colSds();
        return this.scale(center, scale);
    }

    public Matrix scale(double[] center, double[] scale) {
        if (center == null && scale == null) {
            throw new IllegalArgumentException("Both center and scale are null");
        }
        Matrix matrix = new Matrix(this.m, this.n);
        if (center == null) {
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    matrix.set(i, j, this.get(i, j) / scale[j]);
                }
            }
        } else if (scale == null) {
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    matrix.set(i, j, this.get(i, j) - center[j]);
                }
            }
        } else {
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    matrix.set(i, j, (this.get(i, j) - center[j]) / scale[j]);
                }
            }
        }
        return matrix;
    }

    public Matrix inverse() {
        if (this.m != this.n) {
            throw new IllegalArgumentException(String.format("The matrix is not square: %d x %d", this.m, this.n));
        }
        Matrix lu = this.clone();
        Matrix inv = Matrix.eye(this.n);
        int[] ipiv = new int[this.n];
        if (this.isSymmetric()) {
            int info = LAPACK.engine.sysv(lu.layout(), this.uplo, this.n, this.n, lu.A, lu.ld, IntBuffer.wrap(ipiv), inv.A, inv.ld);
            if (info != 0) {
                throw new ArithmeticException("SYSV fails: " + info);
            }
        } else {
            int info = LAPACK.engine.gesv(lu.layout(), this.n, this.n, lu.A, lu.ld, IntBuffer.wrap(ipiv), inv.A, inv.ld);
            if (info != 0) {
                throw new ArithmeticException("GESV fails: " + info);
            }
        }
        return inv;
    }

    public void mv(Transpose trans, double alpha, DoubleBuffer x, double beta, DoubleBuffer y) {
        if (this.uplo != null) {
            if (this.diag != null) {
                if (alpha == 1.0 && beta == 0.0 && x == y) {
                    BLAS.engine.trmv(this.layout(), this.uplo, trans, this.diag, this.m, this.A, this.ld, y, 1);
                } else {
                    BLAS.engine.gemv(this.layout(), trans, this.m, this.n, alpha, this.A, this.ld, x, 1, beta, y, 1);
                }
            } else {
                BLAS.engine.symv(this.layout(), this.uplo, this.m, alpha, this.A, this.ld, x, 1, beta, y, 1);
            }
        } else {
            BLAS.engine.gemv(this.layout(), trans, this.m, this.n, alpha, this.A, this.ld, x, 1, beta, y, 1);
        }
    }

    @Override
    public void mv(Transpose trans, double alpha, double[] x, double beta, double[] y) {
        this.mv(trans, alpha, DoubleBuffer.wrap(x), beta, DoubleBuffer.wrap(y));
    }

    @Override
    public void mv(double[] work, int inputOffset, int outputOffset) {
        DoubleBuffer xb = DoubleBuffer.wrap(work, inputOffset, this.n);
        DoubleBuffer yb = DoubleBuffer.wrap(work, outputOffset, this.m);
        this.mv(Transpose.NO_TRANSPOSE, 1.0, xb, 0.0, yb);
    }

    @Override
    public void tv(double[] work, int inputOffset, int outputOffset) {
        DoubleBuffer xb = DoubleBuffer.wrap(work, inputOffset, this.m);
        DoubleBuffer yb = DoubleBuffer.wrap(work, outputOffset, this.n);
        this.mv(Transpose.TRANSPOSE, 1.0, xb, 0.0, yb);
    }

    private Transpose flip(Transpose trans) {
        return trans == Transpose.NO_TRANSPOSE ? Transpose.TRANSPOSE : Transpose.NO_TRANSPOSE;
    }

    public void mm(Transpose transA, Transpose transB, double alpha, Matrix B, double beta, Matrix C) {
        if (this.isSymmetric() && transB == Transpose.NO_TRANSPOSE && B.layout() == C.layout()) {
            BLAS.engine.symm(C.layout(), Side.LEFT, this.uplo, C.m, C.n, alpha, this.A, this.ld, B.A, B.ld, beta, C.A, C.ld);
        } else if (B.isSymmetric() && transA == Transpose.NO_TRANSPOSE && this.layout() == C.layout()) {
            BLAS.engine.symm(C.layout(), Side.RIGHT, B.uplo, C.m, C.n, alpha, B.A, B.ld, this.A, this.ld, beta, C.A, C.ld);
        } else {
            if (C.layout() != this.layout()) {
                transA = this.flip(transA);
            }
            if (C.layout() != B.layout()) {
                transB = this.flip(transB);
            }
            int k = transA == Transpose.NO_TRANSPOSE ? this.n : this.m;
            BLAS.engine.gemm(this.layout(), transA, transB, C.m, C.n, k, alpha, this.A, this.ld, B.A, B.ld, beta, C.A, C.ld);
        }
    }

    public Matrix ata() {
        Matrix C = new Matrix(this.n, this.n);
        this.mm(Transpose.TRANSPOSE, Transpose.NO_TRANSPOSE, 1.0, this, 0.0, C);
        C.uplo(UPLO.LOWER);
        return C;
    }

    public Matrix aat() {
        Matrix C = new Matrix(this.m, this.m);
        this.mm(Transpose.NO_TRANSPOSE, Transpose.TRANSPOSE, 1.0, this, 0.0, C);
        C.uplo(UPLO.LOWER);
        return C;
    }

    public Matrix adb(Transpose transA, Transpose transB, Matrix B, double[] diag) {
        Matrix C;
        if (transA == Transpose.NO_TRANSPOSE) {
            C = new Matrix(this.m, this.n);
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    C.set(i, j, diag[j] * this.get(i, j));
                }
            }
        } else {
            C = new Matrix(this.n, this.m);
            for (int j = 0; j < this.m; ++j) {
                for (int i = 0; i < this.n; ++i) {
                    C.set(i, j, diag[j] * this.get(j, i));
                }
            }
        }
        return transB == Transpose.NO_TRANSPOSE ? C.mm(B) : C.mt(B);
    }

    public Matrix mm(Matrix B) {
        if (this.n != B.m) {
            throw new IllegalArgumentException(String.format("Matrix multiplication A * B: %d x %d vs %d x %d", this.m, this.n, B.m, B.n));
        }
        Matrix C = new Matrix(this.m, B.n);
        this.mm(Transpose.NO_TRANSPOSE, Transpose.NO_TRANSPOSE, 1.0, B, 0.0, C);
        return C;
    }

    public Matrix mt(Matrix B) {
        if (this.n != B.n) {
            throw new IllegalArgumentException(String.format("Matrix multiplication A * B': %d x %d vs %d x %d", this.m, this.n, B.m, B.n));
        }
        Matrix C = new Matrix(this.m, B.m);
        this.mm(Transpose.NO_TRANSPOSE, Transpose.TRANSPOSE, 1.0, B, 0.0, C);
        return C;
    }

    public Matrix tm(Matrix B) {
        if (this.m != B.m) {
            throw new IllegalArgumentException(String.format("Matrix multiplication A' * B: %d x %d vs %d x %d", this.m, this.n, B.m, B.n));
        }
        Matrix C = new Matrix(this.n, B.n);
        this.mm(Transpose.TRANSPOSE, Transpose.NO_TRANSPOSE, 1.0, B, 0.0, C);
        return C;
    }

    public Matrix tt(Matrix B) {
        if (this.m != B.n) {
            throw new IllegalArgumentException(String.format("Matrix multiplication A' * B': %d x %d vs %d x %d", this.m, this.n, B.m, B.n));
        }
        Matrix C = new Matrix(this.n, B.m);
        this.mm(Transpose.TRANSPOSE, Transpose.TRANSPOSE, 1.0, B, 0.0, C);
        return C;
    }

    public LU lu() {
        return this.lu(false);
    }

    public LU lu(boolean overwrite) {
        Matrix lu = overwrite ? this : this.clone();
        int[] ipiv = new int[Math.min(this.m, this.n)];
        int info = LAPACK.engine.getrf(lu.layout(), lu.m, lu.n, lu.A, lu.ld, IntBuffer.wrap(ipiv));
        if (info < 0) {
            logger.error("LAPACK GETRF error code: {}", (Object)info);
            throw new ArithmeticException("LAPACK GETRF error code: " + info);
        }
        return new LU(lu, ipiv, info);
    }

    public Cholesky cholesky() {
        return this.cholesky(false);
    }

    public Cholesky cholesky(boolean overwrite) {
        if (this.uplo == null) {
            throw new IllegalArgumentException("The matrix is not symmetric");
        }
        Matrix lu = overwrite ? this : this.clone();
        int info = LAPACK.engine.potrf(lu.layout(), lu.uplo, lu.n, lu.A, lu.ld);
        if (info != 0) {
            logger.error("LAPACK GETRF error code: {}", (Object)info);
            throw new ArithmeticException("LAPACK GETRF error code: " + info);
        }
        return new Cholesky(lu);
    }

    public QR qr() {
        return this.qr(false);
    }

    public QR qr(boolean overwrite) {
        Matrix qr = overwrite ? this : this.clone();
        double[] tau = new double[Math.min(this.m, this.n)];
        int info = LAPACK.engine.geqrf(qr.layout(), qr.m, qr.n, qr.A, qr.ld, DoubleBuffer.wrap(tau));
        if (info != 0) {
            logger.error("LAPACK GEQRF error code: {}", (Object)info);
            throw new ArithmeticException("LAPACK GEQRF error code: " + info);
        }
        return new QR(qr, tau);
    }

    public SVD svd() {
        return this.svd(true, false);
    }

    public SVD svd(boolean vectors, boolean overwrite) {
        Matrix W;
        int k = Math.min(this.m, this.n);
        double[] s = new double[k];
        Matrix matrix = W = overwrite ? this : this.clone();
        if (vectors) {
            Matrix U = new Matrix(this.m, k);
            Matrix VT = new Matrix(k, this.n);
            int info = LAPACK.engine.gesdd(W.layout(), SVDJob.COMPACT, W.m, W.n, W.A, W.ld, DoubleBuffer.wrap(s), U.A, U.ld, VT.A, VT.ld);
            if (info != 0) {
                logger.error("LAPACK GESDD error code: {}", (Object)info);
                throw new ArithmeticException("LAPACK GESDD error code: " + info);
            }
            return new SVD(s, U, VT.transpose());
        }
        Matrix U = new Matrix(1, 1);
        Matrix VT = new Matrix(1, 1);
        int info = LAPACK.engine.gesdd(W.layout(), SVDJob.NO_VECTORS, W.m, W.n, W.A, W.ld, DoubleBuffer.wrap(s), U.A, U.ld, VT.A, VT.ld);
        if (info != 0) {
            logger.error("LAPACK GESDD error code: {}", (Object)info);
            throw new ArithmeticException("LAPACK GESDD error code: " + info);
        }
        return new SVD(this.m, this.n, s);
    }

    public EVD eigen() {
        return this.eigen(false, true, false);
    }

    public EVD eigen(boolean vl, boolean vr, boolean overwrite) {
        Matrix eig;
        if (this.m != this.n) {
            throw new IllegalArgumentException(String.format("The matrix is not square: %d x %d", this.m, this.n));
        }
        Matrix matrix = eig = overwrite ? this : this.clone();
        if (this.isSymmetric()) {
            double[] w = new double[this.n];
            int info = LAPACK.engine.syevd(eig.layout(), vr ? EVDJob.VECTORS : EVDJob.NO_VECTORS, eig.uplo, this.n, eig.A, eig.ld, DoubleBuffer.wrap(w));
            if (info != 0) {
                logger.error("LAPACK SYEV error code: {}", (Object)info);
                throw new ArithmeticException("LAPACK SYEV error code: " + info);
            }
            return new EVD(w, vr ? eig : null);
        }
        double[] wr = new double[this.n];
        double[] wi = new double[this.n];
        Matrix Vl = vl ? new Matrix(this.n, this.n) : new Matrix(1, 1);
        Matrix Vr = vr ? new Matrix(this.n, this.n) : new Matrix(1, 1);
        int info = LAPACK.engine.geev(eig.layout(), vl ? EVDJob.VECTORS : EVDJob.NO_VECTORS, vr ? EVDJob.VECTORS : EVDJob.NO_VECTORS, this.n, eig.A, eig.ld, DoubleBuffer.wrap(wr), DoubleBuffer.wrap(wi), Vl.A, Vl.ld, Vr.A, Vr.ld);
        if (info != 0) {
            logger.error("LAPACK GEEV error code: {}", (Object)info);
            throw new ArithmeticException("LAPACK GEEV error code: " + info);
        }
        double[] w = new double[2 * this.n];
        System.arraycopy(wr, 0, w, 0, this.n);
        System.arraycopy(wi, 0, w, this.n, this.n);
        return new EVD(wr, wi, vl ? Vl : null, vr ? Vr : null);
    }

    public static class QR {
        public final Matrix qr;
        public final double[] tau;

        public QR(Matrix qr, double[] tau) {
            this.qr = qr;
            this.tau = tau;
        }

        public Cholesky CholeskyOfAtA() {
            int n = this.qr.n;
            Matrix L = new Matrix(n, n);
            for (int i = 0; i < n; ++i) {
                for (int j = 0; j <= i; ++j) {
                    L.set(i, j, this.qr.get(j, i));
                }
            }
            L.uplo(UPLO.LOWER);
            return new Cholesky(L);
        }

        public Matrix R() {
            int n = this.qr.n;
            Matrix R = Matrix.diag(this.tau);
            for (int i = 0; i < n; ++i) {
                for (int j = i + 1; j < n; ++j) {
                    R.set(i, j, this.qr.get(i, j));
                }
            }
            return R;
        }

        public Matrix Q() {
            int m = this.qr.m;
            int n = this.qr.n;
            Matrix Q = new Matrix(m, n);
            for (int k = n - 1; k >= 0; --k) {
                Q.set(k, k, 1.0);
                for (int j = k; j < n; ++j) {
                    int i;
                    if (this.qr.get(k, k) == 0.0) continue;
                    double s = 0.0;
                    for (i = k; i < m; ++i) {
                        s += this.qr.get(i, k) * Q.get(i, j);
                    }
                    s = -s / this.qr.get(k, k);
                    for (i = k; i < m; ++i) {
                        Q.add(i, j, s * this.qr.get(i, k));
                    }
                }
            }
            return Q;
        }

        public double[] solve(double[] b) {
            if (b.length != this.qr.m) {
                throw new IllegalArgumentException(String.format("Row dimensions do not agree: A is %d x %d, but B is %d x 1", this.qr.m, this.qr.n, b.length));
            }
            double[] y = (double[])b.clone();
            this.solve(new Matrix(y));
            double[] x = new double[this.qr.n];
            System.arraycopy(y, 0, x, 0, x.length);
            return x;
        }

        public void solve(Matrix B) {
            if (B.m != this.qr.m) {
                throw new IllegalArgumentException(String.format("Row dimensions do not agree: A is %d x %d, but B is %d x %d", this.qr.nrows(), this.qr.nrows(), B.nrows(), B.ncols()));
            }
            int m = this.qr.m;
            int n = this.qr.n;
            int k = Math.min(m, n);
            int info = LAPACK.engine.ormqr(this.qr.layout(), Side.LEFT, Transpose.TRANSPOSE, B.nrows(), B.ncols(), k, this.qr.A, this.qr.ld, DoubleBuffer.wrap(this.tau), B.A, B.ld);
            if (info != 0) {
                logger.error("LAPACK ORMQR error code: {}", (Object)info);
                throw new IllegalArgumentException("LAPACK ORMQR error code: " + info);
            }
            info = LAPACK.engine.trtrs(this.qr.layout(), UPLO.UPPER, Transpose.NO_TRANSPOSE, Diag.NON_UNIT, this.qr.n, B.n, this.qr.A, this.qr.ld, B.A, B.ld);
            if (info != 0) {
                logger.error("LAPACK TRTRS error code: {}", (Object)info);
                throw new IllegalArgumentException("LAPACK TRTRS error code: " + info);
            }
        }
    }

    public static class Cholesky {
        public final Matrix lu;

        public Cholesky(Matrix lu) {
            if (lu.nrows() != lu.ncols()) {
                throw new UnsupportedOperationException("Cholesky constructor on a non-square matrix");
            }
            this.lu = lu;
        }

        public double det() {
            double d = 1.0;
            for (int i = 0; i < this.lu.n; ++i) {
                d *= this.lu.get(i, i);
            }
            return d * d;
        }

        public Matrix inverse() {
            Matrix inv = Matrix.eye(this.lu.n);
            this.solve(inv);
            return inv;
        }

        public double[] solve(double[] b) {
            double[] x = (double[])b.clone();
            this.solve(new Matrix(x));
            return x;
        }

        public void solve(Matrix B) {
            if (B.m != this.lu.m) {
                throw new IllegalArgumentException(String.format("Row dimensions do not agree: A is %d x %d, but B is %d x %d", this.lu.m, this.lu.n, B.m, B.n));
            }
            int info = LAPACK.engine.potrs(this.lu.layout(), this.lu.uplo, this.lu.n, B.n, this.lu.A, this.lu.ld, B.A, B.ld);
            if (info != 0) {
                logger.error("LAPACK POTRS error code: {}", (Object)info);
                throw new ArithmeticException("LAPACK POTRS error code: " + info);
            }
        }
    }

    public static class LU {
        public final Matrix lu;
        public final int[] ipiv;
        public final int info;

        public LU(Matrix lu, int[] ipiv, int info) {
            this.lu = lu;
            this.ipiv = ipiv;
            this.info = info;
        }

        public boolean isSingular() {
            return this.info > 0;
        }

        public double det() {
            int j;
            int m = this.lu.m;
            int n = this.lu.n;
            if (m != n) {
                throw new IllegalArgumentException(String.format("The matrix is not square: %d x %d", m, n));
            }
            double d = 1.0;
            for (j = 0; j < n; ++j) {
                d *= this.lu.get(j, j);
            }
            for (j = 0; j < n; ++j) {
                if (j + 1 == this.ipiv[j]) continue;
                d = -d;
            }
            return d;
        }

        public Matrix inverse() {
            Matrix inv = Matrix.eye(this.lu.n);
            this.solve(inv);
            return inv;
        }

        public double[] solve(double[] b) {
            double[] x = (double[])b.clone();
            this.solve(new Matrix(x));
            return x;
        }

        public void solve(Matrix B) {
            if (this.lu.m != this.lu.n) {
                throw new IllegalArgumentException(String.format("The matrix is not square: %d x %d", this.lu.m, this.lu.n));
            }
            if (B.m != this.lu.m) {
                throw new IllegalArgumentException(String.format("Row dimensions do not agree: A is %d x %d, but B is %d x %d", this.lu.m, this.lu.n, B.m, B.n));
            }
            if (this.lu.layout() != B.layout()) {
                throw new IllegalArgumentException("The matrix layout is inconsistent.");
            }
            if (this.info > 0) {
                throw new RuntimeException("The matrix is singular.");
            }
            int ret = LAPACK.engine.getrs(this.lu.layout(), Transpose.NO_TRANSPOSE, this.lu.n, B.n, this.lu.A, this.lu.ld, IntBuffer.wrap(this.ipiv), B.A, B.ld);
            if (ret != 0) {
                logger.error("LAPACK GETRS error code: {}", (Object)ret);
                throw new ArithmeticException("LAPACK GETRS error code: " + ret);
            }
        }
    }

    public static class EVD {
        public final double[] wr;
        public final double[] wi;
        public final Matrix Vl;
        public final Matrix Vr;

        public EVD(double[] w, Matrix V) {
            this.wr = w;
            this.wi = null;
            this.Vl = V;
            this.Vr = V;
        }

        public EVD(double[] wr, double[] wi, Matrix Vl, Matrix Vr) {
            this.wr = wr;
            this.wi = wi;
            this.Vl = Vl;
            this.Vr = Vr;
        }

        public Matrix diag() {
            Matrix D = Matrix.diag(this.wr);
            if (this.wi != null) {
                int n = this.wr.length;
                for (int i = 0; i < n; ++i) {
                    if (this.wi[i] > 0.0) {
                        D.set(i, i + 1, this.wi[i]);
                        continue;
                    }
                    if (!(this.wi[i] < 0.0)) continue;
                    D.set(i, i - 1, this.wi[i]);
                }
            }
            return D;
        }

        public EVD sort() {
            int i;
            int n = this.wr.length;
            double[] w = new double[n];
            if (this.wi != null) {
                for (i = 0; i < n; ++i) {
                    w[i] = -(this.wr[i] * this.wr[i] + this.wi[i] * this.wi[i]);
                }
            } else {
                for (i = 0; i < n; ++i) {
                    w[i] = -(this.wr[i] * this.wr[i]);
                }
            }
            int[] index = QuickSort.sort(w);
            double[] wr2 = new double[n];
            for (int j = 0; j < n; ++j) {
                wr2[j] = this.wr[index[j]];
            }
            double[] wi2 = null;
            if (this.wi != null) {
                wi2 = new double[n];
                for (int j = 0; j < n; ++j) {
                    wi2[j] = this.wi[index[j]];
                }
            }
            Matrix Vl2 = null;
            if (this.Vl != null) {
                int m = this.Vl.m;
                Vl2 = new Matrix(m, n);
                for (int j = 0; j < n; ++j) {
                    for (int i2 = 0; i2 < m; ++i2) {
                        Vl2.set(i2, j, this.Vl.get(i2, index[j]));
                    }
                }
            }
            Matrix Vr2 = null;
            if (this.Vr != null) {
                int m = this.Vr.m;
                Vr2 = new Matrix(m, n);
                for (int j = 0; j < n; ++j) {
                    for (int i3 = 0; i3 < m; ++i3) {
                        Vr2.set(i3, j, this.Vr.get(i3, index[j]));
                    }
                }
            }
            return new EVD(wr2, wi2, Vl2, Vr2);
        }
    }

    public static class SVD {
        public final int m;
        public final int n;
        public final double[] s;
        public final Matrix U;
        public final Matrix V;

        public SVD(int m, int n, double[] s) {
            this.m = m;
            this.n = n;
            this.s = s;
            this.U = null;
            this.V = null;
        }

        public SVD(double[] s, Matrix U, Matrix V) {
            this.m = U.m;
            this.n = V.m;
            this.s = s;
            this.U = U;
            this.V = V;
        }

        public Matrix diag() {
            Matrix S = new Matrix(this.U.m, this.V.m);
            for (int i = 0; i < this.s.length; ++i) {
                S.set(i, i, this.s[i]);
            }
            return S;
        }

        public double norm() {
            return this.s[0];
        }

        private double rcond() {
            return 0.5 * Math.sqrt(this.m + this.n + 1) * this.s[0] * MathEx.EPSILON;
        }

        public int rank() {
            if (this.s.length != Math.min(this.m, this.n)) {
                throw new UnsupportedOperationException("The operation cannot be called on a partial SVD.");
            }
            int r = 0;
            double tol = this.rcond();
            for (int i = 0; i < this.s.length; ++i) {
                if (!(this.s[i] > tol)) continue;
                ++r;
            }
            return r;
        }

        public int nullity() {
            return Math.min(this.m, this.n) - this.rank();
        }

        public double condition() {
            if (this.s.length != Math.min(this.m, this.n)) {
                throw new UnsupportedOperationException("The operation cannot be called on a partial SVD.");
            }
            return this.s[0] <= 0.0 || this.s[this.s.length - 1] <= 0.0 ? Double.POSITIVE_INFINITY : this.s[0] / this.s[this.s.length - 1];
        }

        public Matrix range() {
            if (this.s.length != Math.min(this.m, this.n)) {
                throw new UnsupportedOperationException("The operation cannot be called on a partial SVD.");
            }
            if (this.U == null) {
                throw new IllegalStateException("The left singular vectors are not available.");
            }
            int r = this.rank();
            if (r == 0) {
                return null;
            }
            Matrix R = new Matrix(this.m, r);
            for (int j = 0; j < r; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    R.set(i, j, this.U.get(i, j));
                }
            }
            return R;
        }

        public Matrix nullspace() {
            if (this.s.length != Math.min(this.m, this.n)) {
                throw new UnsupportedOperationException("The operation cannot be called on a partial SVD.");
            }
            if (this.V == null) {
                throw new IllegalStateException("The right singular vectors are not available.");
            }
            int nr = this.nullity();
            if (nr == 0) {
                return null;
            }
            Matrix N = new Matrix(this.n, nr);
            for (int j = 0; j < nr; ++j) {
                for (int i = 0; i < this.n; ++i) {
                    N.set(i, j, this.V.get(i, this.n - j - 1));
                }
            }
            return N;
        }

        public Matrix pinv() {
            int k = this.s.length;
            double[] sigma = new double[k];
            int r = this.rank();
            for (int i = 0; i < r; ++i) {
                sigma[i] = 1.0 / this.s[i];
            }
            return this.V.adb(Transpose.NO_TRANSPOSE, Transpose.TRANSPOSE, this.U, sigma);
        }

        public double[] solve(double[] b) {
            if (this.U == null || this.V == null) {
                throw new IllegalStateException("The singular vectors are not available.");
            }
            if (b.length != this.m) {
                throw new IllegalArgumentException(String.format("Row dimensions do not agree: A is %d x %d, but B is %d x 1", this.m, this.n, b.length));
            }
            int r = this.rank();
            double[] Utb = new double[this.s.length];
            this.U.submatrix(0, 0, this.m - 1, r - 1).tv(b, Utb);
            for (int i = 0; i < r; ++i) {
                int n = i;
                Utb[n] = Utb[n] / this.s[i];
            }
            return this.V.mv(Utb);
        }
    }
}

