/*
 * Decompiled with CFR 0.152.
 */
package smile.math.matrix;

import java.nio.FloatBuffer;
import java.nio.IntBuffer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import smile.math.MathEx;
import smile.math.blas.BLAS;
import smile.math.blas.LAPACK;
import smile.math.blas.Layout;
import smile.math.blas.Transpose;
import smile.math.blas.UPLO;
import smile.math.matrix.FloatMatrix;
import smile.math.matrix.SMatrix;

public class FloatSymmMatrix
extends SMatrix {
    private static final long serialVersionUID = 2L;
    private static final Logger logger = LoggerFactory.getLogger(FloatSymmMatrix.class);
    final float[] AP;
    final int n;
    final UPLO uplo;

    public FloatSymmMatrix(UPLO uplo, int n) {
        if (uplo == null) {
            throw new NullPointerException("UPLO is null");
        }
        this.uplo = uplo;
        this.n = n;
        this.AP = new float[n * (n + 1) / 2];
    }

    public FloatSymmMatrix(UPLO uplo, float[][] AP) {
        this(uplo, AP.length);
        if (uplo == UPLO.LOWER) {
            for (int i = 0; i < this.n; ++i) {
                for (int j = 0; j <= i; ++j) {
                    this.AP[i + (2 * this.n - j - 1) * j / 2] = AP[i][j];
                }
            }
        } else {
            for (int i = 0; i < this.n; ++i) {
                for (int j = i; j < this.n; ++j) {
                    this.AP[i + j * (j + 1) / 2] = AP[i][j];
                }
            }
        }
    }

    public FloatSymmMatrix clone() {
        FloatSymmMatrix matrix = new FloatSymmMatrix(this.uplo, this.n);
        System.arraycopy(this.AP, 0, matrix.AP, 0, this.AP.length);
        return matrix;
    }

    @Override
    public int nrows() {
        return this.n;
    }

    @Override
    public int ncols() {
        return this.n;
    }

    @Override
    public long size() {
        return this.AP.length;
    }

    public Layout layout() {
        return Layout.COL_MAJOR;
    }

    public UPLO uplo() {
        return this.uplo;
    }

    public boolean equals(Object o) {
        if (o == null || !(o instanceof FloatSymmMatrix)) {
            return false;
        }
        return this.equals((FloatSymmMatrix)o, 1.0E-7f);
    }

    public boolean equals(FloatSymmMatrix o, float eps) {
        if (this.n != o.n) {
            return false;
        }
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.n; ++i) {
                if (MathEx.isZero(this.get(i, j) - o.get(i, j), eps)) continue;
                return false;
            }
        }
        return true;
    }

    @Override
    public float get(int i, int j) {
        if (this.uplo == UPLO.LOWER) {
            if (j > i) {
                int tmp = i;
                i = j;
                j = tmp;
            }
            return this.AP[i + (2 * this.n - j - 1) * j / 2];
        }
        if (i > j) {
            int tmp = i;
            i = j;
            j = tmp;
        }
        return this.AP[i + j * (j + 1) / 2];
    }

    @Override
    public FloatSymmMatrix set(int i, int j, float x) {
        if (this.uplo == UPLO.LOWER) {
            if (j > i) {
                int tmp = i;
                i = j;
                j = tmp;
            }
            this.AP[i + (2 * this.n - j - 1) * j / 2] = x;
        } else {
            if (i > j) {
                int tmp = i;
                i = j;
                j = tmp;
            }
            this.AP[i + j * (j + 1) / 2] = x;
        }
        return this;
    }

    @Override
    public void mv(Transpose trans, float alpha, float[] x, float beta, float[] y) {
        BLAS.engine.spmv(this.layout(), this.uplo, this.n, alpha, this.AP, x, 1, beta, y, 1);
    }

    @Override
    public void mv(float[] work, int inputOffset, int outputOffset) {
        FloatBuffer xb = FloatBuffer.wrap(work, inputOffset, this.n);
        FloatBuffer yb = FloatBuffer.wrap(work, outputOffset, this.n);
        BLAS.engine.spmv(this.layout(), this.uplo, this.n, 1.0f, FloatBuffer.wrap(this.AP), xb, 1, 0.0f, yb, 1);
    }

    @Override
    public void tv(float[] work, int inputOffset, int outputOffset) {
        this.mv(work, inputOffset, outputOffset);
    }

    public BunchKaufman bk() {
        FloatSymmMatrix lu = this.clone();
        int[] ipiv = new int[this.n];
        int info = LAPACK.engine.sptrf(lu.layout(), lu.uplo, lu.n, lu.AP, ipiv);
        if (info < 0) {
            logger.error("LAPACK SPTRF error code: {}", (Object)info);
            throw new ArithmeticException("LAPACK SPTRF error code: " + info);
        }
        return new BunchKaufman(lu, ipiv, info);
    }

    public Cholesky cholesky() {
        if (this.uplo == null) {
            throw new IllegalArgumentException("The matrix is not symmetric");
        }
        FloatSymmMatrix lu = this.clone();
        int info = LAPACK.engine.pptrf(lu.layout(), lu.uplo, lu.n, lu.AP);
        if (info != 0) {
            logger.error("LAPACK PPTRF error code: {}", (Object)info);
            throw new ArithmeticException("LAPACK PPTRF error code: " + info);
        }
        return new Cholesky(lu);
    }

    public static class Cholesky {
        public final FloatSymmMatrix lu;

        public Cholesky(FloatSymmMatrix lu) {
            if (lu.nrows() != lu.ncols()) {
                throw new UnsupportedOperationException("Cholesky constructor on a non-square matrix");
            }
            this.lu = lu;
        }

        public float det() {
            float d = 1.0f;
            for (int i = 0; i < this.lu.n; ++i) {
                d *= this.lu.get(i, i);
            }
            return d * d;
        }

        public FloatMatrix inverse() {
            FloatMatrix inv = FloatMatrix.eye(this.lu.n);
            this.solve(inv);
            return inv;
        }

        public float[] solve(float[] b) {
            float[] x = (float[])b.clone();
            this.solve(new FloatMatrix(x));
            return x;
        }

        public void solve(FloatMatrix B) {
            if (B.m != this.lu.n) {
                throw new IllegalArgumentException(String.format("Row dimensions do not agree: A is %d x %d, but B is %d x %d", this.lu.n, this.lu.n, B.m, B.n));
            }
            int info = LAPACK.engine.pptrs(this.lu.layout(), this.lu.uplo, this.lu.n, B.n, FloatBuffer.wrap(this.lu.AP), B.A, B.ld);
            if (info != 0) {
                logger.error("LAPACK POTRS error code: {}", (Object)info);
                throw new ArithmeticException("LAPACK POTRS error code: " + info);
            }
        }
    }

    public static class BunchKaufman {
        public final FloatSymmMatrix lu;
        public final int[] ipiv;
        public final int info;

        public BunchKaufman(FloatSymmMatrix lu, int[] ipiv, int info) {
            this.lu = lu;
            this.ipiv = ipiv;
            this.info = info;
        }

        public boolean isSingular() {
            return this.info > 0;
        }

        public float det() {
            int j;
            int n = this.lu.n;
            float d = 1.0f;
            for (j = 0; j < n; ++j) {
                d *= this.lu.get(j, j);
            }
            for (j = 0; j < n; ++j) {
                if (j + 1 == this.ipiv[j]) continue;
                d = -d;
            }
            return d;
        }

        public FloatMatrix inverse() {
            FloatMatrix inv = FloatMatrix.eye(this.lu.n);
            this.solve(inv);
            return inv;
        }

        public float[] solve(float[] b) {
            float[] x = (float[])b.clone();
            this.solve(new FloatMatrix(x));
            return x;
        }

        public void solve(FloatMatrix B) {
            if (B.m != this.lu.n) {
                throw new IllegalArgumentException(String.format("Row dimensions do not agree: A is %d x %d, but B is %d x %d", this.lu.n, this.lu.n, B.m, B.n));
            }
            if (this.lu.layout() != B.layout()) {
                throw new IllegalArgumentException("The matrix layout is inconsistent.");
            }
            if (this.info > 0) {
                throw new RuntimeException("The matrix is singular.");
            }
            int ret = LAPACK.engine.sptrs(this.lu.layout(), this.lu.uplo, this.lu.n, B.n, FloatBuffer.wrap(this.lu.AP), IntBuffer.wrap(this.ipiv), B.A, B.ld);
            if (ret != 0) {
                logger.error("LAPACK GETRS error code: {}", (Object)ret);
                throw new ArithmeticException("LAPACK GETRS error code: " + ret);
            }
        }
    }
}

