/*
 * #%L
 * Wikitty :: generators
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2010 CodeLutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.wikitty.generator;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.lang.StringEscapeUtils;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.eugene.Transformer;
import org.nuiton.eugene.java.ObjectModelTransformerToJava;
import org.nuiton.eugene.models.object.ObjectModel;
import org.nuiton.eugene.models.object.ObjectModelAttribute;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelModifier;
import org.nuiton.eugene.models.object.ObjectModelOperation;





/**
 * @plexus.component role="org.nuiton.eugene.Template" role-hint="org.nuiton.wikitty.generator.WikittyAbstractGenerator"
 */
public class WikittyAbstractGenerator extends ObjectModelTransformerToJava {

    private static final Log log = LogFactory.getLog(WikittyAbstractGenerator.class);
    
    /** map busines entity from source model to generated abstract class */
    protected Map<ObjectModelClass, ObjectModelClass> processedClasses = 
                new HashMap<ObjectModelClass, ObjectModelClass>();

    /** pattern to parse toString tagValue
     * for the string "abc{foo|bar}defrzeg{uvw|xyz}oeira"
     * will match {foo|bar} and {uvw|xyz} with groups for foo, bar, uvw and xyz
     */
    protected Pattern toStringTagValuePattern = Pattern.compile("\\{(([^|}])*)(?:\\|([^}]*))?\\}");

    /** map "Client.name" to "getName()" or any getter to read this attribute
     * those getter names are stored while generating in order to be found
     * later when generating toString() */
    protected Map<String, String> attributeToGetterName = new HashMap<String, String>();

    @Override
    protected Transformer<ObjectModel, ObjectModel> initPreviousTransformer() {
        return new WikittyPurifierTransformer();
    }

    @Override
    public void transformFromModel(ObjectModel model) {

        // contains boths businessEntities and metaExtensions
        // elements may have one of the two sterotypes or both
        // but elements with none of them aren't in this list
        List<ObjectModelClass> modelBoth = new ArrayList<ObjectModelClass>();

        // fill modelBusinessEntities with entities found in model
        for (ObjectModelClass clazz : model.getClasses()) {
            if (WikittyTransformerUtil.isBusinessEntity(clazz) ||
                WikittyTransformerUtil.isMetaExtension(clazz)) {
                modelBoth.add(clazz);
            }
        }

        for (ObjectModelClass clazz : modelBoth) {
            ObjectModelClass abstractClass = createAbstractClass(
                    WikittyTransformerUtil.businessEntityToAbstractName(clazz),
                    clazz.getPackageName());
            processedClasses.put(clazz, abstractClass);
            setSuperClass(abstractClass, "BusinessEntityWikitty");
            addInterface(abstractClass, clazz.getQualifiedName());
        }

        for (ObjectModelClass clazz : modelBoth) {
            if (WikittyTransformerUtil.isMetaExtension(clazz)) {
                addMetaExtensionOperations(clazz, processedClasses.get(clazz));
            } else if (WikittyTransformerUtil.isBusinessEntity(clazz)) {
                addOperations(clazz, processedClasses.get(clazz));
            }
        }
        
        // at this time, all operations in generated abstracts are just the operations
        // like get/set etc. we will copy all operations of a given class to all children
        // that's why constructors and others operations are not yet added
        for (ObjectModelClass clazz : modelBoth) {
            if (WikittyTransformerUtil.isBusinessEntity(clazz)) {
                addInheritedOperations(clazz, processedClasses.get(clazz));
            }
        }

        for (ObjectModelClass clazz : modelBoth) {
            ObjectModelClass abstractClassForThisEntity = processedClasses.get(clazz);
            addImports(abstractClassForThisEntity);
            addConstructors(abstractClassForThisEntity);
            addConstants(clazz, abstractClassForThisEntity);
            addToString(clazz, abstractClassForThisEntity);
        }

        processedClasses.clear();
    }

    protected void addImports(ObjectModelClass clazz) {
        // TODO 20100811 bleny remove unused imports
        addImport(clazz, WikittyTransformerUtil.BUSINESS_ENTITY_CLASS_FQN);
        addImport(clazz, WikittyTransformerUtil.BUSINESS_ENTITY_WIKITTY_CLASS_FQN);
        addImport(clazz, WikittyTransformerUtil.WIKITTY_CLASS_FQN);
        addImport(clazz, "org.nuiton.wikitty.WikittyExtension");
        addImport(clazz, "org.nuiton.wikitty.WikittyUtil");
        addImport(clazz, "org.nuiton.wikitty.WikittyUser");
        addImport(clazz, "org.nuiton.wikitty.WikittyUserAbstract");
        addImport(clazz, "org.nuiton.wikitty.WikittyUserImpl");
        addImport(clazz, "org.nuiton.wikitty.WikittyTreeNode");
        addImport(clazz, "org.nuiton.wikitty.WikittyTreeNodeAbstract");
        addImport(clazz, "org.nuiton.wikitty.WikittyTreeNodeImpl");
        addImport(clazz, java.util.List.class);
        addImport(clazz, java.util.ArrayList.class);
        addImport(clazz, java.util.Collection.class);
        addImport(clazz, java.util.Collections.class);
        addImport(clazz, java.util.Set.class);
        addImport(clazz, java.util.Date.class);
        addImport(clazz, java.util.LinkedHashSet.class);
    }
    
    protected void addSerialVersionUID(ObjectModelClass clazz) {        
        // adding a generated serialVersionUID
        Random random = new Random();
        Long serialVersionUIDs = random.nextLong();
        addConstant(clazz,
                   "serialVersionUID",
                   "long",
                   serialVersionUIDs.toString() + "L",
                   ObjectModelModifier.PRIVATE);
    }
    
    protected void addConstructors(ObjectModelClass clazz) {
        
        ObjectModelOperation constructor = addConstructor(clazz, ObjectModelModifier.PUBLIC);
        setOperationBody(constructor, ""
+"\n"
+"        super();\n"
+"");

        constructor = addConstructor(clazz, ObjectModelModifier.PUBLIC);
        addParameter(constructor, WikittyTransformerUtil.WIKITTY_CLASS_FQN, "wikitty");
        setOperationBody(constructor, ""
+"\n"
+"        super(wikitty);\n"
+"");

        constructor = addConstructor(clazz, ObjectModelModifier.PUBLIC);
        addParameter(constructor, WikittyTransformerUtil.BUSINESS_ENTITY_WIKITTY_CLASS_FQN, "businessEntityWikitty");
        setOperationBody(constructor, ""
+"\n"
+"        super(businessEntityWikitty.getWikitty());\n"
+"");
        
    }
    
    protected void addConstants(ObjectModelClass businessEntity, ObjectModelClass abstractClass) {
        // adding some constants about extension to abstract
        addConstant(abstractClass, "extensions", "List<WikittyExtension>", null, ObjectModelModifier.PUBLIC);
        addConstant(abstractClass, "extension" + businessEntity.getName(), "WikittyExtension", null, ObjectModelModifier.PUBLIC);

        // ... and a getter
        ObjectModelOperation getStaticExtensions = addOperation(abstractClass, "getStaticExtensions", "Collection<WikittyExtension>", ObjectModelModifier.PUBLIC);
        addAnnotation(abstractClass, getStaticExtensions, "Override");
        setOperationBody(getStaticExtensions, ""
+"\n"
+"        return extensions;\n"
+"");

        //// preparing a static block to initialize those constants
        ObjectModelOperation staticInitialization = addBlock(abstractClass, ObjectModelModifier.STATIC);

        // generating constructor call for extensionClient
        // we will build a string to write a call
        List<String> buildFieldMapExtensionParameters = new ArrayList<String>();


        
        // now process attributes
        for(ObjectModelAttribute attribute : businessEntity.getAttributes()) {
            if (attribute.isNavigable()) {                
                // now add the attribute to the piece of code that build the extension
                String wikittyType = WikittyTransformerUtil.typeToWikittyColumn(attribute.getType());
                String multiplicity = "";
                if (attribute.getMinMultiplicity() != 1 ||
                    attribute.getMaxMultiplicity() != 1) {
                    // generate a string like [1-10] or [0-*] etc.
                    multiplicity = "["
                                 + attribute.getMinMultiplicity()
                                 + "-"
                                 + (attribute.getMaxMultiplicity() == -1 ? "*" : attribute.getMaxMultiplicity())
                                 + "]";
                }
                
                // generate a string line like " unique=true" or ""
                String unique = attribute.isUnique() ? " unique=true" : "";
                // generate a string line like " deprecated=true" or ""
                String deprecated = attribute.hasTagValue("deprecated") ? " deprecated=" + attribute.getTagValue("deprecated") : "";
                // generate a string line like ' documentation="my documentation"' or ""
                String attributeDocumentation =
                        attribute.hasTagValue("documentation") ? " documentation=\\\""
                        + StringEscapeUtils.escapeJava(attribute.getTagValue("documentation"))
                        + "\\\"": "";
                // generate a string like " notNull=true" or ""
                String notNull = attribute.hasTagValue("notNull") ? " notNull=" + attribute.getTagValue("notNull") : "";
                buildFieldMapExtensionParameters.add("" // generate a line like '"Wikitty attributName[0-*] unique=true deprecated=true documentation=\"my documentation\""'
+"                                             \""+wikittyType+" "+attribute.getName()+""+multiplicity+""+unique+""+deprecated+""+notNull+""+attributeDocumentation+"\"");
            }
        }
        
        // finishing static block
        String extensionVersion = businessEntity.getTagValue("version");
        if (extensionVersion == null || "".equals(extensionVersion)) {
            extensionVersion = "1.0";
            log.warn("no version specified in model for " + businessEntity.getQualifiedName() + " using " + extensionVersion);
        }
        
        // a piece of code used in the static block
        String requires = null;
        for (ObjectModelClass superClass : businessEntity.getSuperclasses()) {
            // using "for" but there will be 0 or 1 iteration
            requires = WikittyTransformerUtil.classToExtensionVariableName(superClass, true);            
        }

        String buildFieldMapExtensionParametersInLine = StringUtils.join(buildFieldMapExtensionParameters, ", \n");
        String extensionVariableName = WikittyTransformerUtil.classToExtensionVariableName(businessEntity, false);
        String staticInitializationBody = ""
+"\n"
+"        extension"+businessEntity.getName()+" =\n"
+"                new WikittyExtension("+extensionVariableName+",\n"
+"                                     \""+extensionVersion+"\", // version\n"
+"                                     "+ requires +",\n"
+"                                     WikittyUtil.buildFieldMapExtension( // building field map\n"
+""+buildFieldMapExtensionParametersInLine+"));\n"
+"\n"
+"        // init extensions\n"
+"        List<WikittyExtension> exts = new ArrayList<WikittyExtension>();\n"
+"";

        for (ObjectModelClass superClass : businessEntity.getSuperclasses()) {
            // using "for" but there will be 0 or 1 iteration
            staticInitializationBody += ""
+"\n"
+"        exts.addAll("+superClass.getName()+"Abstract.extensions); \n"
+"        // current after requires ones\n"
+"";
        }

        staticInitializationBody += ""
+"\n"
+"        exts.add(extension"+businessEntity.getName()+");\n"
+"        extensions = Collections.unmodifiableList(exts);\n"
+"";
        setOperationBody(staticInitialization, staticInitializationBody);
        
    }
    
    protected void addOperations(ObjectModelClass businessEntity, ObjectModelClass abstractClass) {
        String extensionVariableName = WikittyTransformerUtil.classToExtensionVariableName(businessEntity, true);
        String helperClassName = WikittyTransformerUtil.businessEntityToHelperName(businessEntity);
        
        // generating operations with bodies to realize contract
        for (ObjectModelAttribute attribute : businessEntity.getAttributes()) {
            if (attribute.isNavigable()) {
                // needed below, in templates
                String fieldVariableName = WikittyTransformerUtil.attributeToFielVariableName(attribute, true);
                String attributeType = WikittyTransformerUtil.generateResultType(attribute, false);
    
                String attributeName = attribute.getName();
                if (attribute.hasTagValue(WikittyTransformerUtil.TAG_ALTERNATIVE_NAME)) {
                    // there is a conflict, purifier transformer give as the right name to use
                    attributeName = attribute.getTagValue(WikittyTransformerUtil.TAG_ALTERNATIVE_NAME);
                }

                String getterName = null;

                if (WikittyTransformerUtil.isAttributeCollection(attribute)) {
                    // attributed is a collection, we will generate operations get, add, remove and clear
    
                    String attributeTypeSimpleNameInSet = WikittyTransformerUtil.generateResultType(attribute, true);
                    String getFieldMethodName = WikittyTransformerUtil.generateGetFieldAsCall(attribute);
            
                    // now, for this attribute, we will generate add, remove and clear methods
                    // adding operations to contract
                    getterName = "get" + StringUtils.capitalize(attributeName);
                    ObjectModelOperation getter = addOperation(abstractClass, getterName, attributeTypeSimpleNameInSet);
                    addAnnotation(abstractClass, getter, "Override");
                    String getterBody = ""
+"\n"
+"        "+attributeTypeSimpleNameInSet+" result = "+helperClassName+"."+getterName+"(getWikitty());\n"
+"        return result;\n"
+"";            
                    setOperationBody(getter, getterBody);
            
                    String addName = "add" + StringUtils.capitalize(attributeName);
                    ObjectModelOperation adder = addOperation(abstractClass, addName, "void");
                    addAnnotation(abstractClass, adder, "Override");
                    addParameter(adder, "String", "element");
                    String adderBody = ""
+"\n"
+"        "+helperClassName+"."+addName+"(getWikitty(), element);\n"
+"        getPropertyChangeSupport().firePropertyChange("+fieldVariableName+", null, "+ getter.getName() +"());\n"
+"";
                    setOperationBody(adder, adderBody);

                    String removeName = "remove" + StringUtils.capitalize(attributeName);
                    ObjectModelOperation remover = addOperation(abstractClass, removeName, "void");
                    addAnnotation(abstractClass, remover, "Override");
                    addParameter(remover, "String", "element");
                    String removerBody = ""
+"\n"
+"        "+helperClassName+"."+removeName+"(getWikitty(), element);\n"
+"        getPropertyChangeSupport().firePropertyChange("+fieldVariableName+", null, "+getter.getName()+"());\n"
+"";
                    setOperationBody(remover, removerBody);

                    String clearName = "clear" + StringUtils.capitalize(attributeName);
                    ObjectModelOperation clear = addOperation(abstractClass, clearName, "void");
                    addAnnotation(abstractClass, clear, "Override");
                    String clearBody = ""
+"\n"
+"        "+helperClassName+"."+clearName+"(getWikitty());\n"
+"        getPropertyChangeSupport().firePropertyChange("+fieldVariableName+", null, "+getter.getName()+"());\n"
+"";
                    setOperationBody(clear, clearBody);

                } else {
                    String getFieldMethodName = WikittyTransformerUtil.generateGetFieldAsCall(attribute);
                    
                    // adding getter and setter to contract
                    getterName = "get" + StringUtils.capitalize(attributeName);
                    ObjectModelOperation getter = addOperation(abstractClass, getterName, attributeType);
                    addAnnotation(abstractClass, getter, "Override");
                    setOperationBody(getter, ""
+"\n"
+"        "+attributeType+" value = "+helperClassName+"."+getterName+"(getWikitty());\n"
+"        return value;\n"
+"");

                    String setterName = "set" + StringUtils.capitalize(attributeName);
                    ObjectModelOperation setter = addOperation(abstractClass, setterName, "void");
                    addAnnotation(abstractClass, setter, "Override");
                    addParameter(setter, attributeType, attributeName);
                    setOperationBody(setter, ""
+"\n"
+"        "+attributeType+" oldValue = "+getterName+"();\n"
+"        "+helperClassName+"."+setterName+"(getWikitty(), "+attributeName+");\n"
+"        getPropertyChangeSupport().firePropertyChange("+fieldVariableName+", oldValue, "+getter.getName()+"());\n"
+"");
                }
                
                // save the getter name for this attribute
                attributeToGetterName.put(businessEntity.getName() + "." + attributeName, getterName);
            }
        }
    }

    protected void addInheritedOperations(ObjectModelClass businessEntity, ObjectModelClass abstractClass) {
        // now, add to this abstract all operation due to inheritence from
        // other business entities

        for (ObjectModelClass superClass : businessEntity.getSuperclasses()) {

            // if super class is not in the same package, import it Helper
            if (! businessEntity.getPackageName().equals(superClass.getPackageName())) {
                addImport(abstractClass, superClass.getPackageName() + "." +
                        WikittyTransformerUtil.businessEntityToHelperName(superClass));
            }
            
            if (WikittyTransformerUtil.isBusinessEntity(superClass)) {
                addInterface(abstractClass, WikittyTransformerUtil.businessEntityToContractName(superClass)); // extends
                // getting the signatures and bodies of those operations
                for (ObjectModelOperation operation : processedClasses.get(superClass).getOperations()) {
                    
                    ObjectModelOperation operationClone = cloneOperationSignature(operation, abstractClass, true);
                    setOperationBody(operationClone, operation.getBodyCode());
                    
                    // XXX 20100816 bleny should be a call to cloneOperation(operation, abstractClass, true);
                }
            } else {
                addInterface(abstractClass, superClass.getQualifiedName()); // extends
            }
        }
    }
    
    /** add a toString method
     * if a toString is tagValue is attached to businessEntity, it will be used
     * to generate a toString as this :
     * 
     * given "hello {Person.name|unknow}"
     * 
     * will try to replace first {...} by name field value for extension Person.
     * if this information is not available, will do unknow.
     */ 
    protected void addToString(ObjectModelClass businessEntity, ObjectModelClass abstractClass) {

        String toStringOperationBody = null;
        
        if (businessEntity.hasTagValue(WikittyTransformerUtil.TAG_TO_STRING)) {
            String toStringPattern = businessEntity.getTagValue(WikittyTransformerUtil.TAG_TO_STRING);
            
            // toStringPattern is something like
            // "hello {Person.name|unknow} employe of {Company.name|unknow}"
            // 

            Matcher matcher = toStringTagValuePattern.matcher(toStringPattern);

            while (matcher.find()) {
                String wholeMatch = matcher.group(0); // "{foo|bar}"
                String variableName = matcher.group(1); // "foo"
                String defaultValue = matcher.group(3); // "bar", may be null

                if (defaultValue == null) {
                    defaultValue = "";
                }

                if (attributeToGetterName.containsKey(variableName)) {
                    String getterName =  attributeToGetterName.get(variableName);
                    toStringPattern = toStringPattern.replace(wholeMatch, ""
+"\"\n"
+"             + "+getterName+"().toString() + \n"
+"             \"");
                } else {
                    log.warn("no field " + variableName + " in " + businessEntity.getQualifiedName());
                    toStringPattern = toStringPattern.replace(wholeMatch, defaultValue);
                }
            }

            toStringOperationBody = ""
+"\n"
+"        return \""+toStringPattern+"\";\n"
+"";
        } else {
            // no toString tagValue provided, generating a default toString
            toStringOperationBody = ""
+"\n"
+"        return getWikitty().toString();\n"
+"";
        }

        ObjectModelOperation toString = addOperation(abstractClass, "toString", "String");
        addAnnotation(abstractClass, toString, "Override");
        setOperationBody(toString, toStringOperationBody);
    }
    
    protected void addMetaExtensionOperations(ObjectModelClass metaExtension,
                           ObjectModelClass abstractClassForThisMetaExtension) {

        String helperClassName = WikittyTransformerUtil.businessEntityToHelperName(metaExtension);

        ObjectModelAttribute extension = addAttribute(abstractClassForThisMetaExtension, "extensionForMetaExtension", WikittyTransformerUtil.WIKITTY_EXTENSION_CLASS_FQN);
        setDocumentation(extension, "the metaExtension operations target this extension, may be null");

        ObjectModelOperation setExtensionForMetaExtension = addOperation(abstractClassForThisMetaExtension, "setExtensionForMetaExtension", "void");
        addParameter(setExtensionForMetaExtension, WikittyTransformerUtil.WIKITTY_EXTENSION_CLASS_FQN, "extension");
        setDocumentation(setExtensionForMetaExtension, String.format(
                "add %s meta-extension on given extension to this entity",
                metaExtension.getName()));
        setOperationBody(setExtensionForMetaExtension, ""
+"\n"
+"        extensionForMetaExtension = extension;\n"
+"        "+helperClassName+".addMetaExtension(extension, getWikitty());\n"
+"");

        // generating operations with bodies to realize contract
        for (ObjectModelAttribute attribute : metaExtension.getAttributes()) {
            if (attribute.isNavigable()) {
                // needed below, in templates
                String fieldVariableName = WikittyTransformerUtil.attributeToFielVariableName(attribute, true);
                String attributeType = WikittyTransformerUtil.generateResultType(attribute, false);
    
                String attributeName = attribute.getName();
                if (attribute.hasTagValue(WikittyTransformerUtil.TAG_ALTERNATIVE_NAME)) {
                    // there is a conflict, purifier transformer give as the right name to use
                    attributeName = attribute.getTagValue(WikittyTransformerUtil.TAG_ALTERNATIVE_NAME);
                }

                String getterName = null;

                if (WikittyTransformerUtil.isAttributeCollection(attribute)) {
                    // attributed is a collection, we will generate operations get, add, remove and clear
    
                    String attributeTypeSimpleNameInSet = WikittyTransformerUtil.generateResultType(attribute, true);
                    String getFieldMethodName = WikittyTransformerUtil.generateGetFieldAsCall(attribute);
            
                    // now, for this attribute, we will generate add, remove and clear methods
                    // adding operations to contract
                    getterName = "get" + StringUtils.capitalize(attributeName);
                    ObjectModelOperation getter = addOperation(abstractClassForThisMetaExtension, getterName, attributeTypeSimpleNameInSet);
                    addAnnotation(abstractClassForThisMetaExtension, getter, "Override");
                    String getterBody = ""
+"\n"
+"        "+attributeTypeSimpleNameInSet+" result;\n"
+"        if (extensionForMetaExtension == null) {\n"
+"            result = "+helperClassName+"."+getterName+"(getWikitty());\n"
+"        } else {\n"
+"            result = "+helperClassName+"."+getterName+"(extensionForMetaExtension.getName(), getWikitty());\n"
+"        }\n"
+"        return result;\n"
+"";            
                    setOperationBody(getter, getterBody);
            
                    String addName = "add" + StringUtils.capitalize(attributeName);
                    ObjectModelOperation adder = addOperation(abstractClassForThisMetaExtension, addName, "void");
                    addAnnotation(abstractClassForThisMetaExtension, adder, "Override");
                    addParameter(adder, "String", "element");
                    String adderBody = ""
+"\n"
+"        if (extensionForMetaExtension == null) {\n"
+"            "+helperClassName+"."+addName+"(getWikitty(), element);\n"
+"            getPropertyChangeSupport().firePropertyChange("+fieldVariableName+", null, "+getter.getName()+"());\n"
+"        } else {\n"
+"            "+helperClassName+"."+addName+"(extensionForMetaExtension.getName(), getWikitty(), element);\n"
+"            String fieldName = "+helperClassName+".getMetaFieldName(extensionForMetaExtension, \""+attributeName+"\");\n"
+"            getPropertyChangeSupport().firePropertyChange(fieldName, null, "+getter.getName()+"());\n"
+"        }\n"
+"";
                    setOperationBody(adder, adderBody);
                    
                    String removeName = "remove" + StringUtils.capitalize(attributeName);
                    ObjectModelOperation remover = addOperation(abstractClassForThisMetaExtension, removeName, "void");
                    addAnnotation(abstractClassForThisMetaExtension, remover, "Override");
                    addParameter(remover, "String", "element");
                    String removerBody = ""
+"\n"
+"        if (extensionForMetaExtension == null) {\n"
+"            "+helperClassName+"."+removeName+"(getWikitty(), element);\n"
+"            getPropertyChangeSupport().firePropertyChange("+fieldVariableName+", null, "+getter.getName()+"());\n"
+"        } else {\n"
+"            "+helperClassName+"."+removeName+"(extensionForMetaExtension.getName(), getWikitty(), element);\n"
+"            String fieldName = "+helperClassName+".getMetaFieldName(extensionForMetaExtension, \""+attributeName+"\");\n"
+"            getPropertyChangeSupport().firePropertyChange(fieldName, null, "+getter.getName()+"());\n"
+"        }\n"
+"";
                    setOperationBody(remover, removerBody);
                    
                    String clearName = "clear" + StringUtils.capitalize(attributeName);
                    ObjectModelOperation clear = addOperation(abstractClassForThisMetaExtension, clearName, "void");
                    addAnnotation(abstractClassForThisMetaExtension, clear, "Override");
                    String clearBody = ""
+"\n"
+"        if (extensionForMetaExtension == null) {\n"
+"            "+helperClassName+"."+clearName+"(getWikitty());\n"
+"            getPropertyChangeSupport().firePropertyChange("+fieldVariableName+", null, "+getter.getName()+"());\n"
+"        } else {\n"
+"            "+helperClassName+"."+clearName+"(extensionForMetaExtension.getName(), getWikitty());\n"
+"            String fieldName = "+helperClassName+".getMetaFieldName(extensionForMetaExtension, \""+attributeName+"\");\n"
+"            getPropertyChangeSupport().firePropertyChange(fieldName, null, "+getter.getName()+"());\n"
+"        }\n"
+"";
                    setOperationBody(clear, clearBody);
    
                    
                } else {
                    String getFieldMethodName = WikittyTransformerUtil.generateGetFieldAsCall(attribute);
                    
                    // adding getter and setter to contract
                    getterName = "get" + StringUtils.capitalize(attributeName);
                    ObjectModelOperation getter = addOperation(abstractClassForThisMetaExtension, getterName, attributeType);
                    addAnnotation(abstractClassForThisMetaExtension, getter, "Override");
                    setOperationBody(getter, ""
+"\n"
+"        "+attributeType+" value;\n"
+"        if (extensionForMetaExtension == null) {\n"
+"            value = "+helperClassName+"."+getterName+"(getWikitty());\n"
+"        } else {\n"
+"            value = "+helperClassName+"."+getterName+"(extensionForMetaExtension.getName(), getWikitty());\n"
+"        }\n"
+"        return value;\n"
+"");
                    
                    String setterName = "set" + StringUtils.capitalize(attributeName);
                    ObjectModelOperation setter = addOperation(abstractClassForThisMetaExtension, setterName, "void");
                    addAnnotation(abstractClassForThisMetaExtension, setter, "Override");
                    addParameter(setter, attributeType, attributeName);
                    setOperationBody(setter, ""
+"\n"
+"        "+attributeType+" oldValue;\n"
+"        if (extensionForMetaExtension == null) {\n"
+"            oldValue = "+getterName+"();\n"
+"            "+helperClassName+"."+setterName+"(getWikitty(), "+attributeName+");\n"
+"            getPropertyChangeSupport().firePropertyChange("+fieldVariableName+", oldValue, "+getter.getName()+"());\n"
+"        } else {\n"
+"            oldValue = "+getterName+"();\n"
+"            "+helperClassName+"."+setterName+"(extensionForMetaExtension.getName(), getWikitty(), "+attributeName+");\n"
+"            String fieldName = "+helperClassName+".getMetaFieldName(extensionForMetaExtension, \""+attributeName+"\");\n"
+"            getPropertyChangeSupport().firePropertyChange(fieldName, oldValue, "+getter.getName()+"());\n"
+"        }\n"
+"");
                }
            }
        }
    }
}
