package org.nuiton.topia.service.sql.batch.actions;

/*
 * #%L
 * ToPIA :: Service Sql batch
 * %%
 * Copyright (C) 2004 - 2016 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.service.sql.batch.tables.TopiaSqlTable;

import java.io.IOException;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.List;

/**
 * Created on 01/01/16.
 *
 * @author Tony Chemit - chemit@codelutin.com
 * @since 3.0.1
 */
public class ReplicateTablesAction extends AbstractTablesAction<ReplicateTablesRequest> {

    public static final String INSERT_STATEMENT = "INSERT INTO %s.%s(%s) VALUES (%%s);\n";
    /**
     * Logger.
     */
    private static final Log log = LogFactory.getLog(ReplicateTablesAction.class);

    public ReplicateTablesAction(ReplicateTablesRequest request) {
        super(request);
    }

    @Override
    protected void executeOnTable(ReplicateTablesRequest request, TopiaSqlTable table, PreparedStatement readStatement) throws SQLException {

        ResultSet readResultSet = readStatement.getResultSet();

        ResultSetMetaData readResultSetMetaData = readResultSet.getMetaData();
        int columnCount = readResultSetMetaData.getColumnCount();

        List<String> columnNames = getColumnNames(readResultSetMetaData, columnCount);

        boolean useOutputWriter = useOutputWriter();
        boolean useOutputDb = useOutputDb();

        PreparedStatement writeStatement = null;

        String insertStatementSql = newInsertStatementSql(table, columnNames);

        if (useOutputDb) {

            String arguments = generateWildcardArguments(columnNames);
            String sql = String.format(insertStatementSql, arguments).trim();
            writeStatement = targetConnection.prepareStatement(sql);

        }

        int writeBatchSize = request.getWriteBatchSize();
        String tableName = table.getFullyTableName();

        long index = 0;
        while (readResultSet.next()) {

            if (log.isTraceEnabled()) {
                log.trace("Copy " + readResultSet.getString(1));
            }

            if (useOutputDb) {

                writeStatement.clearParameters();
                for (int i = 1; i <= columnCount; i++) {
                    Object object = readResultSet.getObject(i);
                    writeStatement.setObject(i, object);
                }
                writeStatement.addBatch();

            }

            if (useOutputWriter) {

                try {

                    String arguments = generateSqlArguments(readResultSet, columnNames);
                    String sql = String.format(insertStatementSql, arguments);
                    writer.append(sql);

                } catch (IOException e) {
                    throw new RuntimeException("Could not copyRow", e);
                }

            }

            if ((++index % writeBatchSize) == 0) {
                flush(writeStatement, writer, tableName, index);
            }

        }

        flush(writeStatement, writer, tableName, index);

    }

    protected String newInsertStatementSql(TopiaSqlTable table, List<String> columnNames) throws SQLException {

        StringBuilder columnNamesBuilder = new StringBuilder();

        for (String columnName : columnNames) {
            columnNamesBuilder.append(", ").append(columnName);
        }

        String sql = String.format(INSERT_STATEMENT,
                table.getSchemaName(),
                table.getTableName(),
                columnNamesBuilder.substring(2));
        if (log.isDebugEnabled()) {
            log.debug("Insert sql: " + sql);
        }

        return sql;

    }

}
