/*
 * *##%
 * JAXX Runtime
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package jaxx.runtime.swing.navigation;

import jaxx.runtime.JAXXContext;

import javax.swing.tree.TreeNode;
import java.util.regex.Pattern;

/**
 * Interface to create model of the tree used for a navigation tree.
 * <p/>
 * Il est composé de {@link NavigationTreeNode}
 *
 * @author sletellier
 * @since 2.0.0
 */
public interface NavigationModel {

    Object getRoot();

    TreeNode[] getPathToRoot(TreeNode aNode);

    /**
     * Search from the root node a node named by his fully path (concatenation
     * of nodes {@link NavigationTreeNode#path} valued separated by dot.
     * <p/>
     * Example :
     * <p/>
     * <pre>$root.child1.leaf1</pre>
     *
     * @param path the fully path of the searched node.
     * @return the node matching the fully context from the root node,
     *         or <code>null</code> if not find.
     */
    NavigationTreeNode findNode(String path);

    /**
     * Apply first the regex pattern to obtain the searched node fi the given
     * <code>regex</code> is not null.
     * <p/>
     * Search then from the root node a node named by his fully path
     * (concatenation of nodes {@link NavigationTreeNode#path} valued separated
     * by {@link #getPathSeparator()}.
     * <p/>
     * <p/>
     * Example :
     * <p/>
     * <pre>$root.child1.leaf1</pre>
     *
     * @param path  the fully path of the searched node.
     * @param regex a optional regex to apply to path before searching
     * @return the node matching the fully context from the root node, or
     *         <code>null</code> if not found.
     */
    NavigationTreeNode findNode(String path, String regex);

    /**
     * Apply first the regex pattern to obtain the searched node.
     * <p/>
     * Search then from the root node a node named by his fully path
     * (concatenation of nodes {@link NavigationTreeNode#path} valued separated
     * by {@link #getPathSeparator()}.
     * <p/>
     * Example :
     * <p/>
     * <pre>$root.child1.leaf1</pre>
     *
     * @param path  the fully path of the searched node.
     * @param regex a optional regex to apply to path before searching
     * @return the node matching the fully context from the root node, or
     *         <code>null</code> if not found.
     */
    NavigationTreeNode findNode(String path, Pattern regex);

    /**
     * Search from a given root node a node named by his fully path
     * (concatenation of nodes {@link NavigationTreeNode#path} valued separated
     * by {@link #getPathSeparator()}.
     *
     * @param root root node to be used
     * @param path the fully path of the searched node.
     * @return the node matching the fully context from the given root node, or
     *         <code>null</code> if not found.
     */
    NavigationTreeNode findNode(NavigationTreeNode root, String path);

    /**
     * Apply first the regex pattern to obtain the searched node.
     * <p/>
     * Search then from a given root node a node named by his fully path
     * (concatenation of nodes) {@link NavigationTreeNode#path} valued
     * separated by {@link #getPathSeparator()}.
     *
     * @param root  root node to be used
     * @param path  the fully path of the searched node.
     * @param regex a previous regex to apply to path : must have a matches
     * @return the node matching the fully context from the given root node,
     *         or <code>null</code> if not found.
     */
    NavigationTreeNode findNode(NavigationTreeNode root, String path,
                                String regex);

    /**
     * Apply first the regex pattern to obtain the searched node.
     * <p/>
     * Search then from a given root node a node named by his fully path
     * (concatenation of nodes {@link NavigationTreeNode#path} valued separated
     * by {@link #getPathSeparator()}.
     *
     * @param root  root node to be used
     * @param path  the fully path of the searched node.
     * @param regex a previous regex to apply to path : must have a matches
     * @return the node matching the fully context from the given root node,
     *         or <code>null</code> if not found.
     */
    NavigationTreeNode findNode(NavigationTreeNode root, String path,
                                Pattern regex);

    JAXXContext getContext();

    /**
     * Obtain the associated bean value from context corresponding to node
     * from given navigation path.
     *
     * @param navigationPath the current context path of the node
     * @return the value associated in context with the given navigation path
     */
    Object getBean(String navigationPath);

    /**
     * Obtain the associated bean value from context corresponding to node
     *
     * @param node the current node
     * @return the value associated in context with the given node.
     */
    Object getBean(NavigationTreeNode node);

    void nodeChanged(TreeNode node);

    void nodeStructureChanged(TreeNode node);

    void nodeChanged(TreeNode node, boolean deep);

    String getPathSeparator();
}
