/*
 * #%L
 * EUGene :: Maven plugin
 * %%
 * Copyright (C) 2006 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.eugene.plugin;

import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.nuiton.eugene.ModelReader;
import org.nuiton.eugene.Template;
import org.nuiton.eugene.models.Model;
import org.nuiton.eugene.models.object.ObjectModel;
import org.nuiton.eugene.models.stereotype.StereotypeDefinition;
import org.nuiton.eugene.models.stereotype.StereotypeDefinitionProvider;
import org.nuiton.eugene.models.stereotype.StereotypeDefinitionProviders;
import org.nuiton.eugene.models.tagvalue.TagValueDefinition;
import org.nuiton.eugene.models.tagvalue.TagValueDefinitionProvider;
import org.nuiton.eugene.models.tagvalue.TagValueDefinitionProviders;
import org.nuiton.eugene.writer.ChainedFileWriter;

import java.util.Arrays;
import java.util.EnumSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;

/**
 * Obtain the list of some known data informations.
 * <p/>
 * Use the {@code dataTypes} property to specify a specific data type to use
 * (otherwise will display all known data types).
 * <p/>
 *
 * @author Tony Chemit - chemit@codelutin.com
 * @since 2.0.0
 */
@Mojo(name = "available-data",
      requiresProject = true,
      requiresDirectInvocation = true,
      requiresDependencyResolution = ResolutionScope.TEST)
public class AvailableDataMojo extends AbstractMojo {

    /**
     * Data type to display (let empty to see all datas).
     * Can specify more than one separated by comma.
     * <p/>
     * Available types are :
     * <pre>
     * modeltype,
     * modelreader,
     * modeltemplate,
     * writer,
     * stereotype,
     * tagvalue
     * </pre>
     * <p/>
     * <b>Note:</b> Let empty to display all data types.
     *
     * @since 2.0.0
     */
    @Parameter(property = "dataTypes", defaultValue = "")
    protected String dataTypes;

    /**
     * All available models (obtain by plexus, keys are plexus roles, values
     * are a instance of corresponding model).
     */
    @Component(role = Model.class)
    protected Map<String, Model> modelTypes;

    /** All available writers introspects via plexus. */
    @Component(role = ModelReader.class)
    protected Map<String, ModelReader<?>> modelReaders;

    /** All available templates introspects via plexus. */
    @Component(role = Template.class)
    protected Map<String, Template<?>> modelTemplates;

    /** All available writers introspects via plexus. */
    @Component(role = ChainedFileWriter.class)
    protected Map<String, ChainedFileWriter> writers;

    /**
     * All available stereotype providers introspects via plexus.
     *
     * @since 2.9
     */
    @Component(role = StereotypeDefinitionProvider.class)
    protected Map<String, StereotypeDefinitionProvider> stereotypeDefinitionProviders;

    /**
     * All available tag value providers introspects via plexus.
     *
     * @since 2.9
     */
    @Component(role = TagValueDefinitionProvider.class)
    protected Map<String, TagValueDefinitionProvider> tagValueDefinitionProviders;

    protected Map<String, TagValueDefinitionProvider> loadedTagValueDefinitionProviders;

    protected Map<String, StereotypeDefinitionProvider> loadedStereotypeDefinitionProviders;

    protected TagValueDefinitionProvider currentTagValueDefinitionProvider;

    protected StereotypeDefinitionProvider currentStereotypeDefinitionProvider;

    @Override
    public void execute() throws MojoExecutionException, MojoFailureException {
        StringBuilder buffer = new StringBuilder();
        dataTypes = dataTypes == null ? "" : dataTypes.trim();

        Set<AvailableData> safeDataTypes;

        if (StringUtils.isEmpty(dataTypes)) {
            // treate all data types
            safeDataTypes = EnumSet.allOf(AvailableData.class);

            if (getLog().isDebugEnabled()) {
                getLog().debug("will use all data types : " + safeDataTypes);
            }

        } else {

            safeDataTypes = EnumSet.noneOf(AvailableData.class);
            for (String s : dataTypes.split(",")) {
                s = s.trim().toLowerCase();
                try {
                    AvailableData data = AvailableData.valueOf(s);
                    if (getLog().isDebugEnabled()) {
                        getLog().debug("will use data type " + data);
                    }
                    safeDataTypes.add(data);
                } catch (IllegalArgumentException e) {
                    getLog().warn(
                            "does not know data type : " + s + " use one of " +
                            Arrays.toString(AvailableData.values())
                    );
                }
            }
        }

        if (safeDataTypes.contains(AvailableData.stereotype)) {

            loadedStereotypeDefinitionProviders = new LinkedHashMap<String, StereotypeDefinitionProvider>();
            // init stores
            for (Map.Entry<String, StereotypeDefinitionProvider> e : stereotypeDefinitionProviders.entrySet()) {
                StereotypeDefinitionProvider provider = StereotypeDefinitionProviders.newProvider(
                        Arrays.asList(e.getValue()),
                        false
                );
                loadedStereotypeDefinitionProviders.put(e.getKey(), provider);
            }
        }

        if (safeDataTypes.contains(AvailableData.tagvalue)) {

            loadedTagValueDefinitionProviders = new LinkedHashMap<String, TagValueDefinitionProvider>();
            // init stores
            for (Map.Entry<String, TagValueDefinitionProvider> e : tagValueDefinitionProviders.entrySet()) {
                TagValueDefinitionProvider provider = TagValueDefinitionProviders.newProvider(
                        Arrays.asList(e.getValue()),
                        false
                );
                loadedTagValueDefinitionProviders.put(e.getKey(), provider);
            }
        }
        for (AvailableData data : safeDataTypes) {
            buffer.append("\n");
            appendData(data, buffer);
        }

        getLog().info("Get datas for data types : " + safeDataTypes +
                      buffer.toString());
    }

    protected void appendData(AvailableData data, StringBuilder buffer) {

        String dataType = data.name();

        if (data == AvailableData.tagvalue) {

            int nbData = 0;
            for (TagValueDefinitionProvider provider : loadedTagValueDefinitionProviders.values()) {
                currentTagValueDefinitionProvider = provider;
                nbData += data.getData(this).size();
            }
            currentTagValueDefinitionProvider = null;

            String format = "\nFound %s %ss in %s provider(s) : %s\n";

            buffer.append(String.format(format,
                                        nbData,
                                        dataType,
                                        loadedTagValueDefinitionProviders.size(),
                                        loadedTagValueDefinitionProviders.keySet())
            );

            for (Map.Entry<String, TagValueDefinitionProvider> e : loadedTagValueDefinitionProviders.entrySet()) {
                String providerName = e.getKey();
                TagValueDefinitionProvider provider = e.getValue();
                currentTagValueDefinitionProvider = provider;
                Map<String, ?> map = data.getData(this);

                int size = map.size();
                buffer.append("\nProvider [").append(providerName).append("] - ");
                if (size == 0) {
                    buffer.append("No available ").append(dataType).append(".");
                } else if (size == 1) {
                    buffer.append("Found one ").append(dataType).append(" : ");
                } else {
                    buffer.append("Found ");
                    buffer.append(size);
                    buffer.append(" ");
                    buffer.append(dataType);
                    buffer.append("s : ");
                }
                for (Map.Entry<String, ?> e2 : map.entrySet()) {
                    data.toString(buffer, e2);
                }
            }
        } else if (data == AvailableData.stereotype) {

            int nbData = 0;
            for (StereotypeDefinitionProvider provider : loadedStereotypeDefinitionProviders.values()) {
                currentStereotypeDefinitionProvider = provider;
                nbData += data.getData(this).size();
            }
            currentStereotypeDefinitionProvider = null;

            String format = "\nFound %s %ss in %s provider(s) : %s\n";

            buffer.append(String.format(format,
                                        nbData,
                                        dataType,
                                        loadedStereotypeDefinitionProviders.size(),
                                        loadedStereotypeDefinitionProviders.keySet())
            );

            for (Map.Entry<String, StereotypeDefinitionProvider> e : loadedStereotypeDefinitionProviders.entrySet()) {
                String providerName = e.getKey();
                StereotypeDefinitionProvider provider = e.getValue();
                currentStereotypeDefinitionProvider = provider;
                Map<String, ?> map = data.getData(this);

                int size = map.size();
                buffer.append("\nProvider [").append(providerName).append("] - ");
                if (size == 0) {
                    buffer.append("No available ").append(dataType).append(".");
                } else if (size == 1) {
                    buffer.append("Found one ").append(dataType).append(" : ");
                } else {
                    buffer.append("Found ");
                    buffer.append(size);
                    buffer.append(" ");
                    buffer.append(dataType);
                    buffer.append("s : ");
                }
                for (Map.Entry<String, ?> e2 : map.entrySet()) {
                    data.toString(buffer, e2);
                }
            }
        } else {
            Map<String, ?> map = data.getData(this);

            int size = map == null ? 0 : map.size();
            if (size == 0) {
                buffer.append("\nNo available ").append(dataType).append(".");
            } else {
                if (size == 1) {
                    buffer.append("\nFound one ").append(dataType).append(" : ");
                } else {
                    buffer.append("\nFound ");
                    buffer.append(size);
                    buffer.append(" ");
                    buffer.append(dataType);
                    buffer.append("s : ");
                }
                for (Map.Entry<String, ?> e : map.entrySet()) {
                    data.toString(buffer, e);
                }
            }
        }
    }

    enum AvailableData {
        modeltype {
            @Override
            public Map<String, ?> getData(AvailableDataMojo mojo) {
                return mojo.modelTypes;
            }
        },
        writer {
            @Override
            public Map<String, ?> getData(AvailableDataMojo mojo) {
                return mojo.writers;
            }

            @Override
            String toString(Object data) {
                ChainedFileWriter w = (ChainedFileWriter) data;
                StringBuilder b = new StringBuilder(super.toString(data));
                b.append("\n").append("  inputProtocol             : ");
                b.append(w.getInputProtocol());
                b.append("\n").append("  outputProtocol            : ");
                b.append(w.getOutputProtocol(ObjectModel.NAME));
                b.append("\n").append("  defaultIncludes           : ");
                b.append(w.getDefaultIncludes());
                b.append("\n").append("  defaultInputDirectory     : ");
                b.append(w.getDefaultInputDirectory());
                b.append("\n").append("  defaultTestInputDirectory : ");
                b.append(w.getDefaultTestInputDirectory());
                return b.toString();
            }
        },
        modelreader {
            @Override
            public Map<String, ?> getData(AvailableDataMojo mojo) {
                return mojo.modelReaders;
            }
        },
        modeltemplate {
            @Override
            public Map<String, ?> getData(AvailableDataMojo mojo) {
                return mojo.modelTemplates;
            }
        },
        tagvalue {
            @Override
            public Map<String, ?> getData(AvailableDataMojo mojo) {
                return mojo.getCurrentTagValueDefinitionProvider().getDefinition();
            }

            @Override
            String toString(Object data) {
                TagValueDefinition d = (TagValueDefinition) data;
                StringBuilder sb = new StringBuilder();
                Class<?>[] targets = d.target();
                for (Class<?> aClass : targets) {
                    sb.append(", ").append(aClass.getSimpleName());
                }
                String result = sb.toString();
                if (targets.length > 0) {
                    result = result.substring(2);
                }
                return result;
            }

            @Override
            void toString(StringBuilder buffer, Map.Entry<String, ?> e) {
                String name = e.getKey();
                Object value = e.getValue();
                buffer.append("\n [");
                buffer.append(name);
                buffer.append("] targets : '");
                buffer.append(toString(value));
                buffer.append("\' : ");
                buffer.append(((TagValueDefinition) value).documentation());
            }
        },
        stereotype {
            @Override
            public Map<String, ?> getData(AvailableDataMojo mojo) {
                return mojo.getCurrentStereotypeDefinitionProvider().getDefinition();
            }

            @Override
            String toString(Object data) {
                StereotypeDefinition d = (StereotypeDefinition) data;
                StringBuilder sb = new StringBuilder();
                Class<?>[] targets = d.target();
                for (Class<?> aClass : targets) {
                    sb.append(", ").append(aClass.getSimpleName());
                }
                String result = sb.toString();
                if (targets.length > 0) {
                    result = result.substring(2);
                }
                return result;
            }

            @Override
            void toString(StringBuilder buffer, Map.Entry<String, ?> e) {
                String name = e.getKey();
                Object value = e.getValue();
                buffer.append("\n [");
                buffer.append(name);
                buffer.append("] targets : '");
                buffer.append(toString(value));
                buffer.append("\' : ");
                buffer.append(((StereotypeDefinition) value).documentation());
            }
        };

        abstract Map<String, ?> getData(AvailableDataMojo mojo);

        String toString(Object data) {
            return data.getClass().getName();
        }

        void toString(StringBuilder buffer, Map.Entry<String, ?> e) {
            String name = e.getKey();
            Object value = e.getValue();
            buffer.append("\n [");
            buffer.append(name);
            buffer.append("] with implementation '");
            buffer.append(toString(value));
            buffer.append('\'');
        }
    }

    public TagValueDefinitionProvider getCurrentTagValueDefinitionProvider() {
        return currentTagValueDefinitionProvider;
    }

    public StereotypeDefinitionProvider getCurrentStereotypeDefinitionProvider() {
        return currentStereotypeDefinitionProvider;
    }

    protected TagValueDefinitionProvider getTagValueDefinitionProvider() throws MojoExecutionException {

        TagValueDefinitionProvider provider;

        if (tagValueDefinitionProviders == null || tagValueDefinitionProviders.isEmpty()) {

            // could not find any model properties via plexus
            // try to obtain them by ServiceLoader

            provider = TagValueDefinitionProviders.newProvider(getClass().getClassLoader(), false);
        } else {
            provider = TagValueDefinitionProviders.newProvider(tagValueDefinitionProviders.values(), false);
        }
        return provider;
    }


    protected StereotypeDefinitionProvider getStereotypeDefinitionProvider() throws MojoExecutionException {

        StereotypeDefinitionProvider provider;

        if (stereotypeDefinitionProviders == null || stereotypeDefinitionProviders.isEmpty()) {

            // could not find any model properties via plexus
            // try to obtain them by ServiceLoader

            provider = StereotypeDefinitionProviders.newProvider(getClass().getClassLoader(), false);
        } else {
            provider = StereotypeDefinitionProviders.newProvider(stereotypeDefinitionProviders.values(), false);
        }
        return provider;
    }

}
