/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.ui.utils;

import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.Writer;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.io.FileUtils;
import org.chorem.pollen.business.utils.MailUtil;
import org.chorem.pollen.ui.services.Configuration;
import org.h2.util.IOUtils;
import org.nuiton.util.FileUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import au.com.bytecode.opencsv.CSVReader;
import au.com.bytecode.opencsv.CSVWriter;

/**
 * Mass mail management class.
 * 
 * This class store email to send in a csv file, and send it (one per second) in
 * a second time.
 * This class can restart to send mail at application restart.
 * 
 * For a mass mail sending to start, following files must be present:
 * <ul>
 *  <li>xxx.emails : CSV file ("email", "subject", "body")</li>
 *  <li>xxx.index : next index to manage ( inited at 0)</li>
 * </ul>
 * 
 * @author chatellier
 * @version $Revision: 2927 $
 * 
 * Last update : $Date: 2010-03-10 19:08:53 +0100 (mer., 10 mars 2010) $
 * By : $Author: echatellier $
 */
public class SendMail extends Thread {

    /** logger. */
    private static final Logger log = LoggerFactory.getLogger(SendMail.class);

    public static final String EXTENSION_MAIL = ".mail";
    public static final String EXTENSION_INDEX = ".index";

    /** Pollen configuration. */
    protected Configuration configuration;

    /** Mail storage directory. */
    protected File mailStorageDirectory;

    public SendMail(Configuration configuration) {
        
        this.configuration = configuration;

        // get email directory in configuration
        // create it if not exists
        String filename = configuration.getProperty(Configuration.PROP_EMAIL_DIRECTORY);
        mailStorageDirectory = new File(filename);
        if (!mailStorageDirectory.exists()) {
            if (mailStorageDirectory.mkdirs()) {
                if (log.isDebugEnabled()) {
                    log.debug("Email storage directory created in : " + mailStorageDirectory.getAbsolutePath());
                }
            }
        }
    }

    /*
     * @see java.lang.Runnable#run()
     */
    @Override
    public void run() {

        // try to find existing files
        while (true) {
            try {
                sendAllMails();
                
                // bloque thread until next notify
                sleep();
            }
            catch (Exception ex) {
                if (log.isErrorEnabled()) {
                    log.error("Error during SendMail main loop", ex);
                }
            }
        }
    }

    protected synchronized void sleep() throws InterruptedException {
        wait();
    }

    public synchronized void wakeUp() {
        notifyAll();
    }

    /**
     * Look for all xx.index file, and restart mail sending on
     * non ending mass mail sending.
     * 
     * After execution, delete mail and index file.
     * @throws IOException 
     */
    protected void sendAllMails() throws IOException {

        // don't do for, allways take the first found
        // a new one can be created when managing one other
        List<File> indexFiles = null;
        do {

            // filter is java valid : .*\\.index
            indexFiles = FileUtil.find(mailStorageDirectory, ".*\\" + EXTENSION_INDEX, false);

            if (!indexFiles.isEmpty()) {
                File indexFile = indexFiles.get(0);
                // convert index content to int
                String indexContent = FileUtils.readFileToString(indexFile, "UTF-8");
                int index = Integer.parseInt(indexContent);

                // get mail content file
                File mailFile = new File(indexFile.getAbsolutePath().replaceAll(EXTENSION_INDEX + "$", EXTENSION_MAIL));

                if (log.isDebugEnabled()) {
                    log.debug("Managing mail file : " + mailFile + " (from index " + indexContent + ")");
                }

                Reader indexFileReader = new BufferedReader(new FileReader(mailFile));
                CSVReader cvsReader = new CSVReader(indexFileReader);

                int currentIndex = 0;
                String[] currentLine = cvsReader.readNext();
                while (currentLine != null) {
                    String receiver = currentLine[0];
                    String subject = currentLine[1];
                    String body = currentLine[2];
                    
                    log.debug("Props = " + configuration.getConf());

                    // index contains next index to treat so == is ok
                    if (currentIndex >= index) {
                        MailUtil.sendMail(configuration.getProperty("email_host"),
                                Integer.parseInt(configuration.getProperty("email_port")),
                                configuration.getProperty("email_from"),
                                receiver, subject, body);
                        
                        // index contains next index to treat
                        FileUtils.writeStringToFile(indexFile, String.valueOf(currentIndex + 1));
                        
                        // wait 2 secondes between each mail to not
                        // load smtp server
                        try {
                            Thread.sleep(1000);
                        } catch (InterruptedException ex) {
                            if (log.isErrorEnabled()) { 
                                log.error("Can't wait between mail", ex);
                            }
                        }
                    }
                    else {
                        if (log.isDebugEnabled()) {
                            log.debug("Mail to " + receiver + " already sent in a previous execution, skip.");
                        }
                    }

                    currentIndex++;
                    currentLine = cvsReader.readNext();
                }

                // delete woth index and mail file
                mailFile.delete();
                indexFile.delete();
            }
            else {
                if (log.isInfoEnabled()) {
                    log.info("No more index mail index file found, go to sleep a while :)");
                }
            }
        } while (!indexFiles.isEmpty());
    }

    /**
     * Prepare mail list.
     * 
     * TODO : improve configuration reading
     * 
     * @param id
     * @param mailData
     * @throws IOException 
     */
    public void prepareMails(String id, List<Map<String, String>> mailData) throws IOException {

        Writer fileWriter = null;
        CSVWriter cvsWriter = null;
        try {
            
            // write CSV datas
            File emailFile = new File(mailStorageDirectory, id + EXTENSION_MAIL);
            fileWriter = new OutputStreamWriter(new BufferedOutputStream(new FileOutputStream(emailFile)));
            cvsWriter = new CSVWriter(fileWriter);
            
            for (Map<String, String> singleMailData : mailData) {
                String[] nextLine = new String[] {
                        singleMailData.get("receiver"),
                        singleMailData.get("subject"),
                        singleMailData.get("body")
                };
                cvsWriter.writeNext(nextLine);
            }
            
            // write index (default to 0)
            File indexFile = new File(mailStorageDirectory, id + EXTENSION_INDEX);
            FileUtil.writeString(indexFile, "0");
        }
        finally {
            if (cvsWriter != null) {
                cvsWriter.close();
            }
            IOUtils.closeSilently(fileWriter);
        }
        
    }
}
