package org.chorem.pollen.ui.components;


import org.apache.tapestry5.BindingConstants;
import org.apache.tapestry5.ComponentResources;
import org.apache.tapestry5.EventConstants;
import org.apache.tapestry5.Link;
import org.apache.tapestry5.MarkupWriter;
import org.apache.tapestry5.annotations.Log;
import org.apache.tapestry5.annotations.Parameter;
import org.apache.tapestry5.ioc.Messages;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.slf4j.Logger;


/**
 * Pager
 *
 * Created: 8 mars 2010
 *
 * @author fdesbois
 * @version $Revision: 2923 $
 *
 * Mise a jour: $Date: 2010-03-10 10:39:32 +0100 (mer., 10 mars 2010) $
 * par : $Author$
 */
public class Pager {

    /**
     * Number of total rows.
     */
    @Parameter(required = true)
    private int nbTotalRows;

    /**
     * Main element of the pager, the page which contains the pager have
     * to manage this property (using @Persist for exemple) to ensure that
     * the pager will done its work correctly. You can use
     * {@link #getFirstPage() } to initialize the first page.
     */
    @Parameter(required = true)
    private int currentPage;

    /**
     * Number of rows to show per page.
     */
    @Parameter(required = true)
    private int nbRowsPerPage;

    @Parameter(defaultPrefix = BindingConstants.LITERAL)
    private String noPagerText;

    @Inject
    private ComponentResources resources;

    @Inject
    private Messages messages;

    @Inject
    private Logger log;

    void beginRender(MarkupWriter writer) {
        if (log.isDebugEnabled()) {
            log.debug("pager from : " + getFirstValue() + " to " + getLastValue());
            log.debug("current page : " + currentPage);
            log.debug("max nb rows : " + nbTotalRows);
            log.debug("has previous : " + hasPrevious());
            log.debug("has next : " + hasNext());
            log.debug("indexes from : " + getStartIndex() + " to " + getEndIndex());
            log.debug("last page : " + getLastPage());
        }
        writer.element("div", "class", "p-pager");
        // If firstPage and lastPage are the same, no need to use the pager
        if (getFirstPage() == getLastPage()) {
            writer.write(noPagerText);
        } else {
            if (hasPrevious()) {
                int previousPage = currentPage - 1;
                if (previousPage != getFirstPage()) {
                    writeLink(writer, getFirstPage(), "<<", "p-pager-first");
                    writer.write(" ");
                }
                writeLink(writer, previousPage, "<", "p-pager-previous");
                writer.write(" ");
            }
            // Text display for current page
            writer.element("span", "title",
                    messages.format("rows-title",
                            getFirstValue(), getLastValue(), nbTotalRows));
            writer.element("strong");
            writer.write(messages.format("page", currentPage));
            writer.end();
            writer.end();

            if (hasNext()) {
                int nextPage = currentPage + 1;
                writer.write(" ");
                writeLink(writer, nextPage, ">", "p-pager-next");
                if (nextPage != getLastPage()) {
                    writer.write(" ");
                    writeLink(writer, getLastPage(), ">>", "p-pager-last");
                }
            }
        }
        writer.end();
    }

    void afterRender(MarkupWriter writer) {
        
    }

    /**
     * Use an action event to change the current page. The link refresh the
     * entire page, no zone is managed yet.
     *
     * @param writer used to write a element
     * @param index new page to change (argument of onAction method)
     * @param text to write inside the a tag
     * @param style of the css (class) for the a tag
     */
    protected void writeLink(MarkupWriter writer, int index,
            String text, String style) {
        Link link = resources.createEventLink(EventConstants.ACTION,
                new Object[] {index});
        String title = messages.format("go-to", index);
        writer.element("a", "href", link, "class", style, "title", title);
        writer.write(text);
        writer.end();
    }

    /**
     * Get the start index of the range shown.
     *
     * @return the start index for elements to show
     */
    public int getStartIndex() {
        return ( (currentPage - 1) * nbRowsPerPage);
    }

    /**
     * Get the end index of the range shown.
     *
     * @return the end index for elements to show
     */
    public int getEndIndex() {
        int end = (currentPage * nbRowsPerPage) - 1;
        if (end >= nbTotalRows) {
            end = nbTotalRows - 1;
        }
        return end;
    }

    protected int getFirstValue() {
        return getStartIndex() + 1;
    }

    protected int getLastValue() {
        return getEndIndex() + 1;
    }

    /**
     * Test if there is previous elements before this actual range.
     *
     * @return true if there is previous elements, false otherwise
     */
    public boolean hasPrevious() {
        return getStartIndex() > 0;
    }

    /**
     * Test if there is next elements after this actual range.
     *
     * @return true if there is next elements, false otherwise
     */
    public boolean hasNext() {
        return getEndIndex() < (nbTotalRows - 1);
    }

    public int getFirstPage() {
        return 1;
    }

    public int getLastPage() {
        return (int)Math.ceil((double)nbTotalRows / (double)nbRowsPerPage);
    }

    /**
     * Action event for changing page. The currentPage is changed to
     * {@code newPage}.
     *
     * @param newPage to set
     */
    void onAction(int newPage) {
        currentPage = newPage;
    }

}
