/**
 * *##% Callao PeriodDTO
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*
 */

package org.chorem.callao.service.dto;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import org.chorem.callao.service.utils.DateUtil;

/**
 * Classe DTO - représente un objet Period.
 *
 * @author Rémi Chapelet
 */
public class PeriodDTO  {

    private String id; // Identifiant Topia

    private Date beginPeriod;

    private Date endPeriod;

    private boolean locked;

    private List<TimeSpanDTO> listTimeSpan;

    private static DateUtil dateUtil = new DateUtil();

    public PeriodDTO ()
    {
        this.id = "";
        this.beginPeriod = new Date();
        this.beginPeriod = dateUtil.InitDateFirstDayMonth(this.beginPeriod);
        this.endPeriod = new Date();
        this.endPeriod = dateUtil.InitDateEndDayMonth(this.endPeriod);
        this.locked = false;
        this.listTimeSpan = new ArrayList<TimeSpanDTO>();
    }

    public PeriodDTO (String id, Date beginPeriod, Date endPeriod, boolean locked, List<TimeSpanDTO> listTimeSpan)
    {
        this.id = id;
        this.beginPeriod = dateUtil.InitDateFirstDayMonth(beginPeriod);
        this.endPeriod = dateUtil.InitDateEndDayMonth(endPeriod);
        this.locked = locked;
        // Si la liste est vide, il faut l'initialiser
        if (listTimeSpan == null )
        {
            this.listTimeSpan = new ArrayList<TimeSpanDTO>();
        } else
        {
            // Vérification de chaque timeSpan
            for (TimeSpanDTO timeSpanDTO : this.listTimeSpan)
            {
                // Ajout du timeSpan avec vérification
                addTimeSpan(timeSpanDTO);
            }
        }
    }

    /**
     * Permet de vérifier pour chaque timeSpan. Il doit en avoir 12. Chaque date
     * doit être comprise dans la période.
     * @param listTimeSpan
     * @return
     */
    public boolean isCorrectTimeSpan (TimeSpanDTO timeSpanDTO)
    {
        boolean result = false;
        if (timeSpanDTO != null )
        {
            // Si le timeSpan n'existe pas déjà
            if (!existTimeSpan(timeSpanDTO))
            {
                // Vérification du nombre de timeSpan appartenant à la période (12 timeSpans)
                if (getListTimeSpan().size() < 12)
                {                    
                    // Si la date de début du timeSpan est avant/après la période
                    if ( dateUtil.betweenDate(timeSpanDTO.getBeginTimeSpan(),this.beginPeriod,this.endPeriod)
                            && dateUtil.betweenDate(timeSpanDTO.getEndTimeSpan(),this.beginPeriod,this.endPeriod) )
                    {
                        result = true;
                    }                    
                }
            }
        }
        return result;
    }

    /**
     * Ajoute un timeSpan dans la liste.
     * @param timeSpanDTO
     */
    public void addTimeSpan (TimeSpanDTO timeSpanDTO)
    {
        // Si le timeSpan est correct
        if (isCorrectTimeSpan(timeSpanDTO))
        {
            this.getListTimeSpan().add(timeSpanDTO);
            // Tri la liste dans l'ordre croissant des timeSpans
            Collections.sort(getListTimeSpan());
        }
    }

    /**
     * Vérifie si le timeSpan n'existe pas déja dans la base de données.
     * @param timeSpanDTO
     * @return
     */
    public boolean existTimeSpan (TimeSpanDTO timeSpanDTO)
    {
        boolean exist = false;
        for (TimeSpanDTO timeSpanDtoOfList : this.getListTimeSpan())
        {
            // Si l'identifiant est non vide
            if ( !(timeSpanDTO.getId().equals("")) )
            {
                // Compare sur l'identifiant.
                if (timeSpanDtoOfList.getId().equals(timeSpanDTO.getId()))
                {
                    exist = true;
                }
            }
            /**
             * Compare sur les dates (il est possible que le DTO ne possède pas
             * encore son identifiant topiaId.
             */
            // Si la date de début du timeSpan dans la liste est égal au timeSpan donné en paramètre
            if (timeSpanDtoOfList.getBeginTimeSpan().compareTo(timeSpanDTO.getBeginTimeSpan()) == 0)
            {
                exist = true;
            }
        }
        return exist;
    }

    /**
     * Permet de bloquer une période.
     * ATTENTION : tous les timeSpans, qui composent la période, doivent être
     * bloqués. Une période peut être bloquée, MAIS pas être débloquée.
     */
    public void blockPeriod ()
    {
        // Vérifie si la période n'est pas déjà bloquée.
        if ( !isLocked() )
        {
            // Test pour les timeSpans si ils sont bien bloqués
            boolean existTimeSpanDTONotBlocked = false;
            for (TimeSpanDTO timeSpanDTO : this.listTimeSpan)
            {
                // Si le timeSpan est bloqué
                if (timeSpanDTO.isLocked())
                {
                    existTimeSpanDTONotBlocked = true;
                }
            }
            // Si tous les timeSpans sont bloqués, on bloque la période
            if (existTimeSpanDTONotBlocked)
            {
                this.locked = true;
            }
        }
    }

    /**
     * Permet de bloquer tous les timeSpans.
     */
    public void blockAllTimeSpan ()
    {
        for (TimeSpanDTO timeSpanDTO : this.listTimeSpan)
        {
            timeSpanDTO.setLocked(true);
        }
    }


    /**
     * @return the id
     */
    public String getId() {
        return id;
    }

    /**
     * @param id the id to set
     */
    public void setId(String id) {
        this.id = id;
    }

    /**
     * @return the beginPeriod
     */
    public Date getBeginPeriod() {
        return beginPeriod;
    }

    /**
     * @param beginPeriod the beginPeriod to set
     */
    public void setBeginPeriod(Date beginPeriod) {
        this.beginPeriod = dateUtil.InitDateFirstDayMonth(beginPeriod);
    }

    /**
     * @return the endPeriod
     */
    public Date getEndPeriod() {
        return endPeriod;
    }

    /**
     * @param endPeriod the endPeriod to set
     */
    public void setEndPeriod(Date endPeriod) {
        this.endPeriod = dateUtil.InitDateEndDayMonth(endPeriod);
    }

    /**
     * @return the locked
     */
    public boolean isLocked() {
        return locked;
    }

    /**
     * @param locked the locked to set
     */
    public void setLocked(boolean locked) {
        blockPeriod();
    }

    /**
     * @return the listTimeSpan
     */
    public List<TimeSpanDTO> getListTimeSpan() {
        return listTimeSpan;
    }

    /**
     * @param listTimeSpan the listTimeSpan to set
     */
    public void setListTimeSpan(List<TimeSpanDTO> listTimeSpan) {
        this.listTimeSpan = listTimeSpan;
    }

}