/**
 * *##% Callao TimeSpanServiceImpl
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*
 */

package org.chorem.callao.service;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.commons.logging.LogFactory;
import org.chorem.callao.entity.CallaoDAOHelper;
import org.chorem.callao.entity.Period;
import org.chorem.callao.entity.TimeSpan;
import org.chorem.callao.entity.TimeSpanDAO;
import org.chorem.callao.entity.Transaction;
import org.chorem.callao.service.dto.TimeSpanDTO;
import org.chorem.callao.service.convertObject.ConvertTimeSpan;
import org.chorem.callao.service.utils.ContextCallao;
import org.chorem.callao.service.utils.DateUtil;
import org.chorem.callao.service.utils.ServiceHelper;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;

/**
 * Gestion des périodes intermédiaires durant l'exercice.
 * Chaque timeSpan sera fixe, et devra correspondre à un mois complet.
 *
 * @author Rémi Chapelet
 */
public class TimeSpanServiceImpl { //implements TimeSpanService {

    /** log */
    private static final org.apache.commons.logging.Log log = LogFactory.getLog(TimeSpanServiceImpl.class);

    private TopiaContext rootContext = ContextCallao.getInstance().getContext();

    private ConvertTimeSpan convertTimeSpan = new ConvertTimeSpan();

    private static TransactionServiceImpl transactionServiceImpl = new TransactionServiceImpl();

    /**
     * Création d'une période timeSpan mensuelle avec une date de début et de fin. Une période
     * peut être bloquée ou non.
     * @param beginTimeSpan date de début de période
     * @param endTimeSpan date de fin de période
     * @param locked est à vrai si la période doit être bloquée.
     * @return
     */
    //@Override
    public String createTimeSpan(Date beginTimeSpan, Date endTimeSpan,Period period, boolean locked) {
        /**
         * timeSpan correspond à une période mensuelle. 
         * La fonction va donc prendre seulement pour date de référence la 
         * date beginTimeSpan. Elle calcule automatiquement la date de fin
         * suivant le mois.
         */
        DateUtil dateutil = new DateUtil();
        // Initialise la date de début au premier du mois
        beginTimeSpan = dateutil.InitDateFirstDayMonth(beginTimeSpan);
        // Initialise la date de fin au dernier du mois
        endTimeSpan = dateutil.InitDateEndDayMonth(beginTimeSpan);
        if (log.isInfoEnabled()) {
                log.info("Nouveau timeSpan du " + beginTimeSpan.toString() +
                        " au " + endTimeSpan.toString());
        }
	try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            TimeSpanDAO timeSpanDAO = CallaoDAOHelper.getTimeSpanDAO(topiaContext);
            // Creation du timespan
            TimeSpan newtimeSpan = timeSpanDAO.create();
            newtimeSpan.setBeginTimeSpan(beginTimeSpan);
            newtimeSpan.setEndTimeSpan(endTimeSpan);
            newtimeSpan.setLocked(locked);
            newtimeSpan.setPeriod(period);
            // Création BDD
            topiaContext.commitTransaction();
            // Fermeture BDD
            topiaContext.closeContext();
            if (log.isInfoEnabled()) {
                log.info("Ajout avec succes du timeSpan");
            }
            return ServiceHelper.RESPOND_SUCCESS;
        }catch (TopiaException e) {
			log.error(e);
            return ServiceHelper.RESPOND_ERROR;
		}
	}

    /**
     * Permet de trouver un timespan directement avec une date. La date peut
     * être quelconque. Exemple : d = 17 sept 2000, renvoie la période du
     * 1 sept au 30 sept 2000.
     * @param d
     * @return
     */
    public TimeSpan searchTimeSpanByDate (Date d)
    {
        TimeSpan timeSpan = null;
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            TimeSpanDAO timeSpanDAO = CallaoDAOHelper.getTimeSpanDAO(topiaContext);
            // Par précaution, on initialise la date au debut du mois
            DateUtil dateutil = new DateUtil();
            // Recherche du timeSpan
            Date searchDate = dateutil.InitDateFirstDayMonth(d);
            timeSpan = timeSpanDAO.findByBeginTimeSpan(searchDate);
            // Fermeture BDD
            topiaContext.closeContext();
        }catch (TopiaException e) {
            log.error(e);
        }
        return timeSpan;
    }


    /**
     * Recherche un timeSpan précis, avec sa date de création.
     * @param timeSpanDTO timeSpan au format DTO qu'on recherche
     * @return
     */
    public TimeSpan searchTimeSpanByDate (TimeSpanDTO timeSpanDTO)
    {
        TimeSpan timeSpan = searchTimeSpanByDate(timeSpanDTO.getBeginTimeSpan());
        return timeSpan;
    }


    /**
     * Recherche un timeSpan UNIQUE. Celui est identifié par son topiaId.
     * @param topiaId Identifiant du timeSpan recherché
     * @return
     */
    public TimeSpan searchTimeSpanWithTopiaId (String topiaId)
    {
        TimeSpan timeSpan = null;
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            TimeSpanDAO timeSpanDAO = CallaoDAOHelper.getTimeSpanDAO(topiaContext);
            // Recherche du timeSpan
            timeSpan = timeSpanDAO.findByTopiaId(topiaId);
            // Fermeture BDD
            topiaContext.closeContext();
        }catch (TopiaException e) {
            log.error(e);
        }
        return timeSpan;
    }


    /**
     * Permet de rechercher un timeSpan à l'aide d'une date. Il fait appel
     * à la méthode searchTimeSpanByDate.
     * La méthode renvoie un objet DTO.
     * @param d
     * @return
     */
    public TimeSpanDTO searchTimeSpanDTOByDate (Date d)
    {
        TimeSpan timeSpan = searchTimeSpanByDate(d);
        TimeSpanDTO timeSpanDTO = convertTimeSpan.timeSpanEntityToDto(timeSpan);
        return timeSpanDTO;
    }

    /**
     * Permet de faire une recherche de tous les timeSpans d'une période.
     * @param period
     * @return
     */
    public List<TimeSpan> searchListTimeSpan (Period period)
    {
        List<TimeSpan> listTimeSpan = null;
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            TimeSpanDAO timeSpanDAO = CallaoDAOHelper.getTimeSpanDAO(topiaContext);
            // Recherche les timeSpans avec cette période
            listTimeSpan = timeSpanDAO.findAllByPeriod(period);
            // Fermeture BDD
            topiaContext.closeContext();
        }catch (TopiaException e) {
            log.error(e);
        }
        return listTimeSpan;
    }


    /**
     * Permet de rechercher tous les timeSpans suivant une période donnée.
     * Chaque timeSpan est converti ensuite en DTO.
     * @param period
     * @return
     */
    public List<TimeSpanDTO> searchListTimeSpanDTO (Period period)
    {
        List<TimeSpanDTO> listTimeSpanDTO = new ArrayList<TimeSpanDTO>();
        List<TimeSpan> listTimeSpan = searchListTimeSpan(period);
        for (TimeSpan timeSpan : listTimeSpan)
        {
            TimeSpanDTO timeSpanDTO = convertTimeSpan.timeSpanEntityToDto(timeSpan);
            listTimeSpanDTO.add(timeSpanDTO);
        }
        return listTimeSpanDTO;
    }


    /**
     * Permet de bloquer un timespan, celle-ci ne doit pas être bloquée.
     * La fonction doit s'assurer que les timeSpans précédents doivent être
     * bloqués.
     * Les transactions doivent être équilibrées
     * @param timespan période qui doit être fermée
     * @param period période (12 smois)
     */
    public String blockTimeSpan (TimeSpan timespan)
    {
        String result = ServiceHelper.RESPOND_ERROR;
        // Si la période n'est pas déjà fermée.
        if (timespan.getLocked())
        {
            if (log.isInfoEnabled()) {
                log.info("Timespan deja bloqué !");
            }
        } else {
            Period period = getPeriodWithTimeSpan(timespan);
            if ( period.getLocked() )
            {
                if (log.isErrorEnabled()) {
                        log.error("La période du timespan est bloquée !");
                }
            } else {
                // Vérifie si tous les timeSpans précédents sont bloqués
                boolean AllTimeSpanBlock = alltimeSpanBlocked(timespan,period);
                if ( AllTimeSpanBlock )
                {
                    // Enregistrement dans la base de données
                    try {
                        // Acces BDD
                        TopiaContext topiaContext = rootContext.beginTransaction();
                        // Chargement du DAO
                        TimeSpanDAO timeSpanDAO = CallaoDAOHelper.getTimeSpanDAO(topiaContext);
                        /**
                         * Vérifie si les transactons sont équilibrées
                         */
                        List<Transaction> listTransaction = transactionServiceImpl.searchListTransactionWithTimeSpan(timespan);
                        boolean isTransactionBalanced = true;
                        boolean existTransactionNotBalanced = false;
                        for (Transaction transaction : listTransaction )
                        {
                            isTransactionBalanced = transactionServiceImpl.isTransactionBalanced(transaction);
                            if ( !isTransactionBalanced )
                            {
                                if (log.isInfoEnabled()) {
                                    log.info("La transaction n'est pas bloquée.");
                                }
                                existTransactionNotBalanced = true;
                            }
                        }
                        if ( !existTransactionNotBalanced )
                        {
                            // Bloque la période
                            timespan.setLocked(true);
                            // Mise à jour de timespan dans la BDD
                            timeSpanDAO.update(timespan);
                            // Création BDD
                            topiaContext.commitTransaction();
                            if (log.isInfoEnabled()) {
                                log.info("Timespan bloqué avec succès");
                            }
                            result = ServiceHelper.RESPOND_SUCCESS;
                        } else {
                            result = ServiceHelper.TRANSACTION_NOT_BALANCED;
                        }
                        // Fermeture BDD
                        topiaContext.closeContext();
                    }catch (TopiaException e) {
                        log.error(e);
                    }
                } else {
                    if (log.isErrorEnabled()) {
                        log.error("Il existe un ou plusieurs timespans précédents" +
                                " non bloqués.");
                    }
                    result = ServiceHelper.TIMESPAN_PREC_NOT_BLOCK;
                }
            }
        }
        return result;
    }


    /**
     * Permet de bloquer une période au format DTO.
     * 
     * @param timespanDTO
     * @return
     */
    public String blockTimeSpan (TimeSpanDTO timespanDTO)
    {
        return blockTimeSpan(searchTimeSpanByDate(timespanDTO));
    }


    /**
     * Permet de débloquer un timespan, à condition que les timespans suivants
     * celui-ci soient bien débloqués.
     * @param timespan
     * @param period
     * @return
     */
    public String unblockTimeSpan (TimeSpan timespan)
    {
        String result = ServiceHelper.RESPOND_ERROR;
        // Si le timspan n'est pas bloqué
        if (!timespan.getLocked())
        {
            if (log.isWarnEnabled()) {
                log.warn("Timespan non bloqué !");
            }
        }else{
            Period period = getPeriodWithTimeSpan(timespan);
            if ( period.getLocked() )
            {
                if (log.isErrorEnabled()) {
                        log.error("La période du timespan est bloquée !");
                }
            } else {
                boolean NextTimeSpanUnblock = nextTimeSpanUnblock(timespan, period);
                // Si aucun timespan suivant bloqué
                if ( NextTimeSpanUnblock )
                {
                    try {
                        // Acces BDD
                        TopiaContext topiaContext = rootContext.beginTransaction();
                        // Chargement du DAO
                        TimeSpanDAO timeSpanDAO = CallaoDAOHelper.getTimeSpanDAO(topiaContext);
                        // débloque la période
                        timespan.setLocked(false);
                        // Mise à jour de timespan dans la BDD
                        timeSpanDAO.update(timespan);
                        // Création BDD
                        topiaContext.commitTransaction();
                        if (log.isInfoEnabled()) {
                            log.info("Timespan débloqué avec succès");
                        }
                        result = ServiceHelper.RESPOND_SUCCESS;
                        // Fermeture BDD
                        topiaContext.closeContext();
                    }catch (TopiaException e) {
                        log.error(e);
                    }
                } else {
                    result = ServiceHelper.TIMESPAN_NEXT_NOT_BLOCK;
                }
            }
        }
        return result;
    }

    /**
     * Permet de bloquer une période au format DTO.
     *
     * @param timespanDTO
     * @return
     */
    public String unblockTimeSpan (TimeSpanDTO timespanDTO)
    {
        return unblockTimeSpan(searchTimeSpanByDate(timespanDTO));
    }


    /**
     * Cette méthode va rechercher tous les timeSpans non bloqués après
     * le timeSpan renseigné en paramètre.
     * Si elle trouve au moins un timeSpan NON bloqué, elle renvoie alors faux.
     * Cette méthode permet essentiellement de définir si il est possible
     * ou non de bloquer un timeSpan. A savoir, un timeSpan ne peut être bloqué,
     * si il existe des timeSpans suivant (dans le calendrier) qui sont non
     * bloqués.
     * @param timespan
     * @param period
     * @return
     */
    private boolean nextTimeSpanUnblock (TimeSpan timespan, Period period)
    {
        boolean NextTimeSpanUnblock = false;
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            TimeSpanDAO timeSpanDAO = CallaoDAOHelper.getTimeSpanDAO(topiaContext);
            // Recherche des TimeSpans avec la même période.
            // (on doit récupérer normalement 12 timeSpans)
            List<TimeSpan> listeTimeSpanOfPeriod = timeSpanDAO.findAllByPeriod(period);
            NextTimeSpanUnblock = true;
            for (TimeSpan timeSpanNext : listeTimeSpanOfPeriod)
            {
                // Si le timespan est après celui qu'on souhaite bloquer ET est non bloqué
                if (timeSpanNext.getBeginTimeSpan().compareTo(timespan.getBeginTimeSpan()) == 1 &&
                        timeSpanNext.getLocked() )
                {
                    NextTimeSpanUnblock = false;
                    if (log.isInfoEnabled()) {
                        log.info("Timespan "+timeSpanNext.toString()+" est bloqué");
                    }
                }
            }
            // Fermeture BDD
            topiaContext.closeContext();
        }catch (TopiaException e) {
            log.error(e);
        }
        return NextTimeSpanUnblock;
    }

    /**
     * Permet de renvoyer une période dont appartient le timeSpan.
     * @param timeSpan timeSpan dont on souhaite retourner la période
     * @return
     */
    private Period getPeriodWithTimeSpan (TimeSpan timeSpan)
    {
        PeriodServiceImpl periodServiceImpl = new PeriodServiceImpl();
        Date beginTimeSpan = timeSpan.getBeginTimeSpan();
        Period period = periodServiceImpl.searchPeriodWithDate(beginTimeSpan);
        return period;
    }


    /**
     * Cette méthode renvoie vrai lorsque tous les timespans de la période
     * précédent au timespan en paramètre sont bloqués.
     * @param timespan
     * @param period le paramètre period est important, en effet un timespan ne
     * peut être indépendant.
     * @return
     */
    private boolean alltimeSpanBlocked (TimeSpan timespan, Period period)
    {
        boolean AllTimeSpanBlock = false;
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            TimeSpanDAO timeSpanDAO = CallaoDAOHelper.getTimeSpanDAO(topiaContext);
            // Recherche des TimeSpans avec la même période.
            // (on doit récupérer normalement 12 timeSpans)
            List<TimeSpan> listeTimeSpanOfPeriod = timeSpanDAO.findAllByPeriod(period);
            // Fermeture BDD
            topiaContext.closeContext();
            // On contrôle que les timeSpans précédents sont bloqués.
            AllTimeSpanBlock = true;
            for (TimeSpan timeSpanPrev : listeTimeSpanOfPeriod)
            {
                // Si la période est avant celle qu'on souhaite bloquer ET est non bloquée
                if (timeSpanPrev.getBeginTimeSpan().compareTo(timespan.getBeginTimeSpan()) == -1 &&
                        !timeSpanPrev.getLocked() )
                {
                    AllTimeSpanBlock = false;
                    if (log.isInfoEnabled()) {
                        log.info("Timespan "+timeSpanPrev.toString()+" non bloqué");
                    }
                }
            }
        }catch (TopiaException e) {
            log.error(e);
        }
        return AllTimeSpanBlock;
    }

    /**
     * Permet d'effacer un timeSpan. Ce dernier ne doit pas être bloqué.
     * @param timeSpan
     * @return
     */
    public String removeTimeSpan (TimeSpan timeSpan)
    {
        String result = ServiceHelper.RESPOND_ERROR;
        // Si le timspan est bloqué
        if (timeSpan.getLocked())
        {
            if (log.isWarnEnabled()) {
                log.warn("Timespan bloqué !");
            }
        }else{            
            try {
                // Acces BDD
                TopiaContext topiaContext = rootContext.beginTransaction();
                // Chargement du DAO
                TimeSpanDAO timeSpanDAO = CallaoDAOHelper.getTimeSpanDAO(topiaContext);                
                // Mise à jour de timespan dans la BDD
                timeSpanDAO.delete(timeSpan);
                // Création BDD
                topiaContext.commitTransaction();
                if (log.isInfoEnabled()) {
                    log.info("Timespan supprimé avec succès");
                }
                result = ServiceHelper.RESPOND_SUCCESS;
                // Fermeture BDD
                topiaContext.closeContext();
            }catch (TopiaException e) {
                log.error(e);
            }
        }
        return result;
    }

    /**
     * Vérifie si le timeSpan donné en paramètre est bloqué ou non.
     * @param timeSpan
     * @return
     */
    public boolean isTimeSpanBlocked (TimeSpan timeSpan)
    {
        boolean isTimeSpanBlocked = true;
         try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            TimeSpanDAO timeSpanDAO = CallaoDAOHelper.getTimeSpanDAO(topiaContext);
            // Mise à jour de timespan dans la BDD
            TimeSpan timeSpanSearch = timeSpanDAO.findByTopiaId(timeSpan.getTopiaId());
            isTimeSpanBlocked = timeSpanSearch.getLocked();
            // Fermeture BDD
            topiaContext.closeContext();
        }catch (TopiaException e) {
            log.error(e);
        }
        return isTimeSpanBlocked;
    }



	//@Override
	public String[] getMethods() {
		// TODO Auto-generated method stub
		return null;
	}

	//@Override
	public void destroy() {
		// TODO Auto-generated method stub

	}

	//@Override
	public void init(TopiaContext arg0) {
		// TODO Auto-generated method stub

	}

}
