package fr.ifremer.tutti.ui.swing.action;

/*
 * #%L
 * Tutti :: UI
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import fr.ifremer.tutti.persistence.ProgressionModel;
import fr.ifremer.tutti.persistence.entities.data.Cruise;
import fr.ifremer.tutti.service.export.sumatra.CatchesSumatraExportService;
import fr.ifremer.tutti.service.export.sumatra.SumatraExportResult;
import fr.ifremer.tutti.ui.swing.content.MainUIHandler;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.DateUtil;

import java.io.File;
import java.util.Date;

import static org.nuiton.i18n.I18n.t;

/**
 * Opens a file chooser, exports the cruise catches into the selected file and open the default email editor.
 *
 * @author kmorin <morin@codelutin.com>
 * @since 1.0
 */
public class ExportCruiseForSumatraAction extends AbstractMainUITuttiAction {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(ExportCruiseForSumatraAction.class);

    protected File file;

    protected SumatraExportResult sumatraExportResult;

    public ExportCruiseForSumatraAction(MainUIHandler handler) {
        super(handler, false);
    }

    @Override
    public boolean prepareAction() throws Exception {

        boolean doAction = super.prepareAction();

        if (doAction) {

            if (!getDataContext().isProtocolFilled()) {
                displayWarningMessage(
                        t("tutti.exportCruiseForSumatra.title.missing.protocol"),
                        t("tutti.exportCruiseForSumatra.message.missing.protocol")
                );
            }
        }

        if (doAction) {

            String date = DateUtil.formatDate(new Date(), "dd-MM-yyyy");
            // choose file to export
            file = saveFile(
                    String.format("sumatra_%s_%s", getDataContext().getCruise().getName(), date),
                    "csv",
                    t("tutti.exportCruiseForSumatra.title.choose.exportFile"),
                    t("tutti.exportCruiseForSumatra.action.chooseFile"),
                    "^.+\\.csv$", t("tutti.common.file.csv")
            );
            doAction = file != null;
        }
        return doAction;
    }

    @Override
    public void releaseAction() {
        file = null;
        sumatraExportResult = null;
        super.releaseAction();
    }

    @Override
    public void doAction() throws Exception {
        Cruise cruise = getDataContext().getCruise();
        Preconditions.checkNotNull(cruise);
        Preconditions.checkNotNull(file);

        if (log.isInfoEnabled()) {
            log.info("Will export cruise " + cruise.getId() +
                     " to file: " + file);
        }
        ProgressionModel pm = new ProgressionModel();
        pm.setTotal(3); // loading cruise + loading fishing operationIds + export
        setProgressionModel(pm);

        // export catches
        CatchesSumatraExportService service =
                getContext().getCatchesSumatraExportService();
        sumatraExportResult = service.exportCruiseForSumatra(file, cruise.getId(), pm);

    }

    @Override
    public void postSuccessAction() {
        super.postSuccessAction();

        if (sumatraExportResult.withBadSpecies()) {

            StringBuilder badSpeciesList = new StringBuilder();
            for (String s : sumatraExportResult.getBadSpecies()) {
                badSpeciesList.append("<li>").append(s);
            }

            displayWarningMessage(
                    t("tutti.exportFishingOperationForSumatra.title.badSpecies"),
                    t("tutti.exportFishingOperationForSumatra.message.badSpecies", badSpeciesList.toString()));
        }

        if (sumatraExportResult.withBadBenthos()) {

            StringBuilder badBenthosList = new StringBuilder();
            for (String s : sumatraExportResult.getBadBenthos()) {
                badBenthosList.append("<li>").append(s);
            }

            displayWarningMessage(
                    t("tutti.exportFishingOperationForSumatra.title.badBenthos"),
                    t("tutti.exportFishingOperationForSumatra.message.badBenthos", badBenthosList.toString()));
        }

        sendMessage(t("tutti.exportCruiseForSumatra.action.success", file));
    }
}
