package fr.ifremer.tutti.ui.swing.action;

/*
 * #%L
 * Tutti :: UI
 * $Id: RemoveBenthosBatchAction.java 1566 2014-02-04 08:31:02Z tchemit $
 * $HeadURL: https://svn.codelutin.com/tutti/tags/tutti-3.3/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/action/RemoveBenthosBatchAction.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import fr.ifremer.tutti.persistence.TuttiPersistence;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.BenthosBatchRowModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.BenthosBatchTableModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.BenthosBatchUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.BenthosBatchUIHandler;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.BenthosBatchUIModel;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;
import jaxx.runtime.SwingUtil;
import org.jdesktop.swingx.JXTable;

import javax.swing.JOptionPane;
import java.util.Set;

import static org.nuiton.i18n.I18n.t;

/**
 * To remove a species batch and all his children.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.0
 */
public class RemoveBenthosBatchAction extends AbstractTuttiAction<BenthosBatchUIModel, BenthosBatchUI, BenthosBatchUIHandler> {

    /**
     * Selected row to treat.
     *
     * @since 2.8
     */
    protected int rowIndex;

    /**
     * Rows to delete after all.
     *
     * @since 3.0-rc-1
     */
    protected Set<BenthosBatchRowModel> rowToRemove;

    public RemoveBenthosBatchAction(BenthosBatchUIHandler handler) {
        super(handler, false);
    }

    @Override
    public boolean prepareAction() throws Exception {
        boolean result = super.prepareAction();

        rowIndex = -1;

        if (result) {
            int answer = JOptionPane.showConfirmDialog(getContext().getActionUI(),
                                                       t("tutti.editBenthosBatch.action.removeBatch.confirm.message"),
                                                       t("tutti.editBenthosBatch.action.removeBatch.confirm.title"),
                                                       JOptionPane.YES_NO_OPTION);
            result = answer == JOptionPane.YES_OPTION;
        }

        return result;
    }

    @Override
    public void doAction() throws Exception {

        TuttiPersistence persistenceService =
                getContext().getPersistenceService();

        JXTable table = handler.getTable();

        rowIndex = SwingUtil.getSelectedModelRow(table);

        Preconditions.checkState(rowIndex != -1,
                                 "Cant remove batch if no batch selected");

        BenthosBatchTableModel tableModel = handler.getTableModel();
        BenthosBatchRowModel selectedBatch = tableModel.getEntry(rowIndex);

        Preconditions.checkState(!TuttiEntities.isNew(selectedBatch),
                                 "Can't remove batch if batch is not persisted");

        // remove selected batch and all his children

        // remove parent batch (will destroy all his childs from db)
        persistenceService.deleteBenthosBatch(selectedBatch.getId());

        if (selectedBatch.isBatchRoot()) {
            // update speciesUsed
            handler.removeFromSpeciesUsed(selectedBatch);
        } else {

            // remove from his parent
            BenthosBatchRowModel parentBatch = selectedBatch.getParentBatch();
            parentBatch.getChildBatch().remove(selectedBatch);
        }

        // collect of rows to remove from model
        rowToRemove = Sets.newHashSet();
        rowToRemove.add(selectedBatch);

        handler.collectChildren(selectedBatch, rowToRemove);
    }

    @Override
    public void postSuccessAction() {
        super.postSuccessAction();

        JXTable table = handler.getTable();

        // remove all rows from the model
        getModel().getRows().removeAll(rowToRemove);

        // refresh table from parent batch row index to the end
        handler.getTableModel().fireTableDataChanged();

        // select parent batch row
        TuttiUIUtil.selectFirstCellOnFirstRowAndStopEditing(table);
    }

}
