package fr.ifremer.tutti.ui.swing.content.operation.catches.individualobservation.create;

/*
 * #%L
 * Tutti :: UI
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import fr.ifremer.tutti.persistence.entities.CaracteristicMap;
import fr.ifremer.tutti.persistence.entities.data.IndividualObservationBatch;
import fr.ifremer.tutti.persistence.entities.data.IndividualObservationBatchs;
import fr.ifremer.tutti.persistence.entities.referential.Caracteristic;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiBeanUIModel;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.jdesktop.beans.AbstractSerializableBean;

import java.io.Serializable;
import java.util.List;
import java.util.Map;

/**
 * @author kmorin <kmorin@codelutin.com>
 * @since 1.4
 */
public class CreateIndividualObservationBatchUIModel extends AbstractTuttiBeanUIModel<IndividualObservationBatch, CreateIndividualObservationBatchUIModel> {

    private static final long serialVersionUID = 1L;

    public static final String PROPERTY_AVAILABLE_SPECIES = "availableSpecies";

    public static final String PROPERTY_BATCH_SPECIES = "batchSpecies";

    public static final String PROPERTY_SPECIES_FROM_BATCH = "speciesFromBatch";

    public static final String PROPERTY_CREATE_FROM_BATCH = "createFromBatch";

    public static final String PROPERTY_CARACTERISTICS = "caracteristics";

    /**
     * All available species (sets by protocol).
     *
     * @since 2.0
     */
    protected List<Species> availableSpecies;

    /**
     * All species used in batches with their count.
     *
     * @since 2.7
     */
    protected Map<Species, Integer> batchSpecies;

    /**
     * Does we want to create rows from batch?
     *
     * @since 2.7
     */
    protected boolean createFromBatch;

    /**
     * Flag to stop fires when flag is on.
     *
     * @since 2.8
     */
    protected boolean valueIsAdjusting;

    /**
     * Delegate edit object.
     *
     * @since 1.3
     */
    protected final IndividualObservationBatch editObject =
            IndividualObservationBatchs.newIndividualObservationBatch();

    /**
     * Default caracteristics coming from protocol.
     *
     * @since 2.5
     */
    protected final List<Caracteristic> defaultCaracteristic;

    /**
     * Default caracteristics editors.
     *
     * @since 2.5
     */
    protected final List<CaracteristicMapPropertyEditor> caracteristicEditors =
            Lists.newArrayList();

    public CreateIndividualObservationBatchUIModel(List<Caracteristic> defaultCaracteristic) {
        super(null, null);
        this.defaultCaracteristic = defaultCaracteristic;
        editObject.setCaracteristics(new CaracteristicMap());
    }

    public boolean isValueIsAdjusting() {
        return valueIsAdjusting;
    }

    public void setValueIsAdjusting(boolean valueIsAdjusting) {
        this.valueIsAdjusting = valueIsAdjusting;
    }

    public Species getSpecies() {
        return editObject.getSpecies();
    }

    public void setSpecies(Species species) {
        Object oldValue = getSpecies();
        Object oldSpeciesFromBatch = isSpeciesFromBatch();
        editObject.setSpecies(species);
        firePropertyChanged(IndividualObservationBatch.PROPERTY_SPECIES, oldValue, species);
        firePropertyChanged(PROPERTY_SPECIES_FROM_BATCH, oldSpeciesFromBatch, isSpeciesFromBatch());
    }

    public Float getWeight() {
        return editObject.getWeight();
    }

    public void setWeight(Float weight) {
        Object oldValue = getWeight();
        editObject.setWeight(weight);
        firePropertyChanged(IndividualObservationBatch.PROPERTY_WEIGHT, oldValue, weight);
    }

    public Float getSize() {
        return editObject.getSize();
    }

    public void setSize(Float size) {
        Object oldValue = getSize();
        editObject.setSize(size);
        firePropertyChanged(IndividualObservationBatch.PROPERTY_SIZE, oldValue, size);
    }

    public Caracteristic getLengthStepCaracteristic() {
        return editObject.getLengthStepCaracteristic();
    }

    public void setLengthStepCaracteristic(Caracteristic lengthStepCaracteristic) {
        Object oldValue = getLengthStepCaracteristic();
        editObject.setLengthStepCaracteristic(lengthStepCaracteristic);
        firePropertyChanged(IndividualObservationBatch.PROPERTY_LENGTH_STEP_CARACTERISTIC, oldValue, lengthStepCaracteristic);
    }

    public CaracteristicMap getCaracteristics() {
        return editObject.getCaracteristics();
    }

    public void setCaracteristics(CaracteristicMap caracteristics) {

        editObject.setCaracteristics(caracteristics);
        firePropertyChanged(IndividualObservationBatch.PROPERTY_CARACTERISTICS, null, caracteristics);
    }

    public List<Species> getAvailableSpecies() {
        return availableSpecies;
    }

    public void setAvailableSpecies(List<Species> availableSpecies) {
        this.availableSpecies = availableSpecies;
        firePropertyChanged(PROPERTY_AVAILABLE_SPECIES, null, availableSpecies);
    }

    public Map<Species, Integer> getBatchSpecies() {
        return batchSpecies;
    }

    public boolean isSpeciesFromBatch() {
        return MapUtils.isNotEmpty(batchSpecies) &&
               getSpecies() != null &&
               batchSpecies.containsKey(getSpecies());
    }

    public boolean isSpeciesFromBatchWithCount() {
        return isSpeciesFromBatch() && batchSpecies.get(getSpecies()) > 0;
    }

    public void setBatchSpecies(Map<Species, Integer> batchSpecies) {
        this.batchSpecies = batchSpecies;
        firePropertyChanged(PROPERTY_BATCH_SPECIES, null, batchSpecies);
        firePropertyChanged(PROPERTY_SPECIES_FROM_BATCH, null, isSpeciesFromBatch());
        setCreateFromBatch(isCreateFromBatch());
    }

    public boolean isCreateFromBatch() {
        return createFromBatch;
    }

    public void setCreateFromBatch(boolean createFromBatch) {
        Object oldValue = isCreateFromBatch();
        this.createFromBatch = createFromBatch;
        firePropertyChanged(PROPERTY_CREATE_FROM_BATCH, oldValue, createFromBatch);
    }

    public List<Caracteristic> getDefaultCaracteristic() {
        return defaultCaracteristic;
    }

    @Override
    protected IndividualObservationBatch newEntity() {
        return IndividualObservationBatchs.newIndividualObservationBatch();
    }

    public void reset() {
        setAvailableSpecies(Lists.<Species>newArrayList());
        setBatchSpecies(Maps.<Species, Integer>newHashMap());
        setCreateFromBatch(false);
        setSpecies(null);
        setWeight(null);
        setSize(null);
        setLengthStepCaracteristic(null);
        for (CaracteristicMapPropertyEditor editor : caracteristicEditors) {
            editor.setProperty(null);
        }
        getCaracteristics().clear();
    }

    public CaracteristicMapPropertyEditor newCaracteristicEditor(Caracteristic caracteristic) {
        CaracteristicMapPropertyEditor editor = new CaracteristicMapPropertyEditor(caracteristic);
        caracteristicEditors.add(editor);
        return editor;
    }

    /**
     * Model to edit a caracteristic and flush it in the caracteristc map of
     * the global model.
     * <p/>
     * Acts as a bean for the given caracteristic.
     *
     * @since 2.5
     */
    public class CaracteristicMapPropertyEditor extends AbstractSerializableBean {

        private static final long serialVersionUID = 1L;

        public static final String PROPERTY_PROPERTY = "property";

        private final Caracteristic caracteristic;

        protected Serializable property;

        public Serializable getProperty() {
            return property;
        }

        public void setProperty(Serializable property) {
            Object oldValue = getProperty();
            this.property = property;
            getCaracteristics().put(caracteristic, property);
            if (ObjectUtils.notEqual(oldValue, property)) {
                firePropertyChanged(PROPERTY_PROPERTY, oldValue, property);
                CreateIndividualObservationBatchUIModel.this.firePropertyChanged(PROPERTY_CARACTERISTICS, null, getCaracteristics());
            }
        }

        public CaracteristicMapPropertyEditor(Caracteristic caracteristic) {
            this.caracteristic = caracteristic;
        }
    }

    @Override
    public void firePropertyChanged(String propertyName,
                                    Object oldValue,
                                    Object newValue) {
        if (!valueIsAdjusting) {
            firePropertyChange(propertyName, oldValue, newValue);
        }
    }
}
