package fr.ifremer.tutti.ui.swing.util;

/*
 * #%L
 * Tutti :: UI
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import fr.ifremer.tutti.persistence.entities.data.Cruise;
import fr.ifremer.tutti.persistence.entities.data.Cruises;
import fr.ifremer.tutti.ui.swing.TuttiUIContext;
import jaxx.runtime.JAXXObject;
import jaxx.runtime.JAXXUtil;
import jaxx.runtime.SwingUtil;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.jaxx.application.ApplicationBusinessException;
import org.nuiton.jaxx.application.swing.util.ApplicationUIUtil;

import javax.swing.ImageIcon;
import javax.swing.JComponent;
import java.awt.Desktop;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;

import static org.nuiton.i18n.I18n.t;

/**
 * Created: 14/06/12
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public final class TuttiUIUtil extends ApplicationUIUtil {

    /** Logger. */
    private static final Log log = LogFactory.getLog(TuttiUIUtil.class);

    private TuttiUIUtil() {
        // never instanciate util class
    }

    public static final MouseListener GRAB_FOCUS_ON_ENTER_LISTENER = new MouseAdapter() {
        @Override
        public void mouseEntered(MouseEvent e) {
            JComponent source = (JComponent) e.getSource();
            source.grabFocus();
        }
    };

    public static TuttiUIContext getApplicationContext(JAXXObject ui) {
        return (TuttiUIContext) ApplicationUIUtil.getApplicationContext(ui);
    }

    public static void setParentUI(JAXXObject ui, TuttiUI<?, ?> parentUI) {
        JAXXUtil.initContext(ui, parentUI);
        setApplicationContext(ui, parentUI.getHandler().getContext());
    }

    public static void tryToConnectToUpdateUrl(String urlAsString,
                                               String badUrlFormatI18nKey,
                                               String notReachI18nKey,
                                               String notFoundI18nKey) {
        URL url;
        // get url
        try {
            url = new URL(urlAsString);
        } catch (MalformedURLException e) {
            if (log.isDebugEnabled()) {
                log.debug("Bad url syntax at " + urlAsString, e);
            }
            throw new ApplicationBusinessException(t(badUrlFormatI18nKey, urlAsString));
        }

        URLConnection urlConnection;
        // try to connect (fail if network or remote host does not exists)
        try {
            urlConnection = url.openConnection();
            urlConnection.setConnectTimeout(10000);
            urlConnection.connect();
        } catch (Exception e) {
            if (log.isDebugEnabled()) {
                log.debug("Could not connect to " + urlAsString, e);
            }
            throw new ApplicationBusinessException(t(notReachI18nKey, urlAsString));
        }

        // try to open the resource (fail if resources does not exist)
        try {
            urlConnection.setReadTimeout(1000);
            InputStream inputStream = null;
            try {
                inputStream = urlConnection.getInputStream();
            } finally {
                IOUtils.closeQuietly(inputStream);
            }
        } catch (Exception e) {
            if (log.isDebugEnabled()) {
                log.debug("Could not found file at to " + urlAsString, e);
            }
            throw new ApplicationBusinessException(t(notFoundI18nKey, urlAsString));
        }
    }

    public static ImageIcon getCruiseIcon(Cruise cruise) {
        String iconName = "cruise";
        if (cruise != null) {

            if (Cruises.isDirty(cruise)) {
                iconName = "cruise-dirty";
            } else if (Cruises.isReadyToSynch(cruise)) {
                iconName = "cruise-ready-to-sync";
            } else if (Cruises.isSynch(cruise)) {
                iconName = "cruise-waiting";
            }
        }
        return SwingUtil.createActionIcon(iconName);
    }

    public static void browseFile(File file) {

        Desktop desktop = getDesktopForBrowse();
        try {
            desktop.browse(file.toURI());

        } catch (IOException e) {
            TuttiUIContext.getApplicationContext().getErrorHelper().showErrorDialog(t("swing.error.cannot.open.file"));
        }
    }
}
