package fr.ifremer.tutti.ui.swing.content.operation.catches;

/*
 * #%L
 * Tutti :: UI
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Function;
import com.google.common.collect.Sets;
import fr.ifremer.tutti.TuttiConfiguration;
import fr.ifremer.tutti.persistence.entities.data.CatchBatch;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.type.WeightUnit;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.BenthosBatchRowModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.frequency.BenthosFrequencyCellComponent;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.frequency.BenthosFrequencyUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.split.SplitBenthosBatchUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.SpeciesBatchRowModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.frequency.SpeciesFrequencyCellComponent;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.frequency.SpeciesFrequencyUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.split.SplitSpeciesBatchUI;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiTabContainerUIHandler;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import fr.ifremer.tutti.ui.swing.util.TuttiUI;
import fr.ifremer.tutti.ui.swing.util.catches.EnterWeightUI;
import fr.ifremer.tutti.ui.swing.util.computable.ComputableData;
import fr.ifremer.tutti.util.Weights;
import jaxx.runtime.JAXXUtil;
import jaxx.runtime.swing.CardLayout2Ext;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.batik.dom.svg.SAXSVGDocumentFactory;
import org.apache.batik.dom.svg.SVGOMRectElement;
import org.apache.batik.dom.svg.SVGOMTextElement;
import org.apache.batik.swing.JSVGCanvas;
import org.apache.batik.swing.svg.GVTTreeBuilderAdapter;
import org.apache.batik.swing.svg.GVTTreeBuilderEvent;
import org.apache.batik.util.RunnableQueue;
import org.apache.batik.util.XMLResourceDescriptor;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTitledPanel;
import org.nuiton.jaxx.application.swing.tab.TabHandler;
import org.nuiton.util.Resource;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.css.CSSStyleDeclaration;
import org.w3c.dom.events.EventListener;
import org.w3c.dom.events.EventTarget;
import org.w3c.dom.svg.SVGRect;
import org.w3c.dom.svg.SVGStylable;

import javax.swing.*;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import java.awt.*;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.IOException;
import java.net.URL;
import java.util.Objects;
import java.util.Set;

import static org.nuiton.i18n.I18n.n;
import static org.nuiton.i18n.I18n.t;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 0.3
 */
public class EditCatchesUIHandler extends AbstractTuttiTabContainerUIHandler<EditCatchesUIModel, EditCatchesUI>
        implements TabHandler {

    public static final String MAIN_CARD = "main";

    public static final String CREATE_BATCH_CARD = "createBatch";

    public static final String SPLIT_BATCH_CARD = "splitBatch";

    public static final String ADD_SAMPLE_CATEGORY_BATCH_CARD = "addSampleCategoryBatch";

    public static final String EDIT_FREQUENCY_CARD = "editFrequency";

    public static final String EDIT_CARACTERISTICS_CARD = "caracteristicsCard";

    public static final String CAROUSSEL_TREMIE_VESSEL = "carousselTremieVessel";

    public static final String CLASSIC_VESSEL = "classicVessel";

    static {
        n("tutti.editCatchBatch.field.catchTotalWeight");
        n("tutti.editCatchBatch.field.catchTotalRejectedWeight");
        n("tutti.editCatchBatch.field.speciesTotalSortedWeight");
        n("tutti.editCatchBatch.field.benthosTotalSortedWeight");
        n("tutti.editCatchBatch.field.marineLitterTotalWeight");

        n("tutti.editCatchBatch.field.catchTotalSortedComputedWeight");
        n("tutti.editCatchBatch.field.catchTotalSortedSortedComputedWeight");
        n("tutti.editCatchBatch.field.catchTotalUnsortedComputedWeight");
        n("tutti.editCatchBatch.field.speciesTotalComputedWeight");
        n("tutti.editCatchBatch.field.speciesTotalUnsortedComputedWeight");
        n("tutti.editCatchBatch.field.speciesTotalSampleSortedComputedWeight");
        n("tutti.editCatchBatch.field.benthosTotalComputedWeight");
        n("tutti.editCatchBatch.field.benthosTotalUnsortedComputedWeight");
        n("tutti.editCatchBatch.field.benthosTotalSampleSortedComputedWeight");

        n("tutti.editCatchBatch.field.speciesDistinctSortedSpeciesCount");
        n("tutti.editCatchBatch.field.benthosDistinctSortedSpeciesCount");
    }

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(EditCatchesUIHandler.class);

    /**
     * To monitor changes on the incoming fishing operation.
     *
     * @since 0.3
     */
    private TuttiBeanMonitor<EditCatchesUIModel> catchBatchMonitor;

    /**
     * To remove {@code 0} value of total rejected weight.
     *
     * @since 3.5
     */
    private final PropertyChangeListener totalWeightRejectedListener = new PropertyChangeListener() {

        public void propertyChange(PropertyChangeEvent evt) {

            EditCatchesUIModel source = (EditCatchesUIModel) evt.getSource();

            Float totalWeight = (Float) evt.getNewValue();

            if (totalWeight != null && Weights.isEqualWeight(totalWeight, 0.f)) {

                // remove the totalWeight (see https://forge.codelutin.com/issues/5144)
                source.setCatchTotalRejectedWeight(null);

            }
        }
    };

    protected JSVGCanvas canvas;

    protected Document svgDocument;

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    public void beforeInit(EditCatchesUI ui) {
        super.beforeInit(ui);

        this.catchBatchMonitor = new TuttiBeanMonitor<EditCatchesUIModel>(
                EditCatchesUIModel.PROPERTY_MARINE_LITTER_TOTAL_WEIGHT,
                EditCatchesUIModel.PROPERTY_SPECIES_TOTAL_SORTED_WEIGHT,
                EditCatchesUIModel.PROPERTY_SPECIES_TOTAL_INERT_WEIGHT,
                EditCatchesUIModel.PROPERTY_SPECIES_TOTAL_LIVING_NOT_ITEMIZED_WEIGHT,
                EditCatchesUIModel.PROPERTY_BENTHOS_TOTAL_SORTED_WEIGHT,
                EditCatchesUIModel.PROPERTY_BENTHOS_TOTAL_INERT_WEIGHT,
                EditCatchesUIModel.PROPERTY_BENTHOS_TOTAL_LIVING_NOT_ITEMIZED_WEIGHT);

        EditCatchesUIModel model = ui.getContextValue(EditCatchesUIModel.class);

        model.setValidationContext(getContext().getValidationContext());

        listModelIsModify(model);

        catchBatchMonitor.setBean(model);
    }

    @Override
    public void afterInit(EditCatchesUI ui) {

        ui.getBenthosTabSplitBatch().getModel().setSplitMode(true);
        ui.getBenthosTabAddSampleCategoryBatch().getModel().setSplitMode(false);

        ui.getSpeciesTabSplitBatch().getModel().setSplitMode(true);
        ui.getSpeciesTabAddSampleCategoryBatch().getModel().setSplitMode(false);

        initUI(ui);

        EditCatchesUIModel model = getModel();

        changeValidatorContext(model.getValidationContext(), getValidator());
        listenValidationTableHasNoFatalError(getValidator(), model);

        setCustomTab(0, model);
        setCustomTab(1, model);
        setCustomTab(2, ui.getSpeciesTabContent().getModel());
        setCustomTab(3, ui.getBenthosTabContent().getModel());
        setCustomTab(4, ui.getMarineLitterTabContent().getModel());
        setCustomTab(5, ui.getIndividualObservationTabContent().getModel());
        setCustomTab(6, ui.getAccidentalTabContent().getModel());

        getTabPanel().setSelectedIndex(1);
        // when internal tab change, close any attachments popup
        getTabPanel().addChangeListener(new ChangeListener() {
            @Override
            public void stateChanged(ChangeEvent e) {
                closeAttachments();
            }
        });

        try {
            initResumeSvg();

        } catch (IOException err) {
            if (log.isErrorEnabled()) {
                log.error("error while initializing the resume background", err);
            }
            getContext().getErrorHelper().showErrorDialog(t("tutti.editCatchBatch.svgLoading.error"), err);
        }
    }

    public void initResumeSvg() throws IOException {
        String parser = XMLResourceDescriptor.getXMLParserClassName();
        SAXSVGDocumentFactory f = new SAXSVGDocumentFactory(parser);
        URL url = Resource.getURL("EcranResume.svg");
        svgDocument = f.createDocument(url.toString());

        canvas = new JSVGCanvas();
        canvas.setSize(new Dimension(1, 1));
        canvas.setMySize(new Dimension(1, 1));

        getUI().getSvgCanvasPanel().add(canvas, BorderLayout.CENTER);

        canvas.setRecenterOnResize(true);

        canvas.addGVTTreeBuilderListener(new GVTTreeBuilderAdapter() {
            public void gvtBuildCompleted(GVTTreeBuilderEvent e) {

                TuttiConfiguration config = getConfig();
                WeightUnit catchWeightUnit = WeightUnit.KG;
                WeightUnit speciesWeightUnit = config.getSpeciesWeightUnit();
                WeightUnit benthosWeightUnit = config.getBenthosWeightUnit();
                WeightUnit marineLitterWeightUnit = config.getMarineLitterWeightUnit();

                EditCatchesUIModel model = getModel();
                initSvgField(CatchBatch.PROPERTY_CATCH_TOTAL_WEIGHT, model.getCatchTotalComputedOrNotWeight(), catchWeightUnit);

                initSvgField(CatchBatch.PROPERTY_CATCH_TOTAL_SORTED_COMPUTED_WEIGHT, catchWeightUnit);
                initSvgField(CatchBatch.PROPERTY_CATCH_TOTAL_REJECTED_WEIGHT, model.getCatchTotalRejectedComputedOrNotWeight(), catchWeightUnit);
                initSvgField(CatchBatch.PROPERTY_CATCH_TOTAL_SORTED_SORTED_COMPUTED_WEIGHT, catchWeightUnit);

                initSvgField(CatchBatch.PROPERTY_SPECIES_TOTAL_SORTED_WEIGHT, model.getSpeciesTotalSortedComputedOrNotWeight(), speciesWeightUnit);
                initSvgField(CatchBatch.PROPERTY_SPECIES_TOTAL_SAMPLE_SORTED_COMPUTED_WEIGHT, speciesWeightUnit);

                initSvgField(CatchBatch.PROPERTY_BENTHOS_TOTAL_SORTED_WEIGHT, model.getBenthosTotalSortedComputedOrNotWeight(), benthosWeightUnit);
                initSvgField(CatchBatch.PROPERTY_BENTHOS_TOTAL_SAMPLE_SORTED_COMPUTED_WEIGHT, benthosWeightUnit);

                initSvgField(CatchBatch.PROPERTY_CATCH_TOTAL_UNSORTED_COMPUTED_WEIGHT, catchWeightUnit);
                initSvgField(CatchBatch.PROPERTY_SPECIES_TOTAL_UNSORTED_COMPUTED_WEIGHT, speciesWeightUnit);
                initSvgField(CatchBatch.PROPERTY_BENTHOS_TOTAL_UNSORTED_COMPUTED_WEIGHT, benthosWeightUnit);

                model.addPropertyChangeListener(new ChangeElementBackgroundColorPropertyChangeListener(
                        CatchBatch.PROPERTY_SPECIES_TOTAL_UNSORTED_COMPUTED_WEIGHT,
                        Sets.newHashSet(CatchBatch.PROPERTY_CATCH_TOTAL_REJECTED_WEIGHT,
                                        CatchBatch.PROPERTY_CATCH_TOTAL_REJECTED_COMPUTED_WEIGHT,
                                        CatchBatch.PROPERTY_SPECIES_TOTAL_SORTED_WEIGHT,
                                        CatchBatch.PROPERTY_SPECIES_TOTAL_SORTED_COMPUTED_WEIGHT,
                                        CatchBatch.PROPERTY_SPECIES_TOTAL_SAMPLE_SORTED_COMPUTED_WEIGHT),
                        canvas,
                        svgDocument,
                        new Function<EditCatchesUIModel, Color>() {
                            @Override
                            public Color apply(EditCatchesUIModel model) {
                                boolean warning = model.isSpeciesTotalUnsortedComputedWeightInWarning();
                                return warning ? Color.ORANGE : Color.decode("#006bba");
                            }
                        }));

                model.addPropertyChangeListener(new ChangeElementBackgroundColorPropertyChangeListener(
                        CatchBatch.PROPERTY_BENTHOS_TOTAL_UNSORTED_COMPUTED_WEIGHT,
                        Sets.newHashSet(CatchBatch.PROPERTY_CATCH_TOTAL_REJECTED_WEIGHT,
                                        CatchBatch.PROPERTY_CATCH_TOTAL_REJECTED_COMPUTED_WEIGHT,
                                        CatchBatch.PROPERTY_BENTHOS_TOTAL_SORTED_WEIGHT,
                                        CatchBatch.PROPERTY_BENTHOS_TOTAL_SORTED_COMPUTED_WEIGHT,
                                        CatchBatch.PROPERTY_BENTHOS_TOTAL_SAMPLE_SORTED_COMPUTED_WEIGHT),
                        canvas,
                        svgDocument,
                        new Function<EditCatchesUIModel, Color>() {
                            @Override
                            public Color apply(EditCatchesUIModel model) {
                                boolean warning = model.isBenthosTotalUnsortedComputedWeightInWarning();
                                return warning ? Color.ORANGE : Color.decode("#006bba");
                            }
                        }));

                initSvgField(CatchBatch.PROPERTY_SPECIES_TOTAL_COMPUTED_WEIGHT, speciesWeightUnit);
                initSvgField(CatchBatch.PROPERTY_BENTHOS_TOTAL_COMPUTED_WEIGHT, benthosWeightUnit);
                initSvgField(CatchBatch.PROPERTY_MARINE_LITTER_TOTAL_WEIGHT, model.getMarineLitterTotalComputedOrNotWeight(), marineLitterWeightUnit);

                model.addPropertyChangeListener(new RatioPropertyChangeListener("ratioSpeciesSampleSortedOverSpeciesSortedWeightLabel",
                                                                                CatchBatch.PROPERTY_SPECIES_TOTAL_SAMPLE_SORTED_COMPUTED_WEIGHT,
                                                                                CatchBatch.PROPERTY_SPECIES_TOTAL_SORTED_WEIGHT,
                                                                                CatchBatch.PROPERTY_SPECIES_TOTAL_SORTED_COMPUTED_WEIGHT,
                                                                                canvas,
                                                                                svgDocument));
                model.addPropertyChangeListener(new RatioPropertyChangeListener("ratioBenthosSampleSortedOverBenthosSortedWeightLabel",
                                                                                CatchBatch.PROPERTY_BENTHOS_TOTAL_SAMPLE_SORTED_COMPUTED_WEIGHT,
                                                                                CatchBatch.PROPERTY_BENTHOS_TOTAL_SORTED_WEIGHT,
                                                                                CatchBatch.PROPERTY_BENTHOS_TOTAL_SORTED_COMPUTED_WEIGHT,
                                                                                canvas,
                                                                                svgDocument));

                initSpeciesCount(CatchBatch.PROPERTY_SPECIES_DISTINCT_SORTED_SPECIES_COUNT);
                initSpeciesCount(CatchBatch.PROPERTY_BENTHOS_DISTINCT_SORTED_SPECIES_COUNT);

            }
        });

        canvas.setDocumentState(JSVGCanvas.ALWAYS_DYNAMIC);
        canvas.setDocument(svgDocument);
    }

    protected void initSpeciesCount(final String property) {

        if (log.isDebugEnabled()) {
            log.debug("init " + property + " field");
        }

        canvas.getUpdateManager().getUpdateRunnableQueue().invokeLater(new Runnable() {
            @Override
            public void run() {
                Element labelElement = svgDocument.getElementById(property + "Label");
                labelElement.setTextContent(t("tutti.editCatchBatch.field." + property));
            }
        });

        getModel().addPropertyChangeListener(property, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                Integer value = (Integer) evt.getNewValue();
                Element labelElement = svgDocument.getElementById(property + "Value");
                labelElement.setTextContent(JAXXUtil.getStringValue(value));
            }
        });
    }

    protected void initSvgField(String property, WeightUnit weightUnit) {
        initSvgField(property, null, weightUnit);
    }

    protected void initSvgField(final String property,
                                final ComputableData<Float> computableData,
                                final WeightUnit weightUnit) {

        if (log.isDebugEnabled()) {
            log.debug("init " + property + " field");
        }

        canvas.getUpdateManager().getUpdateRunnableQueue().invokeLater(new Runnable() {
            @Override
            public void run() {
                Element rectElement = svgDocument.getElementById(property + "Rect");
                SVGOMRectElement rectElem = (SVGOMRectElement) rectElement;
                SVGRect bbox = rectElem.getBBox();
                Float x = bbox.getX();

                Element labelElement = svgDocument.getElementById(property + "Label");
                if (computableData == null) {
                    SVGStylable field = (SVGStylable) labelElement;
                    CSSStyleDeclaration style = field.getStyle();
                    style.setProperty("font-style", "italic", null);
                }

                SVGOMTextElement labelTextElem = (SVGOMTextElement) labelElement;
                labelElement.setTextContent(weightUnit.decorateLabel(t("tutti.editCatchBatch.field." + property)));
                bbox = labelTextElem.getBBox();
                float labelX = bbox.getX();
                float width = Math.abs(x - labelX);

                Element labelRectElement = svgDocument.getElementById(property + "LabelRect");
                labelRectElement.setAttribute("width", Float.toString(width + 10));
                labelRectElement.setAttribute("x", String.valueOf(labelX - 10));
            }
        });

        Color colorComputedWeights = getConfig().getColorComputedWeights();
        if (computableData == null) {

            // computed data value
            OnDataOrComputedDataValueChangedListener listener = new OnDataOrComputedDataValueChangedListener(property, weightUnit, true, canvas, svgDocument, colorComputedWeights);
            getModel().addPropertyChangeListener(property, listener);

        } else {

            // data or computed data value
            Element element = svgDocument.getElementById(property);
            EventTarget target = (EventTarget) element;
            target.addEventListener("click", new OnValueClickListener(computableData, property, weightUnit), false);

            OnDataOrComputedDataValueChangedListener listener = new OnDataOrComputedDataValueChangedListener(property, weightUnit, false, canvas, svgDocument, colorComputedWeights);
            computableData.addPropertyChangeListener(listener);
        }
    }

    public void closeAttachments() {
        ui.getCatchesCaracteristicsAttachmentsButton().onCloseUI();
        ui.getSpeciesTabContent().getSpeciesBatchAttachmentsButton().onCloseUI();
        ui.getBenthosTabContent().getBenthosBatchAttachmentsButton().onCloseUI();
        ui.getMarineLitterTabContent().getMarineLitterBatchAttachmentsButton().onCloseUI();
        ui.getIndividualObservationTabContent().getIndividualObservationBatchAttachmentsButton().onCloseUI();
        ui.getAccidentalTabContent().getAccidentalBatchAttachmentsButton().onCloseUI();
    }

    @Override
    protected JComponent getComponentToFocus() {
        JComponent result;
        //TODO remove when svg resume is done?
        if (getModel().getCatchTotalComputedWeight() != null) {
            // if there is a computed value, never focus inside the component
            // see http://forge.codelutin.com/issues/4151
            result = null;
        } else {

            result = getUI().getCatchTotalWeightField();
        }
        return result;
    }

    @Override
    public void onCloseUI() {

        if (log.isDebugEnabled()) {
            log.debug("closing: " + ui);
        }
        ui.getCatchesCaracteristicsAttachmentsButton().onCloseUI();
        // close batches tabs, then general tab
        closeUI(ui.getSpeciesTabContent());
        closeUI(ui.getBenthosTabContent());
        closeUI(ui.getMarineLitterTabContent());
        closeUI(ui.getAccidentalTabContent());
        closeUI(ui.getIndividualObservationTabContent());
    }

    @Override
    public SwingValidator<EditCatchesUIModel> getValidator() {
        return ui.getValidator();
    }

    @Override
    protected Set<String> getPropertiesToIgnore() {
        Set<String> result = super.getPropertiesToIgnore();
        result.addAll(Sets.newHashSet(
                EditCatchesUIModel.PROPERTY_CATCH_TOTAL_COMPUTED_WEIGHT,
                EditCatchesUIModel.PROPERTY_CATCH_TOTAL_UNSORTED_COMPUTED_WEIGHT,
                EditCatchesUIModel.PROPERTY_CATCH_TOTAL_SORTED_COMPUTED_WEIGHT,
                EditCatchesUIModel.PROPERTY_CATCH_TOTAL_SORTED_SORTED_COMPUTED_WEIGHT,
                EditCatchesUIModel.PROPERTY_CATCH_TOTAL_REJECTED_COMPUTED_WEIGHT,
                EditCatchesUIModel.PROPERTY_SPECIES_TOTAL_COMPUTED_WEIGHT,
                EditCatchesUIModel.PROPERTY_SPECIES_TOTAL_SORTED_COMPUTED_WEIGHT,
                EditCatchesUIModel.PROPERTY_SPECIES_TOTAL_UNSORTED_COMPUTED_WEIGHT,
                EditCatchesUIModel.PROPERTY_SPECIES_TOTAL_SAMPLE_SORTED_COMPUTED_WEIGHT,
                EditCatchesUIModel.PROPERTY_SPECIES_TOTAL_INERT_COMPUTED_WEIGHT,
                EditCatchesUIModel.PROPERTY_SPECIES_TOTAL_LIVING_NOT_ITEMIZED_COMPUTED_WEIGHT,
                EditCatchesUIModel.PROPERTY_BENTHOS_TOTAL_COMPUTED_WEIGHT,
                EditCatchesUIModel.PROPERTY_BENTHOS_TOTAL_SORTED_COMPUTED_WEIGHT,
                EditCatchesUIModel.PROPERTY_BENTHOS_TOTAL_UNSORTED_COMPUTED_WEIGHT,
                EditCatchesUIModel.PROPERTY_BENTHOS_TOTAL_SAMPLE_SORTED_COMPUTED_WEIGHT,
                EditCatchesUIModel.PROPERTY_BENTHOS_TOTAL_INERT_COMPUTED_WEIGHT,
                EditCatchesUIModel.PROPERTY_BENTHOS_TOTAL_LIVING_NOT_ITEMIZED_COMPUTED_WEIGHT,
                EditCatchesUIModel.PROPERTY_SPECIES_DISTINCT_SORTED_SPECIES_COUNT,
                EditCatchesUIModel.PROPERTY_BENTHOS_DISTINCT_SORTED_SPECIES_COUNT,
                EditCatchesUIModel.PROPERTY_ATTACHMENT,
                EditCatchesUIModel.PROPERTY_BATCH_UPDATED
        ));
        return result;
    }

    @Override
    public JTabbedPane getTabPanel() {
        return ui.getTabPane();
    }

    @Override
    public boolean onTabChanged(int currentIndex, int newIndex) {
        ui.getCatchesCaracteristicsAttachmentsButton().onCloseUI();
        ui.getComputeSpeciesBatchButton().setVisible(newIndex < 4);
        ui.getCleanSpeciesBatchButton().setVisible(newIndex < 3);
        return super.onTabChanged(currentIndex, newIndex);
    }

    @Override
    public boolean onHideTab(int currentIndex, int newIndex) {

        closeAttachments();

        //FIXME 20130203 kmorin: cannot change tab if model is modified
        // (I do not even know why it is set to modified and have no time 
        // before the demo)
        EditCatchesUIModel model = getModel();
        boolean result;
        if (model.isModify()) {

            if (model.isValid()) {

                // ask user to save, do not save or cancel action
                int answer = askSaveBeforeLeaving(
                        t("tutti.editCatchBatch.askSaveBeforeLeaving.saveCatchBatch"));
                switch (answer) {
                    case JOptionPane.OK_OPTION:

                        // persist catch batch
                        getContext().getActionEngine().runAction(getUI().getSaveButton());

                        result = true;
                        break;

                    case JOptionPane.NO_OPTION:

                        // won't save modification
                        // so since we will edit a new operation, nothing to do here

                        // persist catch batch
                        getContext().getActionEngine().runAction(getUI().getCancelButton());

                        result = true;
                        break;
                    default:

                        // other case, use cancel action
                        result = false;
                }
            } else {

                // model is not valid, ask user to loose modification or cancel
                result = askCancelEditBeforeLeaving(
                        t("tutti.editCatchBatch.askCancelEditBeforeLeaving.cancelEditCatchBatch"));

                if (result) {

                    // ok will revert any modification
                    getContext().getActionEngine().runAction(ui.getCancelButton());
                }
            }

        } else {

            // model not modify, can change tab
            result = true;
        }
        return result;
    }

    @Override
    public void onShowTab(int currentIndex, int newIndex) {
        registerValidators();
    }

    @Override
    public boolean removeTab(int i) {
        return false;
    }

    @Override
    public boolean onRemoveTab() {
        return false;
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    public void uninstallTotalRejectWeightListener() {
        getModel().removePropertyChangeListener(EditCatchesUIModel.PROPERTY_CATCH_TOTAL_REJECTED_WEIGHT, totalWeightRejectedListener);
    }

    public void installTotalRejectWeightListener() {
        getModel().addPropertyChangeListener(EditCatchesUIModel.PROPERTY_CATCH_TOTAL_REJECTED_WEIGHT, totalWeightRejectedListener);
    }

    public TuttiBeanMonitor<EditCatchesUIModel> getCatchBatchMonitor() {
        return catchBatchMonitor;
    }

    protected void registerValidators() {
        registerValidators(getValidator(),
                           ui.getSpeciesTabContent().getHandler().getValidator(),
                           ui.getBenthosTabContent().getHandler().getValidator(),
                           ui.getMarineLitterTabContent().getHandler().getValidator()
        );
    }

    public void editSpeciesFrequencies(SpeciesFrequencyCellComponent.FrequencyCellEditor editor) {

        SpeciesFrequencyUI frequencyEditor = ui.getSpeciesTabFrequencyEditor();

        frequencyEditor.getHandler().editBatch(editor);

        // open frequency editor
        setSpeciesSelectedCard(EditCatchesUIHandler.EDIT_FREQUENCY_CARD);

        // update title
        SpeciesBatchRowModel editRow = editor.getEditRow();
        String title = buildReminderLabelTitle(editRow.getSpecies(),
                                               editRow,
                                               ui.getSpeciesTabFishingOperationReminderLabel().getTitle(),
                                               t("tutti.editSpeciesFrequencies.title"));
        ui.getSpeciesTabFrequencyEditorReminderLabel().setTitle(title);
    }

    public void splitSpeciesBatch(SpeciesBatchRowModel editRow,
                                  SplitSpeciesBatchUI splitBatchEditor) {

        splitBatchEditor.getHandler().editBatch(editRow);

        // open split editor
        setSpeciesSelectedCard(EditCatchesUIHandler.SPLIT_BATCH_CARD);

        // update title
        String title = buildReminderLabelTitle(editRow.getSpecies(),
                                               editRow,
                                               ui.getSpeciesTabFishingOperationReminderLabel().getTitle(),
                                               t("tutti.splitSpeciesBatch.title"));
        ui.getSpeciesTabSplitBatchReminderLabel().setTitle(title);
    }

    public void addSampleCategorySpeciesBatch(SpeciesBatchRowModel editRow,
                                              SplitSpeciesBatchUI splitBatchEditor,
                                              int sampleCategoryId) {

        splitBatchEditor.getHandler().editBatch(editRow, sampleCategoryId);

        // open split editor
        setSpeciesSelectedCard(EditCatchesUIHandler.ADD_SAMPLE_CATEGORY_BATCH_CARD);

        // update title
        String title = buildReminderLabelTitle(editRow.getSpecies(),
                                               editRow,
                                               ui.getSpeciesTabFishingOperationReminderLabel().getTitle(),
                                               t("tutti.addSampleCategorySpeciesBatch.title"));
        ui.getSpeciesTabAddSampleCategoryBatchReminderLabel().setTitle(title);
    }

    public void setSpeciesSelectedCard(String card) {
        JPanel panel = ui.getSpeciesTabPanel();
        CardLayout2Ext layout = (CardLayout2Ext) panel.getLayout();
        if (!card.equals(layout.getSelected())) {
            layout.setSelected(card);

            JPanel actionPanel = getUI().getCreateFishingOperationActions();
            if (MAIN_CARD.equals(card)) {
                registerValidators();
                actionPanel.setVisible(true);

            } else {
                actionPanel.setVisible(false);
                TuttiUI tuttiUi = null;
                JXTitledPanel titlePanel = null;
                String title = "";

                if (CREATE_BATCH_CARD.equals(card)) {
                    tuttiUi = ui.getSpeciesTabCreateBatch();
                    titlePanel = ui.getSpeciesTabCreateBatchReminderLabel();
                    title = n("tutti.createSpeciesBatch.title");

                } else if (SPLIT_BATCH_CARD.equals(card)) {
                    tuttiUi = ui.getSpeciesTabSplitBatch();

                } else if (ADD_SAMPLE_CATEGORY_BATCH_CARD.equals(card)) {
                    tuttiUi = ui.getSpeciesTabAddSampleCategoryBatch();

                } else if (EDIT_FREQUENCY_CARD.equals(card)) {
                    tuttiUi = ui.getSpeciesTabFrequencyEditor();
                }

                if (tuttiUi != null) {
                    registerValidators(tuttiUi.getHandler().getValidator());
                }
                if (titlePanel != null) {
                    titlePanel.setTitle(ui.getSpeciesTabFishingOperationReminderLabel().getTitle() + " - " + t(title));
                }
            }
        }
    }

    public void editBenthosFrequencies(BenthosFrequencyCellComponent.FrequencyCellEditor editor) {

        BenthosFrequencyUI frequencyEditor = ui.getBenthosTabFrequencyEditor();

        frequencyEditor.getHandler().editBatch(editor);

        // open frequency editor
        setBenthosSelectedCard(EditCatchesUIHandler.EDIT_FREQUENCY_CARD);

        // update title
        BenthosBatchRowModel editRow = editor.getEditRow();
        String title = buildReminderLabelTitle(editRow.getSpecies(),
                                               editRow,
                                               ui.getBenthosTabFishingOperationReminderLabel().getTitle(),
                                               t("tutti.editBenthosFrequencies.title"));
        ui.getBenthosTabFrequencyEditorReminderLabel().setTitle(title);
    }

    public void splitBenthosBatch(BenthosBatchRowModel editRow,
                                  SplitBenthosBatchUI splitBatchEditor) {


        splitBatchEditor.getHandler().editBatch(editRow);

        // open split editor
        setBenthosSelectedCard(EditCatchesUIHandler.SPLIT_BATCH_CARD);

        // update title
        String title = buildReminderLabelTitle(editRow.getSpecies(),
                                               editRow,
                                               ui.getBenthosTabFishingOperationReminderLabel().getTitle(),
                                               t("tutti.splitBenthosBatch.title"));
        ui.getBenthosTabSplitBatchReminderLabel().setTitle(title);
    }

    public void addSampleCategoryBenthosBatch(BenthosBatchRowModel editRow,
                                              SplitBenthosBatchUI splitBatchEditor,
                                              int sampleCategoryId) {

        splitBatchEditor.getHandler().editBatch(editRow, sampleCategoryId);

        // open split editor
        setBenthosSelectedCard(EditCatchesUIHandler.ADD_SAMPLE_CATEGORY_BATCH_CARD);

        // update title
        String title = buildReminderLabelTitle(editRow.getSpecies(),
                                               editRow,
                                               ui.getBenthosTabFishingOperationReminderLabel().getTitle(),
                                               t("tutti.addSampleCategoryBenthosBatch.title"));
        ui.getBenthosTabAddSampleCategoryBatchReminderLabel().setTitle(title);
    }

    public void setBenthosSelectedCard(String card) {
        JPanel panel = ui.getBenthosTabPanel();
        CardLayout2Ext layout = (CardLayout2Ext) panel.getLayout();
        if (!card.equals(layout.getSelected())) {
            layout.setSelected(card);

            JPanel actionPanel = getUI().getCreateFishingOperationActions();
            if (MAIN_CARD.equals(card)) {
                registerValidators();
                actionPanel.setVisible(true);

            } else {
                actionPanel.setVisible(false);
                TuttiUI tuttiUi = null;
                JXTitledPanel titlePanel = null;
                String title = "";

                if (CREATE_BATCH_CARD.equals(card)) {
                    tuttiUi = ui.getBenthosTabCreateBatch();
                    titlePanel = ui.getBenthosTabCreateBatchReminderLabel();
                    title = n("tutti.createBenthosBatch.title");

                } else if (SPLIT_BATCH_CARD.equals(card)) {
                    tuttiUi = ui.getBenthosTabSplitBatch();

                } else if (ADD_SAMPLE_CATEGORY_BATCH_CARD.equals(card)) {
                    tuttiUi = ui.getBenthosTabAddSampleCategoryBatch();

                } else if (EDIT_FREQUENCY_CARD.equals(card)) {
                    tuttiUi = ui.getBenthosTabFrequencyEditor();

                }

                if (tuttiUi != null) {
                    registerValidators(tuttiUi.getHandler().getValidator());
                }
                if (titlePanel != null) {
                    titlePanel.setTitle(ui.getBenthosTabFishingOperationReminderLabel().getTitle() + " - " + t(title));
                }
            }
        }
    }

    public void setMarineLitterSelectedCard(String card) {
        JPanel panel = ui.getMarineLitterTabPanel();
        CardLayout2Ext layout = (CardLayout2Ext) panel.getLayout();
        if (!card.equals(layout.getSelected())) {
            layout.setSelected(card);

            JPanel actionPanel = getUI().getCreateFishingOperationActions();
            if (MAIN_CARD.equals(card)) {
                registerValidators();
                actionPanel.setVisible(true);

            } else {
                actionPanel.setVisible(false);
                TuttiUI tuttiUi = null;
                JXTitledPanel titlePanel = null;
                String title = "";

                if (CREATE_BATCH_CARD.equals(card)) {
                    tuttiUi = ui.getMarineLitterTabCreateBatch();
                    titlePanel = ui.getMarineLitterTabCreateBatchReminderLabel();
                    title = n("tutti.createMarineLitterBatch.title");
                }

                if (tuttiUi != null) {
                    registerValidators(tuttiUi.getHandler().getValidator());
                }
                if (titlePanel != null) {
                    titlePanel.setTitle(ui.getMarineLitterTabCreateBatchReminderLabel().getTitle() + " - " + t(title));
                }
            }
        }
    }

    public void setIndividualObservationSelectedCard(String card) {
        setIndividualObservationSelectedCard(card, null);
    }

    public void setIndividualObservationSelectedCard(String card, Species species) {
        JPanel panel = ui.getIndividualObservationTabPanel();
        CardLayout2Ext layout = (CardLayout2Ext) panel.getLayout();
        if (!card.equals(layout.getSelected())) {
            layout.setSelected(card);

            JPanel actionPanel = getUI().getCreateFishingOperationActions();
            if (MAIN_CARD.equals(card)) {
                registerValidators();
                actionPanel.setVisible(true);

            } else {
                actionPanel.setVisible(false);
                TuttiUI tuttiUi = null;
                JXTitledPanel titlePanel = null;
                String title = "";

                if (CREATE_BATCH_CARD.equals(card)) {
                    tuttiUi = ui.getIndividualObservationTabCreateBatch();
                    titlePanel = ui.getIndividualObservationTabCreateBatchReminderLabel();
                    title = n("tutti.createIndividualObservationBatch.title");

                } else if (EDIT_CARACTERISTICS_CARD.equals(card)) {
                    titlePanel = ui.getIndividualObservationCaracteristicMapEditorReminderLabel();
                    title = n("tutti.editCaracteristics.title");
                }

                if (tuttiUi != null) {
                    registerValidators(tuttiUi.getHandler().getValidator());
                }
                if (titlePanel != null) {
                    titlePanel.setTitle(ui.getIndividualObservationTabFishingOperationReminderLabel().getTitle() + " - " + t(title, decorate(species)));
                }
            }
        }
    }

    public void setAccidentalSelectedCard(String card) {
        setAccidentalSelectedCard(card, null);
    }

    public void setAccidentalSelectedCard(String card, Species species) {
        JPanel panel = ui.getAccidentalTabPanel();
        CardLayout2Ext layout = (CardLayout2Ext) panel.getLayout();
        if (!card.equals(layout.getSelected())) {
            layout.setSelected(card);

            JPanel actionPanel = getUI().getCreateFishingOperationActions();
            if (MAIN_CARD.equals(card)) {
                registerValidators();
                actionPanel.setVisible(true);

            } else {
                actionPanel.setVisible(false);
                TuttiUI tuttiUi = null;
                JXTitledPanel titlePanel = null;
                String title = "";

                if (CREATE_BATCH_CARD.equals(card)) {
                    tuttiUi = ui.getAccidentalTabCreateBatch();
                    titlePanel = ui.getAccidentalTabCreateBatchReminderLabel();
                    title = n("tutti.createAccidentalBatch.title");

                } else if (EDIT_CARACTERISTICS_CARD.equals(card)) {
                    titlePanel = ui.getAccidentalCaracteristicMapEditorReminderLabel();
                    title = n("tutti.editCaracteristics.title");
                }


                if (tuttiUi != null) {
                    registerValidators(tuttiUi.getHandler().getValidator());
                }
                if (titlePanel != null) {
                    titlePanel.setTitle(ui.getAccidentalTabFishingOperationReminderLabel().getTitle() + " - " + t(title, decorate(species)));
                }
            }
        }
    }

    public String getWeightStringValueForTotalWeight(JLabel label, Float rejectWeight, Float totalWeight) {
        String result;
        if (rejectWeight == null || totalWeight == null) {

            // no reject weight, so can let this weight
            result = getWeightStringValue(label, totalWeight);
        } else {
            result = t("tutti.editCatchBatch.field.speciesOrBenthosTotalWeight.not.computed");
        }
        return result;
    }

    public Color getWeightColorForTotalWeight(Float rejectWeight, Float totalWeight) {
        //jTextField.setDisabledTextColor(
        Color result;
        if (rejectWeight == null || totalWeight == null) {

            // no reject weight, so can let this weight
            result = getConfig().getColorComputedWeights();
        } else {
            result = Color.RED;
        }
        return result;
    }

    private class OnValueClickListener implements EventListener {

        private final ComputableData<Float> computableData;

        private final String property;

        private final WeightUnit weightUnit;

        public OnValueClickListener(ComputableData<Float> computableData, String property, WeightUnit weightUnit) {
            this.computableData = computableData;
            this.property = property;
            this.weightUnit = weightUnit;
        }

        public void handleEvent(org.w3c.dom.events.Event evt) {
            log.info("element clicked");
            EnterWeightUI dialog = new EnterWeightUI(getContext());
            Float originalWeight = computableData.getData();
            Float weight = dialog.openAndGetWeightValue(t("tutti.editCatchBatch.field." + property),
                                                        originalWeight,
                                                        weightUnit);

            if (!Objects.equals(originalWeight, weight)) {

                computableData.setData(weight);
            }
        }
    }

    private static abstract class UpdateRunnableQueuePropertyChangeListener implements PropertyChangeListener {

        protected final JSVGCanvas canvas;
        protected RunnableQueue updateRunnableQueue;
        protected final Document svgDocument;

        public UpdateRunnableQueuePropertyChangeListener(JSVGCanvas canvas,
                                                         Document svgDocument) {
            this.canvas = canvas;
            this.svgDocument = svgDocument;
        }

        protected RunnableQueue getUpdateRunnableQueue() {
            if (updateRunnableQueue == null) {
                updateRunnableQueue = canvas.getUpdateManager().getUpdateRunnableQueue();

            }
            return updateRunnableQueue;
        }
    }

    private static class OnDataOrComputedDataValueChangedListener extends UpdateRunnableQueuePropertyChangeListener {

        private final String property;
        private final WeightUnit weightUnit;
        private final boolean onlyReactOnComputedData;
        private final Color colorComputedWeights;

        public OnDataOrComputedDataValueChangedListener(String property,
                                                        WeightUnit weightUnit,
                                                        boolean onlyReactOnComputedData,
                                                        JSVGCanvas canvas,
                                                        Document svgDocument,
                                                        Color colorComputedWeights) {
            super(canvas, svgDocument);
            this.property = property;
            this.weightUnit = weightUnit;
            this.onlyReactOnComputedData = onlyReactOnComputedData;
            this.colorComputedWeights = colorComputedWeights;
        }

        @Override
        public void propertyChange(PropertyChangeEvent evt) {

            boolean computedData;

            Float newValue;

            if (onlyReactOnComputedData) {

                computedData = true;
                newValue = (Float) evt.getNewValue();

            } else {

                ComputableData<Float> computableData = (ComputableData<Float>) evt.getSource();

                if (computableData.getData() == null) {

                    computedData = true;
                    newValue = computableData.getComputedData();

                } else {

                    computedData = false;
                    newValue = computableData.getData();

                }

            }
            updateValue(newValue, computedData);

        }

        private void updateValue(final Float value, final boolean computed) {

            getUpdateRunnableQueue().invokeLater
                    (new Runnable() {
                        public void run() {
                            if (log.isDebugEnabled()) {
                                log.debug("update " + property + " field");
                            }

                            Element element = svgDocument.getElementById(property + "Value");
                            if (element == null) {
                                return;
                            }

                            //TODO i18n ?
                            String textContent;
                            if (value != null) {
                                textContent = Weights.getWeightStringValue(value) + " " + weightUnit.getShortLabel();
                            } else {
                                textContent = null;
                            }
                            element.setTextContent(textContent);

                            SVGStylable field = (SVGStylable) element;
                            CSSStyleDeclaration style = field.getStyle();

                            String computedColor = "#" + Integer.toHexString(colorComputedWeights.getRGB()).substring(2);
                            style.setProperty("fill", computed ? computedColor : "#000000", null);
                            style.setProperty("font-style", computed ? "italic" : "normal", null);

                            SVGOMTextElement textElem = (SVGOMTextElement) element;
                            SVGRect bbox = textElem.getBBox();
                            if (bbox != null) {
                                float width = bbox.getWidth() + 15;
                                Element rectElement = svgDocument.getElementById(property + "Rect");
                                rectElement.setAttribute("width", Float.toString(width));
                            }
                        }
                    });

        }
    }

    private static class ChangeElementBackgroundColorPropertyChangeListener extends UpdateRunnableQueuePropertyChangeListener {

        private final String elementId;
        private Set<String> propertiesToListen;
        private Function<EditCatchesUIModel, Color> colorFunction;

        public ChangeElementBackgroundColorPropertyChangeListener(String elementId,
                                                                  Set<String> propertiesToListen,
                                                                  JSVGCanvas canvas,
                                                                  Document svgDocument,
                                                                  Function<EditCatchesUIModel, Color> colorFunction) {
            super(canvas, svgDocument);
            this.elementId = elementId;
            this.propertiesToListen = propertiesToListen;
            this.colorFunction = colorFunction;
        }

        @Override
        public void propertyChange(PropertyChangeEvent evt) {
            EditCatchesUIModel model = (EditCatchesUIModel) evt.getSource();
            final Color background = colorFunction.apply(model);

            if (propertiesToListen.contains(evt.getPropertyName())) {

                getUpdateRunnableQueue().invokeLater
                        (new Runnable() {
                            public void run() {
                                if (log.isDebugEnabled()) {
                                    log.debug("update " + elementId + " field");
                                }

                                Element rectElement = svgDocument.getElementById(elementId + "LabelRect");
                                SVGStylable field = (SVGStylable) rectElement;
                                CSSStyleDeclaration style = field.getStyle();

                                String color = "#" + Integer.toHexString(background.getRGB()).substring(2);
                                style.setProperty("fill", color, null);
                            }
                        });
            }
        }
    }

    private static class RatioPropertyChangeListener extends UpdateRunnableQueuePropertyChangeListener {

        private final String elementId;
        private final String numeratorProperty;
        private final String denominatorProperty;
        private final String denominatorComputedProperty;
        private Integer ratio = null;

        public RatioPropertyChangeListener(String elementId,
                                           String numeratorProperty,
                                           String denominatorProperty,
                                           String denominatorComputedProperty,
                                           JSVGCanvas canvas,
                                           Document svgDocument) {
            super(canvas, svgDocument);
            this.elementId = elementId;
            this.numeratorProperty = numeratorProperty;
            this.denominatorProperty = denominatorProperty;
            this.denominatorComputedProperty = denominatorComputedProperty;
        }

        @Override
        public void propertyChange(PropertyChangeEvent evt) {
            String propertyName = evt.getPropertyName();

            if (numeratorProperty.equals(propertyName) || denominatorProperty.equals(propertyName)) {
                EditCatchesUIModel model = (EditCatchesUIModel) evt.getSource();

                try {
                    String numerator = BeanUtils.getProperty(model, numeratorProperty);
                    String denominator = BeanUtils.getProperty(model, denominatorProperty);
                    if (denominator == null) {
                        denominator = BeanUtils.getProperty(model, denominatorComputedProperty);
                    }

                    if (numerator != null && denominator != null) {
                        Float numeratorValue = Float.valueOf(numerator);
                        Float denominatorValue = Float.valueOf(denominator);

                        if (denominatorValue != 0) {
                            ratio = (int) (100 * numeratorValue / denominatorValue);
                        }
                    }

                } catch (ReflectiveOperationException e) {
                    if (log.isErrorEnabled()) {
                        log.error("Error while computing the ration", e);
                    }
                }

                getUpdateRunnableQueue().invokeLater
                        (new Runnable() {
                            public void run() {
                                if (log.isDebugEnabled()) {
                                    log.debug("update " + elementId + " field");
                                }

                                Element ratioElement = svgDocument.getElementById(elementId);
                                String textContent;
                                if (ratio != null) {
                                    textContent = ratio + "%";
                                } else {
                                    textContent = null;
                                }
                                ratioElement.setTextContent(textContent);
                            }
                        });
            }
        }
    }

}
