package fr.ifremer.tutti.ui.swing.content.operation.catches.individualobservation;

/*
 * #%L
 * Tutti :: UI
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import fr.ifremer.tutti.util.Numbers;
import org.nuiton.jaxx.application.swing.table.ColumnIdentifier;
import fr.ifremer.tutti.type.WeightUnit;
import fr.ifremer.tutti.persistence.entities.CaracteristicMap;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.data.Attachment;
import fr.ifremer.tutti.persistence.entities.data.FishingOperation;
import fr.ifremer.tutti.persistence.entities.data.IndividualObservationBatch;
import fr.ifremer.tutti.persistence.entities.data.SampleCategory;
import fr.ifremer.tutti.persistence.entities.referential.Caracteristic;
import fr.ifremer.tutti.persistence.entities.referential.CaracteristicQualitativeValue;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.service.DecoratorService;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.ui.swing.content.operation.catches.AbstractTuttiBatchTableUIHandler;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUIHandler;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUIModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.individualobservation.create.CreateIndividualObservationBatchUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.individualobservation.create.CreateIndividualObservationBatchUIModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.SpeciesBatchRowModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.SpeciesBatchUIModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.frequency.SpeciesFrequencyRowModel;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import fr.ifremer.tutti.ui.swing.util.TuttiUI;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;
import fr.ifremer.tutti.ui.swing.util.attachment.AttachmentCellEditor;
import fr.ifremer.tutti.ui.swing.util.attachment.AttachmentCellRenderer;
import fr.ifremer.tutti.ui.swing.util.caracteristics.CaracteristicMapCellComponent;
import fr.ifremer.tutti.ui.swing.util.caracteristics.CaracteristicMapColumnUIHandler;
import fr.ifremer.tutti.ui.swing.util.caracteristics.CaracteristicMapEditorUI;
import fr.ifremer.tutti.ui.swing.util.comment.CommentCellEditor;
import fr.ifremer.tutti.ui.swing.util.comment.CommentCellRenderer;
import fr.ifremer.tutti.ui.swing.util.table.CaracteristicColumnIdentifier;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.table.DefaultTableColumnModelExt;
import org.nuiton.decorator.Decorator;
import org.nuiton.validator.NuitonValidatorResult;

import javax.swing.JComponent;
import javax.swing.JOptionPane;
import java.io.Serializable;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static org.nuiton.i18n.I18n.t;

/**
 * @author kmorin <kmorin@codelutin.com>
 * @since 1.4
 */
public class IndividualObservationBatchUIHandler
        extends AbstractTuttiBatchTableUIHandler<IndividualObservationBatchRowModel, IndividualObservationBatchUIModel, IndividualObservationBatchTableModel, IndividualObservationBatchUI>
        implements CaracteristicMapColumnUIHandler {

    private static final Log log =
            LogFactory.getLog(IndividualObservationBatchUIHandler.class);

    /**
     * Weight unit.
     *
     * @since 2.5
     */
    protected WeightUnit weightUnit;

    protected WeightUnit speciesWeightUnit;

    public Integer PMFM_ID_SORTED_UNSORTED;

    public IndividualObservationBatchUIHandler() {
        super(IndividualObservationBatchRowModel.PROPERTY_SPECIES,
              IndividualObservationBatchRowModel.PROPERTY_WEIGHT,
              IndividualObservationBatchRowModel.PROPERTY_SIZE,
              IndividualObservationBatchRowModel.PROPERTY_LENGTH_STEP_CARACTERISTIC,
              IndividualObservationBatchRowModel.PROPERTY_CARACTERISTICS,
              IndividualObservationBatchRowModel.PROPERTY_DEFAULT_CARACTERISTICS,
              IndividualObservationBatchRowModel.PROPERTY_COMMENT,
              IndividualObservationBatchRowModel.PROPERTY_ATTACHMENT);
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiBatchTableUIHandler methods                           --//
    //------------------------------------------------------------------------//

    @Override
    protected ColumnIdentifier<IndividualObservationBatchRowModel> getCommentIdentifier() {
        return IndividualObservationBatchTableModel.COMMENT;
    }

    @Override
    protected ColumnIdentifier<IndividualObservationBatchRowModel> getAttachementIdentifier() {
        return IndividualObservationBatchTableModel.ATTACHMENT;
    }

    @Override
    public void selectFishingOperation(FishingOperation bean) {

        boolean empty = bean == null;

        IndividualObservationBatchUIModel model = getModel();

        List<IndividualObservationBatchRowModel> rows;

        if (empty) {
            rows = null;
        } else {

            if (log.isDebugEnabled()) {
                log.debug("Get individualObservation batch for fishingOperation: " +
                          bean.getId());
            }
            rows = Lists.newArrayList();

            if (!TuttiEntities.isNew(bean)) {
                PersistenceService persistenceService = getPersistenceService();
                List<IndividualObservationBatch> batches =
                        persistenceService.getAllIndividualObservationBatch(bean.getId());

                for (IndividualObservationBatch aBatch : batches) {

                    // set the surveycode
                    if (getDataContext().isProtocolFilled()) {
                        // get the surveycode from the species list of the model
                        List<Species> speciesList = getDataContext().getReferentSpeciesWithSurveyCode();
                        int i = speciesList.indexOf(aBatch.getSpecies());
                        if (i > -1) {
                            aBatch.setSpecies(speciesList.get(i));
                        }
                    }

                    IndividualObservationBatchRowModel entry =
                            new IndividualObservationBatchRowModel(
                                    weightUnit,
                                    aBatch,
                                    getModel().getDefaultCaracteristic());

                    List<Attachment> attachments =
                            persistenceService.getAllAttachments(entry.getObjectType(),
                                                                 entry.getObjectId());

                    entry.addAllAttachment(attachments);
                    rows.add(entry);
                }
            }
        }
        model.setRows(rows);
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiTableUIHandler methods                                --//
    //------------------------------------------------------------------------//

    @Override
    public IndividualObservationBatchTableModel getTableModel() {
        return (IndividualObservationBatchTableModel) getTable().getModel();
    }

    @Override
    public JXTable getTable() {
        return ui.getTable();
    }

    @Override
    protected boolean isRowValid(IndividualObservationBatchRowModel row) {
        IndividualObservationBatch batch = row.toBean();
        NuitonValidatorResult validator = getValidationService().validateEditIndividualObservationBatch(batch);
        boolean result = !validator.hasErrorMessagess();
        return result;
    }

    @Override
    protected void saveSelectedRowIfRequired(TuttiBeanMonitor<IndividualObservationBatchRowModel> rowMonitor,
                                             IndividualObservationBatchRowModel row) {

        if (row != null && row.isValid() && rowMonitor.wasModified()) {

            // monitored bean was modified, save it
            if (log.isDebugEnabled()) {
                log.debug("Row " + row + " was modified, will save it");
            }

            String title = buildReminderLabelTitle(row.getSpecies(),
                                                   null,
                                                   "Sauvegarde de Données individuelles : ",
                                                   "Ligne :" + (getTableModel().getRowIndex(row) + 1));

            showInformationMessage(title);

            rowMonitor.setBean(null);
            saveRow(row);
            rowMonitor.setBean(row);

            // clear modified flag on the monitor
            rowMonitor.clearModified();
        }
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    public SwingValidator<IndividualObservationBatchUIModel> getValidator() {
        return ui.getValidator();
    }

    @Override
    public void beforeInit(IndividualObservationBatchUI ui) {

        super.beforeInit(ui);
        if (log.isDebugEnabled()) {
            log.debug("beforeInit: " + ui);
        }

        weightUnit = getConfig().getIndividualObservationWeightUnit();
        speciesWeightUnit = getConfig().getSpeciesWeightUnit();

        PMFM_ID_SORTED_UNSORTED =
                getPersistenceService().getSortedUnsortedCaracteristic().getIdAsInt();

        // get the default caracteristics
        List<Caracteristic> defaultCaracteristic =
                getDataContext().getDefaultIndividualObservationCaracteristics();

        // create model
        EditCatchesUIModel catchesUIModel =
                ui.getContextValue(EditCatchesUIModel.class);
        IndividualObservationBatchUIModel model =
                new IndividualObservationBatchUIModel(catchesUIModel,
                                                      defaultCaracteristic);
        ui.setContextValue(model);
    }

    @Override
    public void afterInit(IndividualObservationBatchUI ui) {

        if (log.isDebugEnabled()) {
            log.debug("afterInit: " + ui);
        }

        initUI(ui);

        JXTable table = getTable();

        // can show / hide some columns in model
        table.setColumnControlVisible(true);

        // create table column model
        DefaultTableColumnModelExt columnModel =
                new DefaultTableColumnModelExt();
        Decorator<Caracteristic> caracteristicDecorator =
                getDecorator(Caracteristic.class, DecoratorService.CARACTERISTIC_PARAMETER_ONLY_WITH_UNIT);
        Decorator<Caracteristic> caracteristicTipDecorator =
                getDecorator(Caracteristic.class, DecoratorService.CARACTERISTIC_WITH_UNIT);

        Decorator<CaracteristicQualitativeValue> caracteristicQualitativeDecorator =
                getDecorator(CaracteristicQualitativeValue.class, null);

        { // Species column

            Decorator<Species> speciesDecorator = getDecorator(
                    Species.class, DecoratorService.FROM_PROTOCOL);
            addComboDataColumnToModel(columnModel,
                                      IndividualObservationBatchTableModel.SPECIES,
                                      speciesDecorator, getDataContext().getReferentSpeciesWithSurveyCode());
        }

        { // Weight column

            addFloatColumnToModel(columnModel,
                                  IndividualObservationBatchTableModel.WEIGHT,
                                  weightUnit,
                                  table);
        }

        { // Size column

            addFloatColumnToModel(columnModel,
                                  IndividualObservationBatchTableModel.SIZE,
                                  TuttiUI.DECIMAL3_PATTERN,
                                  table);
        }

        { // Length step caracteristic column

            addComboDataColumnToModel(columnModel,
                                      IndividualObservationBatchTableModel.LENGTH_STEP_CARACTERISTIC,
                                      getDecorator(Caracteristic.class, null),
                                      getDataContext().getLengthStepCaracteristics());

        }

        List<Caracteristic> defaultCaracteristic =
                getModel().getDefaultCaracteristic();


        for (Caracteristic caracteristic : defaultCaracteristic) {
            String header = caracteristicDecorator.toString(caracteristic);
            String headerTip = caracteristicTipDecorator.toString(caracteristic);

            CaracteristicColumnIdentifier id = CaracteristicColumnIdentifier.newCaracteristicId(
                    caracteristic,
                    IndividualObservationBatchRowModel.PROPERTY_DEFAULT_CARACTERISTICS,
                    header,
                    headerTip
            );

            switch (caracteristic.getCaracteristicType()) {

                case NUMBER:

                    addFloatColumnToModel(columnModel,
                                          id,
                                          TuttiUI.DECIMAL3_PATTERN,
                                          table);

                    break;
                case QUALITATIVE:
                    List<CaracteristicQualitativeValue> values =
                            caracteristic.getQualitativeValue();
                    addComboDataColumnToModel(
                            columnModel,
                            id,
                            caracteristicQualitativeDecorator,
                            values);
                    break;
                case TEXT:

                    addColumnToModel(columnModel, id);

                    break;
            }
        }

        { // Other caracteristics column

            Set<Caracteristic> caracteristicsToSkip = Collections.unmodifiableSet(
                    Sets.newHashSet(getModel().getDefaultCaracteristic()));

            addColumnToModel(columnModel,
                             CaracteristicMapCellComponent.newEditor(ui, caracteristicsToSkip),
                             CaracteristicMapCellComponent.newRender(getContext()),
                             IndividualObservationBatchTableModel.OTHER_CARACTERISTICS);

        }

        { // Comment column

            addColumnToModel(columnModel,
                             CommentCellEditor.newEditor(ui),
                             CommentCellRenderer.newRender(),
                             IndividualObservationBatchTableModel.COMMENT);
        }

        { // File column

            addColumnToModel(columnModel,
                             AttachmentCellEditor.newEditor(ui),
                             AttachmentCellRenderer.newRender(getDecorator(Attachment.class, null)),
                             IndividualObservationBatchTableModel.ATTACHMENT);
        }

        // create table model
        IndividualObservationBatchTableModel tableModel =
                new IndividualObservationBatchTableModel(weightUnit,
                                                         columnModel);

        table.setModel(tableModel);
        table.setColumnModel(columnModel);

        initBatchTable(table, columnModel, tableModel);
    }

    @Override
    protected void beforeOpenPopup(int rowIndex, int columnIndex) {
        super.beforeOpenPopup(rowIndex, columnIndex);

        boolean enableRemove = false;

        if (rowIndex != -1) {

            // there is a selected row
            enableRemove = true;
        }
        IndividualObservationBatchUIModel model = getModel();
        model.setRemoveBatchEnabled(enableRemove);
    }

    @Override
    protected JComponent getComponentToFocus() {
        return getUI().getTable();
    }

    @Override
    public void onCloseUI() {
        if (log.isDebugEnabled()) {
            log.debug("closing: " + ui);
        }
        ui.getIndividualObservationBatchAttachmentsButton().onCloseUI();
    }

    @Override
    public CaracteristicMapEditorUI getCaracteristicMapEditor() {
        EditCatchesUI parent = getParentContainer(EditCatchesUI.class);
        return parent.getIndividualObservationCaracteristicMapEditor();
    }

    @Override
    public void showCaracteristicMapEditor(Species species) {
        EditCatchesUI parent = getParentContainer(EditCatchesUI.class);
        parent.getHandler().setIndividualObservationSelectedCard(EditCatchesUIHandler.EDIT_CARACTERISTICS_CARD, species);
    }

    @Override
    public void hideCaracteristicMapEditor() {
        EditCatchesUI parent = getParentContainer(EditCatchesUI.class);
        parent.getHandler().setIndividualObservationSelectedCard(EditCatchesUIHandler.MAIN_CARD);
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    public void createBatch() {

        EditCatchesUI parent = getParentContainer(EditCatchesUI.class);
        CreateIndividualObservationBatchUI createBatchEditor = parent.getIndividualObservationTabCreateBatch();

        createBatchEditor.getHandler().openUI(getModel());
        parent.getHandler().setIndividualObservationSelectedCard(EditCatchesUIHandler.CREATE_BATCH_CARD);
    }

    public void addBatch(CreateIndividualObservationBatchUIModel model) {
        if (model.isValid()) {

            IndividualObservationBatchTableModel tableModel = getTableModel();

            if (model.isCreateFromBatch()) {

                Species species = model.getSpecies();

                String speciesStr = decorate(species);

                // check if there is some rows to create

                if (!model.isSpeciesFromBatchWithCount()) {

                    String htmlMessage = String.format("<html>%s</html>",
                                                       t("tutti.createIndividualObservationBatch.warn.nocount", speciesStr));
                    JOptionPane.showMessageDialog(getContext().getMainUI(),
                                                  htmlMessage,
                                                  t("tutti.createIndividualObservationBatch.warn.nocount.title"),
                                                  JOptionPane.WARNING_MESSAGE);
                    return;
                }

                // check if there is already some indivudal data for this species

                Set<Species> speciesUsed = getModel().getSpeciesUsed();
                if (speciesUsed.contains(species)) {

                    String htmlMessage = String.format(
                            CONFIRMATION_FORMAT,
                            t("tutti.createIndividualObservationBatch.confirm.alreadyUsedSpecies.message", speciesStr),
                            t("tutti.createIndividualObservationBatch.confirm.alreadyUsedSpecies.help"));
                    int response = JOptionPane.showConfirmDialog(
                            getTopestUI(),
                            htmlMessage,
                            t("tutti.createIndividualObservationBatch.confirm.alreadyUsedSpecies.title"),
                            JOptionPane.OK_CANCEL_OPTION,
                            JOptionPane.WARNING_MESSAGE);


                    boolean create = response == JOptionPane.OK_OPTION;

                    if (!create) {
                        return;
                    }
                }
                EditCatchesUI parent = getParentContainer(EditCatchesUI.class);

                SpeciesBatchUIModel speciesBatchUIModel = parent.getSpeciesTabContent().getModel();

                // get species rows to use (all leafs for the given species)

                List<SpeciesBatchRowModel> leafs =
                        speciesBatchUIModel.getLeafs(species);

                List<IndividualObservationBatchRowModel> rowsToCreate = Lists.newArrayList();

                CaracteristicMap defaultCaracteristics = new CaracteristicMap();
                defaultCaracteristics.putAll(model.getCaracteristics());

                List<Caracteristic> defaultCaracteristic =
                        Lists.newArrayList(model.getDefaultCaracteristic());
                defaultCaracteristic.addAll(defaultCaracteristics.keySet());

                for (SpeciesBatchRowModel leaf : leafs) {

                    CaracteristicMap batchCaracteristics = new CaracteristicMap();

                    for (SampleCategory<?> sampleCategory : leaf) {
                        if (PMFM_ID_SORTED_UNSORTED.equals(sampleCategory.getCategoryId())) {
                            // do not use vrac / hors vrac caracteristic ?
                            continue;
                        }
                        if (sampleCategory.getCategoryValue() == null) {
                            // not using this category
                            continue;
                        }
                        if (defaultCaracteristic.contains(sampleCategory.getCategoryDef().getCaracteristic())) {

                            // use default caracteristics
                            defaultCaracteristics.put(sampleCategory.getCategoryDef().getCaracteristic(),
                                                      sampleCategory.getCategoryValue());
                        } else {

                            // use other caracteristics
                            batchCaracteristics.put(sampleCategory.getCategoryDef().getCaracteristic(),
                                                    sampleCategory.getCategoryValue());
                        }

                    }

                    List<SpeciesFrequencyRowModel> frequencies = leaf.getFrequency();

                    if (CollectionUtils.isEmpty(frequencies)) {

                        // no frequency, create a simple row

                        int nbRow = Numbers.getValueOrComputedValue(leaf.getNumber(), 0);

                        for (int i = 0; i < nbRow; i++) {
                            IndividualObservationBatchRowModel newRow = tableModel.createNewRow();
                            newRow.setSpecies(species);
                            newRow.getCaracteristics().putAll(batchCaracteristics);

//                            //FIXME Keep this ?
//                            newRow.getDefaultCaracteristics().putAll(defaultCaracteristics);
//                            //FIXME Keep this?
//                            newRow.setWeight(model.getWeight());
//                            //FIXME Keep this?
//                            newRow.setSize(model.getSize());
//                            //FIXME Keep this?
//                            newRow.setLengthStepCaracteristic(model.getLengthStepCaracteristic());

                            rowsToCreate.add(newRow);

                        }

                        if (nbRow == 1) {

                            // apply also weight found
                            IndividualObservationBatchRowModel lastRow =
                                    rowsToCreate.get(rowsToCreate.size() - 1);
                            Float weight = Numbers.getValueOrComputedValue(
                                    leaf.getWeight(),
                                    leaf.getFinestCategory().getCategoryWeight());
                            if (weight != null) {

                                // convert weight units
                                weight = speciesWeightUnit.toEntity(weight);
                                weight = weightUnit.fromEntity(weight);
                            }
                            lastRow.setWeight(weight);
                        }

                    } else {

                        boolean withOneFrequency = frequencies.size() == 1;

                        for (SpeciesFrequencyRowModel frequency : frequencies) {

                            // create a row for each frequency

                            int nbRow = Numbers.getValueOrComputedValue(
                                    frequency.getNumber(), 0);

                            Float weight = null;
                            if (nbRow == 1) {

                                // apply also weight found
                                weight = frequency.getWeight();
                                if (weight == null) {

                                    if (withOneFrequency) {

                                        // special case : only one frequency
                                        // with one value, can use the batch weight

                                        weight = Numbers.getValueOrComputedValue(
                                                leaf.getWeight(),
                                                leaf.getFinestCategory().getCategoryWeight());
                                    }
                                }

                                if (weight != null) {

                                    // convert weight units
                                    weight = speciesWeightUnit.toEntity(weight);
                                    weight = weightUnit.fromEntity(weight);
                                }


                            }

                            for (int i = 0; i < nbRow; i++) {
                                IndividualObservationBatchRowModel newRow = tableModel.createNewRow();
                                newRow.setSpecies(species);
                                newRow.setWeight(weight);
                                newRow.setSize(frequency.getLengthStep());
                                newRow.setLengthStepCaracteristic(frequency.getLengthStepCaracteristic());

                                newRow.getCaracteristics().putAll(batchCaracteristics);
                                newRow.getDefaultCaracteristics().putAll(defaultCaracteristics);

                                rowsToCreate.add(newRow);
                            }
                        }
                    }
                }

                int nbRowsToCreate = rowsToCreate.size();

                boolean create;

                int maxIndividualObservationRowsToCreate =
                        getConfig().getMaxIndividualObservationRowsToCreate();

                if (maxIndividualObservationRowsToCreate < nbRowsToCreate) {

                    // ask confirmation

                    String htmlMessage = String.format(
                            CONFIRMATION_FORMAT,
                            t("tutti.createIndividualObservationBatch.confirm.message", speciesStr, nbRowsToCreate),
                            t("tutti.createIndividualObservationBatch.confirm.help"));
                    int response = JOptionPane.showConfirmDialog(
                            getTopestUI(),
                            htmlMessage,
                            t("tutti.createIndividualObservationBatch.confirm.title"),
                            JOptionPane.OK_CANCEL_OPTION,
                            JOptionPane.WARNING_MESSAGE);


                    create = response == JOptionPane.OK_OPTION;
                } else {

                    create = true;
                }

                if (create) {

                    if (log.isInfoEnabled()) {
                        log.info("Will create " + rowsToCreate.size() + " individual observation batches");
                    }

                    for (IndividualObservationBatchRowModel newRow : rowsToCreate) {

                        recomputeRowValidState(newRow);

                        saveRow(newRow);

                        tableModel.addNewRow(newRow);
                    }

                    TuttiUIUtil.selectFirstCellOnLastRow(getTable());
                }

            } else {

                IndividualObservationBatchRowModel newRow = tableModel.createNewRow();
                newRow.setSpecies(model.getSpecies());
                newRow.setWeight(model.getWeight());
                newRow.setSize(model.getSize());
                newRow.setLengthStepCaracteristic(model.getLengthStepCaracteristic());
                newRow.getDefaultCaracteristics().putAll(model.getCaracteristics());

                recomputeRowValidState(newRow);

                saveRow(newRow);

                tableModel.addNewRow(newRow);
                TuttiUIUtil.selectFirstCellOnLastRow(getTable());
            }
        }

    }

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected void saveRow(IndividualObservationBatchRowModel row) {

        IndividualObservationBatch entityToSave = row.toEntity();

        CaracteristicMap caracteristics = new CaracteristicMap();
        entityToSave.setCaracteristics(caracteristics);

        // push back not null extra caracteristics
        for (Map.Entry<Caracteristic, Serializable> entry : row.getCaracteristics().entrySet()) {
            Serializable value = entry.getValue();
            if (value != null) {
                caracteristics.put(entry.getKey(), value);
            }
        }

        // push back not null default caracteristics
        for (Map.Entry<Caracteristic, Serializable> entry : row.getDefaultCaracteristics().entrySet()) {
            Serializable value = entry.getValue();
            if (value != null) {
                caracteristics.put(entry.getKey(), value);
            }
        }

        FishingOperation fishingOperation = getModel().getFishingOperation();
        entityToSave.setFishingOperation(fishingOperation);

        if (TuttiEntities.isNew(entityToSave)) {

            entityToSave = getPersistenceService().createIndividualObservationBatch(entityToSave);
            row.setId(entityToSave.getId());
        } else {
            getPersistenceService().saveIndividualObservationBatch(entityToSave);
        }

        getModel().fireBatchUpdated(row);
    }
}
