package fr.ifremer.tutti.ui.swing;

/*
 * #%L
 * Tutti :: UI
 * $Id: TuttiDbUpdaterCallBack.java 1531 2014-01-29 03:26:12Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-3.1/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/TuttiDbUpdaterCallBack.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import fr.ifremer.adagio.core.service.technical.synchro.ReferentialSynchroResult;
import fr.ifremer.shared.application.ApplicationTechnicalException;
import fr.ifremer.shared.application.swing.action.ApplicationActionException;
import fr.ifremer.shared.application.swing.action.ApplicationActionUI;
import fr.ifremer.tutti.TuttiConfiguration;
import fr.ifremer.tutti.persistence.ProgressionModel;
import fr.ifremer.tutti.persistence.entities.data.SampleCategoryModel;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.service.referential.TuttiReferentialSynchronizeService;
import fr.ifremer.tutti.ui.swing.action.AbstractTuttiAction;
import fr.ifremer.tutti.ui.swing.util.auth.AuthenticationInfo;
import org.apache.commons.io.FileUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.updater.ApplicationInfo;
import org.nuiton.updater.ApplicationUpdater;
import org.nuiton.updater.ApplicationUpdaterCallback;

import java.io.File;
import java.io.IOException;
import java.util.Map;

import static org.nuiton.i18n.I18n._;
import static org.nuiton.i18n.I18n.n_;

/**
 * CallBack to update db.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 2.6
 */
public class TuttiDbUpdaterCallBack implements ApplicationUpdaterCallback {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(TuttiDbUpdaterCallBack.class);

    public static final String DB_UPDATE_NAME = "db";

    static {
        n_("tutti.update.db");
    }

    protected final TuttiUIContext context;

    protected ProgressionModel progressionModel;

    protected boolean dbInstalled;

    protected boolean dbUpdated;

    protected final AbstractTuttiAction action;

    public TuttiDbUpdaterCallBack(AbstractTuttiAction action,
                                  ProgressionModel progressionModel) {
        this.action = action;
        this.context = action.getContext();
        this.progressionModel = progressionModel;
    }

    public boolean isDbUpdated() {
        return dbUpdated;
    }

    public boolean isDbInstalled() {
        return dbInstalled;
    }

    @Override
    public Map<String, ApplicationInfo> updateToDo(Map<String, ApplicationInfo> appToUpdate) {
        Map<String, ApplicationInfo> result = Maps.newHashMap();

        ApplicationInfo info = appToUpdate.get(DB_UPDATE_NAME);
        if (info == null) {
            dbInstalled = false;
            dbUpdated = false;
        } else {
            result.put(info.name, info);

            if (info.needAuthentication) {

                // ask auth
                AuthenticationInfo authenticationInfo = context.getAuthenticationInfo(info.url);
                if (authenticationInfo != null) {
                    info.setAuthentication(authenticationInfo.getLogin(), authenticationInfo.getPassword());
                }
            }

            if (context.isDbExist()) {

                // when db exists always an update
                dbUpdated = true;
            } else {

                // when no db, then always install
                dbInstalled = true;
            }
        }

        return result;
    }

    @Override
    public void startUpdate(ApplicationInfo info) {

        if (dbInstalled) {
            progressionModel.setMessage(_("tutti.applicationUpdater.startUpdate.db.installation", info.newVersion));
        } else if (dbUpdated) {
            progressionModel.setMessage(_("tutti.applicationUpdater.startUpdate.db.update", info.newVersion));
        }
    }

    @Override
    public void updateDone(Map<String, ApplicationInfo> appToUpdate,
                           Map<String, Exception> appUpdateError) {

        ApplicationInfo info = appToUpdate.get(DB_UPDATE_NAME);
        Exception error = appUpdateError.get(DB_UPDATE_NAME);
        if (error != null) {

            // something bad while updating db
            if (log.isErrorEnabled()) {
                log.error("Could not update db", error);
            }
            String errorMessage;
            if (info != null && info.needAuthentication) {
                errorMessage = _("tutti.updateDb.error.with.auth");
            } else {
                errorMessage = _("tutti.updateDb.error.with.noauth");
            }
            throw ApplicationActionException.propagateError(
                    action, new ApplicationTechnicalException(errorMessage, error));
        } else if (info != null) {

            if (log.isInfoEnabled()) {
                log.info(String.format(
                        "A db update was downloaded (oldVersion: %s, newVersion: %s), will process it.",
                        info.oldVersion, info.newVersion));
            }

            // before install or update, regenerate db configuration files

            TuttiConfiguration configuration = regenerateDbConf();

            configuration.generateExternalDbFiles(true);

            if (dbInstalled) {

                // first database, just copy it to correct directory

                prepareFirstDatabase(info);
            } else if (dbUpdated) {

                // launch a referential synchronize operation
                synchronizetDatabase(info);
            }
        }
    }

    @Override
    public void aborted(String propertiesURL, Exception eee) {
        if (log.isErrorEnabled()) {
            log.error("Could not update from " + propertiesURL, eee);
        }
        throw ApplicationActionException.propagateError(action, eee);
    }

    protected TuttiConfiguration regenerateDbConf() {

        TuttiConfiguration config = context.getConfig();

        context.showInformationMessage(
                "Regénérer les fichiers de configuration de la base");
        return config;
    }

    protected File getDbDirectory(ApplicationInfo info) {
        File[] sources = info.destDir.listFiles();
        Preconditions.checkState(
                sources != null && sources.length == 1,
                "Downloaded db should contains one directory at " + info.destDir);
        File result = sources[0];
        return result;
    }

    protected void prepareFirstDatabase(ApplicationInfo info) {
        if (log.isInfoEnabled()) {
            log.info("First time database was downloaded at version: " + info.newVersion);
        }
        File source = getDbDirectory(info);
        File target = context.getConfig().getDbDirectory();
        if (log.isInfoEnabled()) {
            log.info("Copy from " + source + " to " + target);
        }
        try {
            FileUtils.copyDirectory(source, target);
        } catch (IOException e) {
            throw new ApplicationTechnicalException(_("tutti.applicationUpdater.prepareFirstDB.copyDirectory.error", source, target), e);
        }
        try {
            FileUtils.deleteDirectory(source.getParentFile());
        } catch (IOException e) {
            throw new ApplicationTechnicalException(_("tutti.applicationUpdater.prepareFirstDB.deleteDirectory.error", target), e);
        }
    }

    protected void synchronizetDatabase(ApplicationInfo info) {
        if (log.isInfoEnabled()) {
            log.info(String.format("A database update was downloaded (oldVersion: %s, newVersion: %s), will launch a referential synchronize operation ", info.oldVersion, info.newVersion));
        }
        TuttiReferentialSynchronizeService service = context.getTuttiReferentialSynchronizeService();
        ReferentialSynchroResult result = new ReferentialSynchroResult();
        File dbDirectory = getDbDirectory(info);

        ApplicationActionUI actionUI = context.getActionUI();
        actionUI.getModel().setProgressionModel(result.getProgressionModel());
        service.prepare(dbDirectory, result);

        if (!result.isSuccess()) {
            throw new ApplicationTechnicalException(_("tutti.applicationUpdater.synchroDB.prepare.error"), result.getError());
        }

        service.synchronize(dbDirectory, result);

        if (!result.isSuccess()) {
            throw new ApplicationTechnicalException(_("tutti.applicationUpdater.synchroDB.synchro.error"), result.getError());
        }

        // reset cache
        if (log.isInfoEnabled()) {
            log.info("Reset all caches.");
        }
        PersistenceService persistence = context.getPersistenceService();
        persistence.clearAllCaches();

        // clean data context
        if (log.isInfoEnabled()) {
            log.info("Clean data context.");
        }
        SampleCategoryModel sampleCategoryModel =
                context.getDataContext().getSampleCategoryModel();
        context.getDataContext().clearContext();
        context.getDataContext().loadSampleCategoryModel(sampleCategoryModel);

        // replace the version.appup file content
        File target = context.getConfig().getDbDirectory();
        File versionFile = ApplicationUpdater.getVersionFile(target);
        if (log.isInfoEnabled()) {
            log.info("Replace content of file " + versionFile + " with " + info.newVersion);
        }
        try {
            ApplicationUpdater.storeVersionFile(target, info.newVersion);
        } catch (IOException e) {
            throw new ApplicationTechnicalException(
                    _("tutti.applicationUpdater.synchroDB.writeVersion.error", versionFile));
        }
    }
}
