package fr.ifremer.tutti.ui.swing;

/*
 * #%L
 * Tutti :: UI
 * $Id: TuttiUIContext.java 1272 2013-10-03 20:30:54Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-2.8.1/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/TuttiUIContext.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import fr.ifremer.tutti.PropagatePropertyChangeListener;
import fr.ifremer.tutti.TuttiBusinessException;
import fr.ifremer.tutti.TuttiIOUtil;
import fr.ifremer.tutti.TuttiTechnicalException;
import fr.ifremer.tutti.persistence.RessourceClassLoader;
import fr.ifremer.tutti.service.ClosedPersistenceService;
import fr.ifremer.tutti.service.DecoratorService;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.service.TuttiDataContext;
import fr.ifremer.tutti.service.TuttiServiceContext;
import fr.ifremer.tutti.service.ValidationService;
import fr.ifremer.tutti.service.catches.ExportCatchesReportService;
import fr.ifremer.tutti.service.catches.ValidateCruiseOperationsService;
import fr.ifremer.tutti.service.catches.WeightCleaningService;
import fr.ifremer.tutti.service.catches.WeightComputingService;
import fr.ifremer.tutti.service.catches.multipost.MultiPostExportService;
import fr.ifremer.tutti.service.catches.multipost.MultiPostImportService;
import fr.ifremer.tutti.service.export.TuttiExportService;
import fr.ifremer.tutti.service.protocol.ProtocolImportExportService;
import fr.ifremer.tutti.service.pupitri.PupitriImportExportService;
import fr.ifremer.tutti.service.referential.ReferentialImportExportService;
import fr.ifremer.tutti.service.referential.TuttiReferentialSynchronizeService;
import fr.ifremer.tutti.service.sumatra.CatchesSumatraExportService;
import fr.ifremer.tutti.ui.swing.config.TuttiApplicationConfig;
import fr.ifremer.tutti.ui.swing.content.MainUI;
import fr.ifremer.tutti.ui.swing.util.TuttiErrorHelper;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;
import fr.ifremer.tutti.ui.swing.util.UIMessageNotifier;
import fr.ifremer.tutti.ui.swing.util.action.TuttiActionUI;
import jaxx.runtime.JAXXContext;
import jaxx.runtime.swing.editor.bean.BeanDoubleList;
import jaxx.runtime.swing.editor.bean.BeanFilterableComboBox;
import jaxx.runtime.swing.help.JAXXHelpBroker;
import jaxx.runtime.swing.help.JAXXHelpUIHandler;
import jaxx.runtime.swing.session.BeanDoubleListState;
import jaxx.runtime.swing.session.BeanFilterableComboBoxState;
import jaxx.runtime.swing.session.State;
import jaxx.runtime.swing.session.SwingSession;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.beans.AbstractBean;
import org.nuiton.i18n.I18n;
import org.nuiton.i18n.init.DefaultI18nInitializer;
import org.nuiton.i18n.init.UserI18nInitializer;
import org.nuiton.util.converter.ConverterUtil;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import static org.nuiton.i18n.I18n._;
import static org.nuiton.i18n.I18n.n_;

/**
 * UI application context.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class TuttiUIContext extends AbstractBean implements Closeable, UIMessageNotifier, JAXXHelpUIHandler, PropagatePropertyChangeListener.PropagatePropertyChange {

    /** Logger. */
    private static final Log log = LogFactory.getLog(TuttiUIContext.class);

    public static final String PROPERTY_PROGRAM_ID = "programId";

    public static final String PROPERTY_CRUISE_ID = "cruiseId";

    public static final String PROPERTY_PROTOCOL_ID = "protocolId";

    public static final String PROPERTY_SCREEN = "screen";

    public static final String PROPERTY_PROGRAM_FILLED = "programFilled";

    public static final String PROPERTY_CRUISE__FILLED = "cruiseFilled";

    public static final String PROPERTY_PROTOCOL_FILLED = "protocolFilled";

    public static final String PROPERTY_VALIDATION_CONTEXT = "validationContext";

    public static final String PROPERTY_BUSY = "busy";

    public static final String PROPERTY_HIDE_BODY = "hideBody";

    public static final String PROPERTY_LOCALE = "locale";

    public static final Set<String> PROPERTIES_TO_SAVE = Sets.newHashSet(
            PROPERTY_PROGRAM_ID,
            PROPERTY_CRUISE_ID,
            PROPERTY_PROTOCOL_ID,
            PROPERTY_LOCALE);

    public static final String PROPERTY_DB_EXIST = "dbExist";

    public static final String PROPERTY_DB_LOADED = "dbLoaded";

    /**
     * Application context (only one for all the application).
     *
     * @since 0.1
     */
    private static TuttiUIContext applicationContext;

    /**
     * Application global configuration.
     *
     * @since 0.1
     */
    protected final TuttiApplicationConfig config;

    /**
     * ClassLoader ressource.
     *
     * @since 0.3
     */
    protected final RessourceClassLoader resourceLoader;

    /**
     * Service context used by any service.
     *
     * @since 0.1
     */
    protected final TuttiServiceContext serviceContext;

    /**
     * Swing session used to save ui states.
     *
     * @since 0.1
     */
    protected final SwingSession swingSession;

    /**
     * Erro helper.
     *
     * @since 1.0
     */
    protected final TuttiErrorHelper errorHelper;

    /**
     * Shared data context.
     *
     * @since 1.0.2
     */
    protected TuttiDataContext dataContext;

    /**
     * Tutti help broker.
     *
     * @since 1.1
     */
    protected TuttiHelpBroker helpBroker;

    /**
     * Current screen displayed in ui.
     *
     * @since 0.1
     */
    protected TuttiScreen screen;

    /**
     * Current locale used in application.
     *
     * @since 1.0.3
     */
    protected Locale locale;

    /**
     * Busy state ({@code true} when a blocking action is running).
     *
     * @since 1.0.3
     */
    protected boolean busy;

    /**
     * Flag to hide (or not) the body of application.
     *
     * @since 1.1
     */
    protected boolean hideBody;

    /**
     * Message notifiers.
     *
     * @since 0.3
     */
    protected final Set<UIMessageNotifier> messageNotifiers;

    /**
     * Validation context (used by fishingOperation screens).
     *
     * @since 0.3
     */
    private String validationContext;

    private MainUI mainUI;

    private TuttiActionUI actionUI;

    /**
     * Flag to know if there is an exsiting db.
     *
     * @since 1.0
     */
    private boolean dbExist;

    /**
     * Flag to know if there is a loaded db.
     *
     * @since 1.0
     */
    private boolean dbLoaded;

    private Properties helpMapping;

    public static TuttiUIContext newContext(TuttiApplicationConfig config) {
        Preconditions.checkNotNull(config);
        Preconditions.checkState(applicationContext == null,
                                 "Application context was already opened!");
        applicationContext = new TuttiUIContext(config);
        return applicationContext;
    }

    public static TuttiUIContext getApplicationContext() {
        return applicationContext;
    }

    public static TuttiErrorHelper getErrorHelper() {
        return applicationContext.errorHelper;
    }

    protected TuttiUIContext(TuttiApplicationConfig config) {
        this.config = config;
        this.resourceLoader = new RessourceClassLoader(Thread.currentThread().getContextClassLoader());
        this.serviceContext = new TuttiServiceContext(resourceLoader, config.getServiceConfig());

        Map<Class, State> additionalStates = Maps.newHashMap();
        additionalStates.put(BeanFilterableComboBox.class, new BeanFilterableComboBoxState());
        additionalStates.put(BeanDoubleList.class, new BeanDoubleListState());
        this.swingSession = new SwingSession(getConfig().getUIConfigFile(), false, additionalStates);

        this.errorHelper = new TuttiErrorHelper(this);
        this.dataContext = serviceContext.getDataContext();
        PropagatePropertyChangeListener.listenAndPropagateAll(dataContext, this);
        UIMessageNotifier logMessageNotifier = new UIMessageNotifier() {

            @Override
            public void showInformationMessage(String message) {
                if (StringUtils.isNotBlank(message)) {
                    message = message.replaceAll("\\<strong\\>","");
                    message = message.replaceAll("\\<.strong\\>","");
                    message = message.replaceAll("\\<li\\>","");
                    message = message.replaceAll("\\<.li\\>","");
                    message = message.replaceAll("\\<ul\\>","");
                    message = message.replaceAll("\\<.ul\\>","");
                    if (log.isInfoEnabled()) {
                        log.info(message);
                    }
                }
            }
        };
        this.messageNotifiers = Sets.newHashSet();
        addMessageNotifier(logMessageNotifier);
    }

    //------------------------------------------------------------------------//
    //-- Open / close methods                                               --//
    //------------------------------------------------------------------------//

    public void init() {

        config.getServiceConfig().prepareDirectories();

        // use our special classLoader (which will read some files from resources from a configuration directory)
        Thread.currentThread().setContextClassLoader(getResourceLoader());

        // converters are stored in current classloader, we need then to rescan them
        // each time we change current classloader
        ConverterUtil.deregister();
        ConverterUtil.initConverters();

        // Use shutdownHook to close context on System.exit
        Runtime.getRuntime().addShutdownHook(new Thread(new Runnable() {

            @Override
            public void run() {
                if (log.isInfoEnabled()) {
                    log.info("Close context on shutdown");
                }
                close();
            }
        }));

        //--------------------------------------------------------------------//
        // init db configuration
        //--------------------------------------------------------------------//

        config.getServiceConfig().getPersistenceConfig().initConfig(getResourceLoader());

//        // clean db cache (avoid a lots of headache :(
//        File cacheDirectory = config.getServiceConfig().getPersistenceConfig().getCacheDirectory();
//        if (cacheDirectory.exists()) {
//            // clean cache directory (fix soem headaches...)
//            TuttiIOUtil.cleanDirectory(
//                    cacheDirectory,
//                    _("tutti.db.deleteCache.error", cacheDirectory));
//        }

        //--------------------------------------------------------------------//
        // init i18n
        //--------------------------------------------------------------------//
        File i18nDirectory = config.getI18nDirectory();
        if (!config.isFullLaunchMode()) {

            i18nDirectory = new File(config.getDataDirectory(), "i18n");

            if (i18nDirectory.exists()) {
                // clean i18n cache
                TuttiIOUtil.cleanDirectory(
                        i18nDirectory,
                        _("tutti.i18n.deleteCache.error", i18nDirectory));
            }
        }

        TuttiIOUtil.forceMkdir(i18nDirectory,
                               _("tutti.i18n.mkDir.error", i18nDirectory));

        if (log.isDebugEnabled()) {
            log.debug("I18N directory: " + i18nDirectory);
        }

        Locale i18nLocale = config.getI18nLocale();

        if (log.isInfoEnabled()) {
            log.info(String.format("Starts i18n with locale [%s] at [%s]",
                                   i18nLocale, i18nDirectory));
        }
        I18n.init(new UserI18nInitializer(
                i18nDirectory, new DefaultI18nInitializer("tutti-i18n")),
                  i18nLocale);

        //--------------------------------------------------------------------//
        // init help
        //--------------------------------------------------------------------//

        File helpDirectory = config.getHelpDirectory();

        if (!config.isFullLaunchMode()) {

            if (!helpDirectory.exists()) {
                helpDirectory = new File(config.getDataDirectory(), "help");
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Help directory: " + helpDirectory);
        }
        TuttiIOUtil.forceMkdir(
                helpDirectory,
                _("tutti.help.mkDir.error", helpDirectory));

        // load help mapping
        String mappingProperties =
                "/tutti-help-" + i18nLocale.getLanguage() + ".properties";
        try {

            InputStream resourceAsStream =
                    getClass().getResourceAsStream(mappingProperties);
            helpMapping = new Properties();
            helpMapping.load(resourceAsStream);

        } catch (Exception eee) {
            log.error("Failed to load help mapping file at '" +
                      mappingProperties + "'", eee);
        }
        if (log.isInfoEnabled()) {
            log.info(String.format("Starts help with locale at [%s]",
                                   helpDirectory));
        }

        //--------------------------------------------------------------------//
        // init action UI
        //--------------------------------------------------------------------//
        setActionUI(new TuttiActionUI(null, this));
    }

    public void open() {

        setLocale(config.getI18nLocale());

        if (getProgramId() == null) {

            // load it from config
            setProgramId(config.getProgramId());
        }

        if (getCruiseId() == null) {

            // load it from config
            setCruiseId(config.getCruiseId());
        }

        if (getProtocolId() == null) {

            // load it from config
            setProtocolId(config.getProtocolId());
        }

        boolean dbExists =
                config.getServiceConfig().getPersistenceConfig().isDbExists();

        setDbExist(dbExists);

        if (!dbExists) {

            setProtocolId(null);
            setProgramId(null);
            setCruiseId(null);
            setDbLoaded(false);
        }

        // save back to config
        saveContextToConfig();

        // list when programId or campaingId change to save the configuration
        addPropertyChangeListener(new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {

                if (PROPERTIES_TO_SAVE.contains(evt.getPropertyName())) {
                    saveContextToConfig();
                }
            }
        });
    }

    @Override
    public void close() {

        // Clear data references
        messageNotifiers.clear();
        validationContext = null;
        IOUtils.closeQuietly(dataContext);

        setScreen(null);

        IOUtils.closeQuietly(serviceContext);

        // remove listeners
        PropertyChangeListener[] listeners = getPropertyChangeListeners();
        for (PropertyChangeListener listener : listeners) {
            if (log.isDebugEnabled()) {
                log.debug("Remove listener: " + listener);
            }
            removePropertyChangeListener(listener);
        }
        setMainUI(null);
        if (actionUI != null) {

            // close action ui
            actionUI.getModel().clear();
        }
        setActionUI(null);
    }

    //------------------------------------------------------------------------//
    //-- Service methods                                                    --//
    //------------------------------------------------------------------------//

    public PersistenceService getPersistenceService() {

        PersistenceService service;

        if (useRealPersistenceService()) {
            service = dataContext.getService();
            if (service == null) {

                // use real service
                service = serviceContext.getService(PersistenceService.class);

                dataContext.open(config.getServiceConfig(), service);
            }
        } else {
            service = serviceContext.getService(ClosedPersistenceService.class);
        }
        return service;
    }

    public DecoratorService getDecoratorService() {
        return serviceContext.getService(DecoratorService.class);
    }

    public TuttiReferentialSynchronizeService getTuttiReferentialSynchronizeService() {
        return serviceContext.getService(TuttiReferentialSynchronizeService.class);
    }

    public ProtocolImportExportService getTuttiProtocolImportExportService() {
        return serviceContext.getService(ProtocolImportExportService.class);
    }

    public PupitriImportExportService getTuttiPupitriImportExportService() {
        return serviceContext.getService(PupitriImportExportService.class);
    }

    public ReferentialImportExportService getTuttiReferentialImportExportService() {
        return serviceContext.getService(ReferentialImportExportService.class);
    }

    public WeightComputingService getWeightComputingService() {
        return serviceContext.getService(WeightComputingService.class);
    }

    public WeightCleaningService getWeightCleaningService() {
        return serviceContext.getService(WeightCleaningService.class);
    }

    public ValidateCruiseOperationsService getValidateCruiseOperationsService() {
        return serviceContext.getService(ValidateCruiseOperationsService.class);
    }

    public ValidationService getValidationService() {
        return serviceContext.getService(ValidationService.class);
    }

    public ExportCatchesReportService getGeneratePDFService() {
        return serviceContext.getService(ExportCatchesReportService.class);
    }

    public TuttiExportService getTuttiExportService() {
        return serviceContext.getService(TuttiExportService.class);
    }

    public CatchesSumatraExportService getCatchesSumatraExportService() {
        return serviceContext.getService(CatchesSumatraExportService.class);
    }

    public MultiPostImportService getMultiPostImportService() {
        return serviceContext.getService(MultiPostImportService.class);
    }

    public MultiPostExportService getMultiPostExportService() {
        return serviceContext.getService(MultiPostExportService.class);
    }

    public boolean useRealPersistenceService() {
        return isDbExist() && isDbLoaded();
    }

    public PersistenceService reloadPersistenceService() {

        try {
            serviceContext.close();
        } catch (IOException e) {
            throw new TuttiTechnicalException(_("tutti.context.service.close.error"), e);
        }
        dataContext.close();

        return getPersistenceService();
    }

    public void reloadDecoratorService() {
        serviceContext.reloadService(DecoratorService.class);
    }

    //------------------------------------------------------------------------//
    //-- DataContext methods                                                --//
    //------------------------------------------------------------------------//

    public TuttiDataContext getDataContext() {
        return dataContext;
    }

    public boolean isCruiseFilled() {
        return dataContext.isCruiseFilled();
    }

    public String getProgramId() {
        return dataContext.getProgramId();
    }

    public boolean isProtocolFilled() {
        return dataContext.isProtocolFilled();
    }

    public String getProtocolId() {
        return dataContext.getProtocolId();
    }

    public String getCruiseId() {
        return dataContext.getCruiseId();
    }

    public boolean isProgramFilled() {
        return dataContext.isProgramFilled();
    }

    public void setProgramId(String programId) {
        dataContext.setProgramId(programId);
    }

    public void setCruiseId(String cruiseId) {
        dataContext.setCruiseId(cruiseId);
    }

    public void setProtocolId(String protocolId) {
        dataContext.setProtocolId(protocolId);
    }

    public void clearDbContext() {
        dataContext.clearContext();

//        saveContextToConfig();
    }

    public void checkDbContext() {

        // make sure persistence serivce is loaded
        getPersistenceService();

        // ask data context to check his ids
        dataContext.checkDbContext();

        // save config
        saveContextToConfig();
    }

    //------------------------------------------------------------------------//
    //-- Db methods                                                         --//
    //------------------------------------------------------------------------//

    public boolean isDbExist() {
        return dbExist;
    }

    public void setDbExist(boolean dbExist) {
        this.dbExist = dbExist;
        firePropertyChange(PROPERTY_DB_EXIST, null, dbExist);
    }

    public boolean isDbLoaded() {
        return dbLoaded;
    }

    public void setDbLoaded(boolean dbLoaded) {
        this.dbLoaded = dbLoaded;
        firePropertyChange(PROPERTY_DB_LOADED, null, dbLoaded);
    }

    //------------------------------------------------------------------------//
    //-- Config methods                                                     --//
    //------------------------------------------------------------------------//

    public TuttiApplicationConfig getConfig() {
        return config;
    }

    protected void saveContextToConfig() {
        if (log.isInfoEnabled()) {
            log.info("Save config (programId: " + getProgramId() + ", cruiseId: " +
                     getCruiseId() + ", protocolId: " + getProtocolId() + ", locale: " +
                     getLocale() + ")");
        }
        config.setProgramId(getProgramId());
        config.setCruiseId(getCruiseId());
        config.setProtocolId(getProtocolId());
        config.setI18nLocale(getLocale());
        config.save();
    }

    //------------------------------------------------------------------------//
    //-- UI methods                                                         --//
    //------------------------------------------------------------------------//

    public MainUI getMainUI() {
        return mainUI;
    }

    public void setMainUI(MainUI mainUI) {
        this.mainUI = mainUI;
    }

    public TuttiActionUI getActionUI() {
        return actionUI;
    }

    public void setActionUI(TuttiActionUI actionUI) {
        this.actionUI = actionUI;
    }

    public SwingSession getSwingSession() {
        return swingSession;
    }

    public TuttiScreen getScreen() {
        return screen;
    }

    public void setScreen(TuttiScreen screen) {
        Object oldValue = getScreen();
        this.screen = screen;
        firePropertyChange(PROPERTY_SCREEN, oldValue, screen);
    }

    public boolean isBusy() {
        return busy;
    }

    public void setBusy(boolean busy) {
        this.busy = busy;
        firePropertyChange(PROPERTY_BUSY, null, busy);
    }

    public boolean isHideBody() {
        return hideBody;
    }

    public void setHideBody(boolean hideBody) {
        this.hideBody = hideBody;
        firePropertyChange(PROPERTY_HIDE_BODY, null, hideBody);
    }

    //------------------------------------------------------------------------//
    //-- UIMessageNotifier methods                                          --//
    //------------------------------------------------------------------------//

    public void addMessageNotifier(UIMessageNotifier messageNotifier) {
        this.messageNotifiers.add(messageNotifier);
    }

    public void removeMessageNotifier(UIMessageNotifier messageNotifier) {
        this.messageNotifiers.remove(messageNotifier);
    }

    @Override
    public void showInformationMessage(String message) {
        for (UIMessageNotifier messageNotifier : messageNotifiers) {
            messageNotifier.showInformationMessage(message);
        }
    }

    public RessourceClassLoader getResourceLoader() {
        return resourceLoader;
    }

    //------------------------------------------------------------------------//
    //-- Help methods                                                       --//
    //------------------------------------------------------------------------//

    public TuttiHelpBroker getHelpBroker() {
        return helpBroker;
    }

    public void setHelpBroker(TuttiHelpBroker helpBroker) {
        this.helpBroker = helpBroker;
    }

    @Override
    public void showHelp(JAXXContext context,
                         JAXXHelpBroker broker,
                         String helpId) {
        if (helpId == null) {
            helpId = broker.getDefaultID();
        }

        if (log.isInfoEnabled()) {
            log.info("show help " + helpId);
        }

        String value = (String) helpMapping.get(helpId);

        if (value == null) {
            throw new TuttiTechnicalException(_("tutti.context.helpPage.notFound", helpId));
        }

        String helpDirectory = getConfig().getHelpResourceWithLocale(value);
        boolean withFragment = helpDirectory.contains("#");

        String fragment = null;
        if (withFragment) {
            helpDirectory = StringUtils.substringBefore(helpDirectory, "#");
            fragment = StringUtils.substringAfter(helpDirectory, "#");
        }

        URI resolvedUri = new File(helpDirectory).toURI();
        try {

            if (withFragment) {
                resolvedUri = new URI(resolvedUri.toString() + "#" + fragment);
            }
            if (log.isInfoEnabled()) {
                log.info("help uri = " + resolvedUri);
            }
            TuttiUIUtil.openLink(resolvedUri);
        } catch (URISyntaxException e) {
            throw new TuttiTechnicalException(_("tutti.context.helpPage.notFound", resolvedUri));
        }

    }

    //------------------------------------------------------------------------//
    //-- Other methods                                                      --//
    //------------------------------------------------------------------------//

    public Locale getLocale() {
        return locale;
    }

    public void setLocale(Locale locale) {
        this.locale = locale;
        firePropertyChange(PROPERTY_LOCALE, null, locale);
    }

    public void setValidationContext(String validationContext) {
        Object oldValue = getValidationContext();
        this.validationContext = validationContext;
        firePropertyChange(PROPERTY_VALIDATION_CONTEXT, oldValue, validationContext);
    }

    public String getValidationContext() {
        return validationContext;
    }

    @Override
    public void firePropertyChanged(String propertyName,
                                    Object oldValue,
                                    Object newValue) {
        firePropertyChange(propertyName, oldValue, newValue);
    }

    public void setFallBackScreen() {
        if (isDbLoaded()) {
            setScreen(TuttiScreen.SELECT_CRUISE);
        } else {
            setScreen(TuttiScreen.MANAGE_DB);
        }
    }

    public boolean checkUpdateApplicationReachable() {

        boolean result = true;
        TuttiApplicationConfig config = getConfig();
        String url = config.getUpdateApplicationUrl();

        try {
            TuttiUIUtil.tryToConnectToUpdateUrl(
                    url,
                    n_("tutti.error.update.bad.url.syntax"),
                    n_("tutti.error.update.could.not.reach.url"),
                    n_("tutti.error.update.could.not.found.url")
            );
        } catch (TuttiBusinessException e) {
            TuttiUIContext.getErrorHelper().showWarningDialog(e.getMessage());
            result = false;
        }
        return result;
    }

    public boolean checkUpdateDbReachable() {

        boolean result = true;
        TuttiApplicationConfig config = getConfig();
        String url = config.getUpdateDbUrl();

        try {
            TuttiUIUtil.tryToConnectToUpdateUrl(
                    url,
                    n_("tutti.error.update.bad.url.syntax"),
                    n_("tutti.error.update.could.not.reach.url"),
                    n_("tutti.error.update.could.not.found.url")
            );
        } catch (TuttiBusinessException e) {
            TuttiUIContext.getErrorHelper().showWarningDialog(e.getMessage());
            result = false;
        }
        return result;
    }
}
