package fr.ifremer.tutti.ui.swing.content.operation.catches.individualobservation.create;

/*
 * #%L
 * Tutti :: UI
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Lists;
import fr.ifremer.tutti.persistence.entities.CaracteristicMap;
import fr.ifremer.tutti.persistence.entities.TuttiBeanFactory;
import fr.ifremer.tutti.persistence.entities.data.IndividualObservationBatch;
import fr.ifremer.tutti.persistence.entities.referential.Caracteristic;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiBeanUIModel;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.beans.AbstractSerializableBean;

import java.io.Serializable;
import java.util.List;

/**
 * @author kmorin <kmorin@codelutin.com>
 * @since 1.4
 */
public class CreateIndividualObservationBatchUIModel extends AbstractTuttiBeanUIModel<IndividualObservationBatch, CreateIndividualObservationBatchUIModel> {

    private static final long serialVersionUID = 1L;

    private static final Log log = LogFactory.getLog(CreateIndividualObservationBatchUIModel.class);

    public static final String PROPERTY_AVAILABLE_SPECIES = "availableSpecies";

    public static final String PROPERTY_CARACTERISTICS = "caracteristics";

    /**
     * All available species (sets by protocol).
     *
     * @since 2.0
     */
    protected List<Species> availableSpecies;

    /**
     * Delegate edit object.
     *
     * @since 1.3
     */
    protected final IndividualObservationBatch editObject =
            TuttiBeanFactory.newIndividualObservationBatch();

    /**
     * Default caracteristics coming from protocol.
     *
     * @since 2.5
     */
    protected final List<Caracteristic> defaultCaracteristic;

    /**
     * Default caracteristics editors.
     *
     * @since 2.5
     */
    protected final List<CaracteristicMapPropertyEditor> caracteristicEditors =
            Lists.newArrayList();

    public CreateIndividualObservationBatchUIModel(List<Caracteristic> defaultCaracteristic) {
        super(IndividualObservationBatch.class, null, null);
        this.defaultCaracteristic = defaultCaracteristic;
        editObject.setCaracteristics(new CaracteristicMap());
    }

    public Species getSpecies() {
        return editObject.getSpecies();
    }

    public void setSpecies(Species individualObservationSpecies) {
        Object oldValue = getSpecies();
        editObject.setSpecies(individualObservationSpecies);
        firePropertyChange(IndividualObservationBatch.PROPERTY_SPECIES, oldValue, individualObservationSpecies);
    }

    public Float getWeight() {
        return editObject.getWeight();
    }

    public void setWeight(Float individualObservationWeight) {
        Object oldValue = getWeight();
        editObject.setWeight(individualObservationWeight);
        firePropertyChange(IndividualObservationBatch.PROPERTY_WEIGHT, oldValue, individualObservationWeight);
    }

    public Float getSize() {
        return editObject.getSize();
    }

    public void setSize(Float individualObservationSize) {
        Object oldValue = getSize();
        editObject.setSize(individualObservationSize);
        firePropertyChange(IndividualObservationBatch.PROPERTY_SIZE, oldValue, individualObservationSize);
    }

    public Caracteristic getLengthStepCaracteristic() {
        return editObject.getLengthStepCaracteristic();
    }

    public void setLengthStepCaracteristic(Caracteristic individualObservationLengthStepCaracteristic) {
        Object oldValue = getLengthStepCaracteristic();
        editObject.setLengthStepCaracteristic(individualObservationLengthStepCaracteristic);
        firePropertyChange(IndividualObservationBatch.PROPERTY_LENGTH_STEP_CARACTERISTIC, oldValue, individualObservationLengthStepCaracteristic);
    }

    public CaracteristicMap getCaracteristics() {
        return editObject.getCaracteristics();
    }

    public void setCaracteristics(CaracteristicMap caracteristics) {

        editObject.setCaracteristics(caracteristics);
        firePropertyChange(IndividualObservationBatch.PROPERTY_CARACTERISTICS, null, caracteristics);

    }

    public List<Species> getAvailableSpecies() {
        if (log.isDebugEnabled()) {
            log.debug("getAvailableSpecies " + availableSpecies);
        }
        return availableSpecies;
    }

    public void setAvailableSpecies(List<Species> availableSpecies) {

        if (log.isDebugEnabled()) {
            log.debug("setAvailableSpecies " + availableSpecies);
        }
        this.availableSpecies = availableSpecies;
        firePropertyChange(PROPERTY_AVAILABLE_SPECIES, null, availableSpecies);
    }

    @Override
    protected IndividualObservationBatch newEntity() {
        return TuttiBeanFactory.newIndividualObservationBatch();
    }

    public void reset() {
        setAvailableSpecies(Lists.<Species>newArrayList());
        setSpecies(null);
        setWeight(null);
        setSize(null);
        setLengthStepCaracteristic(null);
        for (CaracteristicMapPropertyEditor editor : caracteristicEditors) {
            editor.setProperty(null);
        }
        getCaracteristics().clear();
    }

    public CaracteristicMapPropertyEditor newCaracteristicEditor(Caracteristic caracteristic) {
        CaracteristicMapPropertyEditor editor = new CaracteristicMapPropertyEditor(caracteristic);
        caracteristicEditors.add(editor);
        return editor;
    }

    /**
     * Model to edit a caracteristic and flush it in the caracteristc map of
     * the global model.
     * <p/>
     * Acts as a bean for the given caracteristic.
     *
     * @since 2.5
     */
    public class CaracteristicMapPropertyEditor extends AbstractSerializableBean {

        private static final long serialVersionUID = 1L;

        public static final String PROPERTY_PROPERTY = "property";

        private final Caracteristic caracteristic;

        protected Serializable property;

        public Serializable getProperty() {
            return property;
        }

        public void setProperty(Serializable property) {
            Object oldValue = getProperty();
            this.property = property;
            getCaracteristics().put(caracteristic, property);
            if (ObjectUtils.notEqual(oldValue, property)) {
                firePropertyChange(PROPERTY_PROPERTY, oldValue, property);
                CreateIndividualObservationBatchUIModel.this.firePropertyChanged(PROPERTY_CARACTERISTICS, null, getCaracteristics());
            }
        }

        public CaracteristicMapPropertyEditor(Caracteristic caracteristic) {
            this.caracteristic = caracteristic;
        }
    }
}
