package fr.ifremer.tutti.ui.swing.content.operation;

/*
 * #%L
 * Tutti :: UI
 * $Id: EditFishingOperationAction.java 935 2013-05-11 15:53:26Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-2.2/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/EditFishingOperationAction.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Lists;
import fr.ifremer.tutti.persistence.InvalidBatchModelException;
import fr.ifremer.tutti.persistence.entities.TuttiBeanFactory;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.data.Attachment;
import fr.ifremer.tutti.persistence.entities.data.CatchBatch;
import fr.ifremer.tutti.persistence.entities.data.Cruise;
import fr.ifremer.tutti.persistence.entities.data.FishingOperation;
import fr.ifremer.tutti.persistence.entities.referential.Person;
import fr.ifremer.tutti.persistence.entities.referential.TuttiLocation;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.ui.swing.content.operation.catches.ComputeWeightsAction;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUIModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.SaveCatchBatchAction;
import fr.ifremer.tutti.ui.swing.content.operation.fishing.GearUseFeatureTabUI;
import fr.ifremer.tutti.ui.swing.content.operation.fishing.VesselUseFeatureTabUI;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import fr.ifremer.tutti.ui.swing.util.action.AbstractTuttiAction;
import fr.ifremer.tutti.ui.swing.util.action.TuttiActionHelper;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.decorator.Decorator;

import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JTabbedPane;
import java.awt.BorderLayout;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Collections;
import java.util.List;

import static org.nuiton.i18n.I18n._;

/**
 * To edit the given fishing operation.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.0
 */
public class EditFishingOperationAction extends AbstractTuttiAction<FishingOperationsUIModel, FishingOperationsUI, FishingOperationsUIHandler> {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(EditFishingOperationAction.class);

    /**
     * The incoming fishing operation to edit.
     * <p/>
     * Can be null (means do not edit any fishing operation), or with no id
     * (means create a ne fishing operation), or with an id (means edit an
     * existing fishing operation).
     *
     * @since 1.0
     */
    protected FishingOperation fishingOperation;

    /**
     * A flag to not check if there is a previous edit.
     * <p/>
     * This flag is used when we launch the cancel action.
     *
     * @since 1.0
     */
    protected boolean checkPreviousEdit = true;

    /**
     * Delegate action to save Fising Operation.
     *
     * @since 1.0
     */
    protected SaveFishingOperationAction saveFishingOperationAction;

    /**
     * Delegate action to compute the weight of the operations.
     *
     * @since 1.1
     */
    protected ComputeWeightsAction computeWeightsAction;

    /**
     * Delegate action to save catch batch.
     *
     * @since 1.0
     */
    protected SaveCatchBatchAction saveCatchBatchAction;

    private final PropertyChangeListener coordinatePropertiesListener = new PropertyChangeListener() {

        private List<String> properties = Lists.newArrayList(
                EditFishingOperationUIModel.PROPERTY_FISHING_OPERATION_RECTILIGNE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_END_LATITUDE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_END_LATITUDE_DECIMAL_MINUTE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_END_LATITUDE_DEGREE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_END_LATITUDE_MINUTE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_END_LATITUDE_SECOND,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_END_LONGITUDE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_END_LONGITUDE_DECIMAL_MINUTE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_END_LONGITUDE_DEGREE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_END_LONGITUDE_MINUTE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_END_LONGITUDE_SECOND,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_START_LATITUDE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_START_LATITUDE_DECIMAL_MINUTE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_START_LATITUDE_DEGREE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_START_LATITUDE_MINUTE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_START_LATITUDE_SECOND,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_START_LONGITUDE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_START_LONGITUDE_DECIMAL_MINUTE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_START_LONGITUDE_DEGREE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_START_LONGITUDE_MINUTE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_START_LONGITUDE_SECOND
        );

        @Override
        public void propertyChange(PropertyChangeEvent evt) {
            if (properties.contains(evt.getPropertyName())) {
                EditFishingOperationUIModel source = (EditFishingOperationUIModel) evt.getSource();
                if (source.isFishingOperationRectiligne()) {
                    source.computeDistance();
                }
            }
        }
    };

    public EditFishingOperationAction(FishingOperationsUIHandler handler) {
        super(handler, true);
        setActionDescription(_("tutti.editFishingOperation.action.editFishingOperation.tip"));
    }

    public void setFishingOperation(FishingOperation fishingOperation) {
        this.fishingOperation = fishingOperation;
    }

    public void setCheckPreviousEdit(boolean checkPreviousEdit) {
        this.checkPreviousEdit = checkPreviousEdit;
    }

    @Override
    protected void releaseAction() {
        fishingOperation = null;
        checkPreviousEdit = true;
        super.releaseAction();
    }

    protected SaveFishingOperationAction getSaveFishingOperationAction() {
        if (saveFishingOperationAction == null) {
            saveFishingOperationAction = new SaveFishingOperationAction(getUI().getFishingOperationTabContent().getHandler());
        }
        return saveFishingOperationAction;
    }

    protected SaveCatchBatchAction getSaveCatchBatchAction() {
        if (saveCatchBatchAction == null) {
            saveCatchBatchAction = new SaveCatchBatchAction(getUI().getCatchesTabContent().getHandler());
        }
        return saveCatchBatchAction;
    }

    protected ComputeWeightsAction getComputeWeightsAction() {
        if (computeWeightsAction == null) {
            computeWeightsAction = new ComputeWeightsAction(getUI().getCatchesTabContent().getHandler());
        }
        return computeWeightsAction;
    }

    @Override
    protected boolean prepareAction() {

        boolean canContinue = true;
        if (checkPreviousEdit) {
            FishingOperationsUI ui = getUI();

            final FishingOperationsUIModel model = ui.getModel();

            FishingOperation editFishingOperation = model.getEditFishingOperation();

            String editFishingOperationId = null;

            if (editFishingOperation == null) {

                // no previous fishing operation in edition, can continue
                canContinue = true;
            } else {

                editFishingOperationId = editFishingOperation.getId();

                boolean create = TuttiEntities.isNew(editFishingOperation);

                boolean fishingOperationModified = getHandler().isFishingOperationModified();
                boolean catchBatchModified = getHandler().isCatchBatchModified();

                boolean fishingOperationValid = getHandler().isFishingOperationValid();
                boolean catchBatchValid = getHandler().isCatchBatchValid();

                boolean needSave = create || fishingOperationModified || catchBatchModified;


                if (needSave) {

                    boolean canSave = fishingOperationValid && catchBatchValid;

                    canContinue = false;

                    if (canSave) {

                        String message;

                        if (create) {
                            message = _("tutti.editFishingOperation.askSaveBeforeLeaving.createFishingOperation");
                        } else if (fishingOperationModified) {
                            message = _("tutti.editFishingOperation.askSaveBeforeLeaving.saveFishingOperation");
                        } else {
                            message = _("tutti.editCatchBatch.askSaveBeforeLeaving.saveCatchBatch");
                        }

                        int answer = getHandler().askSaveBeforeLeaving(message);


                        switch (answer) {
                            case JOptionPane.OK_OPTION:

                                // persist previous fishing operation
                                if (fishingOperationModified) {
                                    getSaveFishingOperationAction().setUpdateUI(false);
                                    TuttiActionHelper.runInternalAction(getSaveFishingOperationAction());
                                }

                                if (catchBatchModified) {
                                    getSaveCatchBatchAction().setUpdateUI(false);
                                    TuttiActionHelper.runInternalAction(getSaveCatchBatchAction());
                                }

                                canContinue = true;
                                break;

                            case JOptionPane.NO_OPTION:

                                // won't save modification
                                // so since we will edit a new operation, nothing to do here

                                canContinue = true;
                                break;
                        }
                    } else {

                        // data are not valid
                        String message;

                        if (fishingOperationValid) {
                            message = _("tutti.editCatchBatch.askCancelEditBeforeLeaving.cancelEditCatchBatch");
                        } else {
                            message = _("tutti.editFishingOperation.askCancelEditBeforeLeaving.cancelEditFishingOperation");
                        }

                        // ok will revert any modification by
                        // editing new fishing operation (if user says yes)
                        canContinue = handler.askCancelEditBeforeLeaving(message);

                    }
                }
            }

            if (!canContinue) {

                FishingOperation selectFishingOperation;
                if (TuttiEntities.isNew(editFishingOperation)) {

                    // no selection
                    selectFishingOperation = null;
                } else {
                    // rollback selected fishing operation
                    selectFishingOperation =
                            model.getFishingOperation(editFishingOperationId);
                }

                model.setEditionAdjusting(true);
                try {
                    model.setSelectedFishingOperation(selectFishingOperation);
                } finally {
                    model.setEditionAdjusting(false);
                }
            }
        }

        return canContinue;
    }

    @Override
    protected void doAction() throws Exception {

        if (log.isInfoEnabled()) {
            log.info("Try to edit fishingOperation: " + fishingOperation);
        }

        FishingOperationsUI ui = getUI();

        FishingOperationsUIModel model = ui.getModel();

        // edit new fishing operation
        if (log.isInfoEnabled()) {
            log.info("Edit in ui fishingOperation: " + fishingOperation);
        }

        // now fishing operation is edited
        model.setEditFishingOperation(fishingOperation);

        String fishingOperationText = getFishingOperationTitle(fishingOperation);

        loadFishingOperation(fishingOperation, fishingOperationText);

        loadCatchBatch(fishingOperation, fishingOperationText, true);

        JTabbedPane form = ui.getTabPane();
        JLabel noContentPane = ui.getNoTraitPane();

        if (fishingOperation == null) {

            // nothing to display

            ui.remove(form);

            // just display <no trait!>
            ui.add(noContentPane, BorderLayout.CENTER);

        } else {

            ui.remove(noContentPane);

            // wait last minute to display (avoid dirty display effects)
            ui.add(form, BorderLayout.CENTER);

            ui.getFishingOperationTabContent().getFishingOperationTabPane().setSelectedIndex(0);

            if (checkPreviousEdit) {
                ui.getTabPane().setSelectedIndex(0);
            }
        }

        model.addPropertyChangeListener(coordinatePropertiesListener);
    }

    @Override
    public void postSuccessAction() {
        super.postSuccessAction();
        getUI().getFishingOperationTabContent().getModel().setModify(false);
        getUI().repaint();
    }

    public void loadFishingOperation(FishingOperation bean,
                                     String fishingOperationText) {

        EditFishingOperationUI ui = getUI().getFishingOperationTabContent();

        EditFishingOperationUIHandler handler = ui.getHandler();

        EditFishingOperationUIModel editFishingOperationUIModel = ui.getModel();

        editFishingOperationUIModel.setLoadingData(true);

        editFishingOperationUIModel.removeAllAttachment(editFishingOperationUIModel.getAttachment());

        handler.uninstallStartDateListener();
        handler.uninstallCoordinatesListener();

        if (bean == null) {

            editFishingOperationUIModel.fromBean(
                    TuttiBeanFactory.newFishingOperation());

            editFishingOperationUIModel.setFishingOperation(bean);

            handler.clearValidators();

            handler.resetAllModels();

        } else { //if (!bean.equals(editFishingOperationUIModel.getFishingOperation()) || handler.isAModelModified()) {

            TuttiLocation strata = bean.getStrata();
            TuttiLocation subStrata = bean.getSubStrata();
            TuttiLocation location = bean.getLocation();

            Cruise cruise = bean.getCruise();
            if (cruise != null) {

                // update gear universe
                ui.getGearComboBox().setData(Lists.newArrayList(cruise.getGear()));
            }

            editFishingOperationUIModel.fromBean(bean);

            // to be sure combo list will be reloaded
            editFishingOperationUIModel.setStrata(null);
            editFishingOperationUIModel.setSubStrata(null);
            editFishingOperationUIModel.setLocation(null);
            editFishingOperationUIModel.convertGearShootingCoordinatesDDToDMS();

            if (strata != null) {
                ui.getStrataComboBox().setSelectedItem(strata);
            }

            if (subStrata != null) {
                ui.getSubStrataComboBox().setSelectedItem(subStrata);
            }

            if (location != null) {
                ui.getLocationComboBox().setSelectedItem(location);
            }

            editFishingOperationUIModel.setFishingOperation(bean);

            // update saisissuer selection
            List<Person> saisisseur = editFishingOperationUIModel.getRecorderPerson();
            ui.getRecorderPersonList().getHandler().setSelected(saisisseur);

            //reset gear use feature tab
            GearUseFeatureTabUI gearUseFeatureTabContent = ui.getGearUseFeatureTabContent();
            gearUseFeatureTabContent.getHandler().reset(bean);

            //reset vessel use feature tab
            VesselUseFeatureTabUI vesselUseFeatureTabContent = ui.getVesselUseFeatureTabContent();
            vesselUseFeatureTabContent.getHandler().reset(bean);

            Integer objectId = editFishingOperationUIModel.getObjectId();
            if (objectId != null) {

                List<Attachment> attachments =
                        getContext().getPersistenceService().getAllAttachments(
                                editFishingOperationUIModel.getObjectType(),
                                objectId);

                editFishingOperationUIModel.addAllAttachment(attachments);
            }

            editFishingOperationUIModel.setModify(false);
            handler.getFishingOperationMonitor().clearModified();
            handler.registerValidator();
        }

        editFishingOperationUIModel.setLoadingData(false);

        ui.getTraitGeneralTabPane().setTitle(fishingOperationText);
        ui.getVesselUseFeatureTabPane().setTitle(fishingOperationText);
        ui.getGearUseFeatureTabPane().setTitle(fishingOperationText);

        handler.installStartDateListener();
        handler.installCoordinatesListener();
    }

    public String getFishingOperationTitle(FishingOperation bean) {
        Decorator<FishingOperation> decorator =
                getDecorator(FishingOperation.class, null);

        String fishingOperationText;

        if (bean == null) {
            fishingOperationText = null;
        } else if (TuttiEntities.isNew(bean)) {
            fishingOperationText = _("tutti.editFishingOperation.label.traitReminder",
                                     _("tutti.editFishingOperation.label.traitReminder.inCreation"));
        } else {
            fishingOperationText = _("tutti.editFishingOperation.label.traitReminder",
                                     decorator.toString(bean));
        }
        return fishingOperationText;
    }

    public void loadCatchBatch(FishingOperation bean,
                               String fishingOperationText,
                               boolean loadOtherTabs) {

        boolean empty = bean == null || TuttiEntities.isNew(bean);

        EditCatchesUI ui = getUI().getCatchesTabContent();
        TuttiBeanMonitor<EditCatchesUIModel> catchBatchMonitor =
                ui.getHandler().getCatchBatchMonitor();

        EditCatchesUIModel catchesUIModel = ui.getModel();

        catchesUIModel.setLoadingData(true);

        catchesUIModel.reset();

        CatchBatch batch;

        List<Attachment> attachments;

        if (empty) {

            // create a new CatchBatch
            if (log.isInfoEnabled()) {
                log.info("Create a new CatchBatch (fishing operation is null)");
            }
            batch = TuttiBeanFactory.newCatchBatch();
            batch.setFishingOperation(bean);

            attachments = Collections.emptyList();

            getModel().setCatchEnabled(true);

        } else {

            String operationId = bean.getId();

            if (log.isInfoEnabled()) {
                log.info("Load existing CatchBatch from operation id: " +
                         operationId);
            }

            PersistenceService persistenceService =
                    getContext().getPersistenceService();

            boolean withCatchBath =
                    persistenceService.isFishingOperationWithCatchBatch(
                            operationId);

            if (withCatchBath) {

                // load it
                try {
                    batch = persistenceService.getCatchBatchFromFishingOperation(operationId);
                    batch.setFishingOperation(bean);
                    Integer objectId = Integer.valueOf(batch.getId());
                    attachments = persistenceService.getAllAttachments(catchesUIModel.getObjectType(), objectId);
                    getModel().setCatchNotFound(false);
                    getModel().setCatchEnabled(true);

                } catch (InvalidBatchModelException e) {

                    // batch is not compatible with Tutti
                    if (log.isDebugEnabled()) {
                        log.debug("Invalid batch model", e);
                    }
                    batch = null;
                    attachments = Collections.emptyList();

                    getModel().setCatchEnabled(false);
                }
            } else {

                // no catch batch
                batch = null;
                attachments = Collections.emptyList();

                getModel().setCatchEnabled(false);
                getModel().setCatchNotFound(true);
            }
        }

        catchesUIModel.setCatchBatch(batch);
        catchesUIModel.setFishingOperation(bean);
        catchesUIModel.fromBean(batch);
        catchesUIModel.addAllAttachment(attachments);
        catchesUIModel.setModify(false);
        catchBatchMonitor.clearModified();

        // 3) Propagate title to others tabs
        ui.getCatchesCaracteristicsTabPane().setTitle(fishingOperationText);
        ui.getSpeciesTabFishingOperationReminderLabel().setTitle(fishingOperationText);
        ui.getBenthosTabFishingOperationReminderLabel().setTitle(fishingOperationText);
        ui.getMarineLitterTabFishingOperationReminderLabel().setTitle(fishingOperationText);
        ui.getAccidentalTabFishingOperationReminderLabel().setTitle(fishingOperationText);
        ui.getIndividualObservationTabFishingOperationReminderLabel().setTitle(fishingOperationText);

        if (loadOtherTabs) {

            // 4) Propagate new selected fishingoperation to others tabs

            FishingOperation operationToLoad = batch == null ? null : bean;

            ui.getSpeciesTabContent().getHandler().selectFishingOperation(operationToLoad);
            ui.getBenthosTabContent().getHandler().selectFishingOperation(operationToLoad);
            ui.getMarineLitterTabContent().getHandler().selectFishingOperation(operationToLoad);
            ui.getAccidentalTabContent().getHandler().selectFishingOperation(operationToLoad);
            ui.getIndividualObservationTabContent().getHandler().selectFishingOperation(operationToLoad);
        }

        catchesUIModel.setLoadingData(false);
    }

}
