package fr.ifremer.tutti.ui.swing.content.operation.catches.species;

/*
 * #%L
 * Tutti :: UI
 * $Id: SpeciesBatchRowModel.java 838 2013-04-20 14:33:56Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-2.1/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/catches/species/SpeciesBatchRowModel.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Lists;
import fr.ifremer.tutti.persistence.entities.TuttiBeanFactory;
import fr.ifremer.tutti.persistence.entities.data.AttachementObjectTypeEnum;
import fr.ifremer.tutti.persistence.entities.data.Attachment;
import fr.ifremer.tutti.persistence.entities.data.FishingOperation;
import fr.ifremer.tutti.persistence.entities.data.SampleCategory;
import fr.ifremer.tutti.persistence.entities.data.SampleCategoryEnum;
import fr.ifremer.tutti.persistence.entities.data.SpeciesBatch;
import fr.ifremer.tutti.persistence.entities.data.SpeciesBatchFrequency;
import fr.ifremer.tutti.persistence.entities.referential.CaracteristicQualitativeValue;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.frequency.SpeciesFrequencyRowModel;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiBeanUIModel;
import fr.ifremer.tutti.ui.swing.util.TuttiComputedOrNotData;
import fr.ifremer.tutti.ui.swing.util.attachment.AttachmentModelAware;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.io.Serializable;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;

/**
 * Represents a species batch (i.e a row in the batch table).
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.2
 */
public class SpeciesBatchRowModel extends AbstractTuttiBeanUIModel<SpeciesBatch, SpeciesBatchRowModel> implements SpeciesBatch, AttachmentModelAware {

    private static final long serialVersionUID = 1L;

    public static final String PROPERTY_SORTED_UNSORTED_CATEGORY = "sortedUnsortedCategory";

    public static final String PROPERTY_SORTED_UNSORTED_CATEGORY_VALUE = "sortedUnsortedCategoryValue";

    public static final String PROPERTY_SORTED_UNSORTED_CATEGORY_WEIGHT = "sortedUnsortedCategoryWeight";

    public static final String PROPERTY_SORTED_UNSORTED_CATEGORY_COMPUTED_WEIGHT = "sortedUnsortedCategoryComputedWeight";

    public static final String PROPERTY_SIZE_CATEGORY = "sizeCategory";

    public static final String PROPERTY_SIZE_CATEGORY_VALUE = "sizeCategoryValue";

    public static final String PROPERTY_SIZE_CATEGORY_WEIGHT = "sizeCategoryWeight";

    public static final String PROPERTY_SIZE_CATEGORY_COMPUTED_WEIGHT = "sexCategoryComputedWeight";

    public static final String PROPERTY_SEX_CATEGORY = "sexCategory";

    public static final String PROPERTY_SEX_CATEGORY_VALUE = "sexCategoryValue";

    public static final String PROPERTY_SEX_CATEGORY_WEIGHT = "sexCategoryWeight";

    public static final String PROPERTY_SEX_CATEGORY_COMPUTED_WEIGHT = "sexCategoryComputedWeight";

    public static final String PROPERTY_MATURITY_CATEGORY = "maturityCategory";

    public static final String PROPERTY_MATURITY_CATEGORY_VALUE = "maturityCategoryValue";

    public static final String PROPERTY_MATURITY_CATEGORY_COMPUTED_WEIGHT = "maturityCategoryComputedWeight";

    public static final String PROPERTY_MATURITY_CATEGORY_WEIGHT = "maturityCategoryWeight";

    public static final String PROPERTY_AGE_CATEGORY = "ageCategory";

    public static final String PROPERTY_AGE_CATEGORY_VALUE = "ageCategoryValue";

    public static final String PROPERTY_AGE_CATEGORY_COMPUTED_WEIGHT = "ageCategoryComputedWeight";

    public static final String PROPERTY_AGE_CATEGORY_WEIGHT = "ageCategoryWeight";

    public static final String PROPERTY_FREQUENCY = "frequency";

    public static final String PROPERTY_COMPUTED_NUMBER = "computedOrNotNumber";

    public static final String PROPERTY_COMPUTED_WEIGHT = "computedOrNotWeight";

    public static final String PROPERTY_CHILD_BATCH = "childBatch";

    public static final String PROPERTY_BATCH_LEAF = "batchLeaf";

    public static final String PROPERTY_BATCH_ROOT = "batchRoot";

    public static final String PROPERTY_SAMPLE_CATEGORY = "sampleCategory";

    /**
     * Delegate edit object.
     *
     * @since 1.3
     */
    protected final SpeciesBatch editObject = TuttiBeanFactory.newSpeciesBatch();

    /**
     * Sorted - Unsorted category (can not be null).
     *
     * @since 0.2
     */
    protected SampleCategory<CaracteristicQualitativeValue> sortedUnsortedCategory;

    /**
     * Weight category (can be null).
     *
     * @since 0.2
     */
    protected SampleCategory<CaracteristicQualitativeValue> sizeCategory;

    /**
     * Sex category (can be null).
     *
     * @since 0.2
     */
    protected SampleCategory<CaracteristicQualitativeValue> sexCategory;

    /**
     * Maturity category (can be null).
     *
     * @since 0.2
     */
    protected SampleCategory<CaracteristicQualitativeValue> maturityCategory;

    /**
     * Age category (can be null).
     *
     * @since 0.2
     */
    protected SampleCategory<Float> ageCategory;

    /**
     * Sample category of this batch.
     * <p/>
     * This is one of the sample category defined above, used to know if the
     * table row which sample category affects this row exactly.
     *
     * @since 0.3
     */
    private SampleCategory<?> sampleCategory;

    /**
     * Observed weight.
     *
     * @since 0.2
     */
    protected TuttiComputedOrNotData<Float> computedOrNotWeight =
            new TuttiComputedOrNotData<Float>();

    /**
     * Total computed number (from frequencies).
     *
     * @since 0.2
     */
    protected TuttiComputedOrNotData<Integer> computedOrNotNumber =
            new TuttiComputedOrNotData<Integer>();

    /**
     * Attachments (should never be null).
     *
     * @since 0.2
     */
    protected final List<Attachment> attachment = Lists.newArrayList();

    /**
     * List of frequencies observed for this batch.
     *
     * @since 0.2
     */
    protected List<SpeciesFrequencyRowModel> frequency = Lists.newArrayList();

    /**
     * List of child batches (can be null or empty if batch is a leaf).
     *
     * @see #isBatchLeaf()
     * @since 0.3
     */
    protected List<SpeciesBatchRowModel> childBatch;

    protected static final Binder<SpeciesBatch, SpeciesBatchRowModel> fromBeanBinder =
            BinderFactory.newBinder(SpeciesBatch.class,
                                    SpeciesBatchRowModel.class);

    protected static final Binder<SpeciesBatchRowModel, SpeciesBatch> toBeanBinder =
            BinderFactory.newBinder(SpeciesBatchRowModel.class,
                                    SpeciesBatch.class);

    public SpeciesBatchRowModel() {
        super(SpeciesBatch.class, fromBeanBinder, toBeanBinder);
        sortedUnsortedCategory = SampleCategory.newSample(SampleCategoryEnum.sortedUnsorted);
        sizeCategory = SampleCategory.newSample(SampleCategoryEnum.size);
        sexCategory = SampleCategory.newSample(SampleCategoryEnum.sex);
        maturityCategory = SampleCategory.newSample(SampleCategoryEnum.maturity);
        ageCategory = SampleCategory.newSample(SampleCategoryEnum.age);

        computedOrNotWeight.addPropagateListener(PROPERTY_WEIGHT, this);
        computedOrNotWeight.addPropagateListener(PROPERTY_COMPUTED_WEIGHT, this);
        computedOrNotNumber.addPropagateListener(PROPERTY_NUMBER, this);
        computedOrNotNumber.addPropagateListener(PROPERTY_COMPUTED_NUMBER, this);
    }

    public SpeciesBatchRowModel(SpeciesBatch aBatch,
                                List<SpeciesBatchFrequency> frequencies) {
        this();
        fromBean(aBatch);
        List<SpeciesFrequencyRowModel> frequencyRows =
                SpeciesFrequencyRowModel.fromBeans(frequencies);
        frequency.addAll(frequencyRows);
        Collections.sort(frequency);
    }

    @Override
    protected SpeciesBatch newEntity() {
        return TuttiBeanFactory.newSpeciesBatch();
    }

    //------------------------------------------------------------------------//
    //-- SpeciesBatch                                                       --//
    //------------------------------------------------------------------------//

    @Override
    public Species getSpecies() {
        return editObject.getSpecies();
    }

    @Override
    public void setSpecies(Species species) {
        Object oldCategory = getSpecies();
        editObject.setSpecies(species);
        firePropertyChange(PROPERTY_SPECIES, oldCategory, species);
    }

    @Override
    public String getComment() {
        return editObject.getComment();
    }

    @Override
    public void setComment(String comment) {
        Object oldValue = getComment();
        editObject.setComment(comment);
        firePropertyChange(PROPERTY_COMMENT, oldValue, comment);
    }

    @Override
    public boolean isSpeciesToConfirm() {
        return editObject.isSpeciesToConfirm();
    }

    @Override
    public void setSpeciesToConfirm(boolean speciesToConfirm) {
        Object oldValue = isSpeciesToConfirm();
        editObject.setSpeciesToConfirm(speciesToConfirm);
        firePropertyChange(PROPERTY_SPECIES_TO_CONFIRM, oldValue, speciesToConfirm);
    }

    @Override
    public Integer getComputedNumber() {
        return computedOrNotNumber.getComputedData();
    }

    @Override
    public void setComputedNumber(Integer computedNumber) {
        computedOrNotNumber.setComputedData(computedNumber);
    }

    @Override
    public Float getComputedWeight() {
        return computedOrNotWeight.getComputedData();
    }

    @Override
    public void setComputedWeight(Float computedWeight) {
        computedOrNotWeight.setComputedData(computedWeight);
    }

    @Override
    public FishingOperation getFishingOperation() {
        return editObject.getFishingOperation();
    }

    @Override
    public void setFishingOperation(FishingOperation fishingOperation) {
        editObject.setFishingOperation(fishingOperation);
    }

    @Override
    public SpeciesBatchRowModel getParentBatch() {
        return (SpeciesBatchRowModel) editObject.getParentBatch();
    }

    @Override
    public void setParentBatch(SpeciesBatch parentBatch) {
        Object oldValue = getParentBatch();
        editObject.setParentBatch(parentBatch);
        firePropertyChange(PROPERTY_PARENT_BATCH, oldValue, parentBatch);
        firePropertyChange(PROPERTY_BATCH_ROOT, null, isBatchRoot());
    }

    @Override
    public Float getWeight() {
        return computedOrNotWeight.getData();
    }

    @Override
    public void setWeight(Float weight) {
        this.computedOrNotWeight.setData(weight);
    }

    @Override
    public SampleCategoryEnum getSampleCategoryType() {
        return null;
    }

    @Override
    public void setSampleCategoryType(SampleCategoryEnum sampleCategoryType) {
    }

    @Override
    public Serializable getSampleCategoryValue() {
        return null;
    }

    @Override
    public void setSampleCategoryValue(Serializable sampleCategoryValue) {
    }

    @Override
    public Float getSampleCategoryWeight() {
        return null;
    }

    @Override
    public void setSampleCategoryWeight(Float sampleCategoryWeight) {
    }

    @Override
    public Integer getNumber() {
        return computedOrNotNumber.getData();
    }

    @Override
    public void setNumber(Integer number) {
        computedOrNotNumber.setData(number);
    }

    @Override
    public Float getSampleCategoryComputedWeight() {
        return null;
    }

    @Override
    public void setSampleCategoryComputedWeight(Float sampleCategoryComputedWeight) {
    }

    @Override
    public SpeciesBatch getChildBatchs(int index) {
        return null;
    }

    @Override
    public boolean isChildBatchsEmpty() {
        return false;
    }

    @Override
    public int sizeChildBatchs() {
        return 0;
    }

    @Override
    public void addChildBatchs(SpeciesBatch childBatchs) {
    }

    @Override
    public void addAllChildBatchs(Collection<SpeciesBatch> childBatchs) {
    }

    @Override
    public boolean removeChildBatchs(SpeciesBatch childBatchs) {
        return false;
    }

    @Override
    public boolean removeAllChildBatchs(Collection<SpeciesBatch> childBatchs) {
        return false;
    }

    @Override
    public boolean containsChildBatchs(SpeciesBatch childBatchs) {
        return false;
    }

    @Override
    public boolean containsAllChildBatchs(Collection<SpeciesBatch> childBatchs) {
        return false;
    }

    @Override
    public List<SpeciesBatch> getChildBatchs() {
        return null;
    }

    @Override
    public void setChildBatchs(List<SpeciesBatch> childBatchs) {
    }

    //------------------------------------------------------------------------//
    //-- Sample category                                                    --//
    //------------------------------------------------------------------------//

    public SampleCategory<?> getSampleCategory() {
        return sampleCategory;
    }

    public void setSampleCategory(SampleCategory<?> sampleCategory) {
        Object oldValue = getSampleCategory();
        this.sampleCategory = sampleCategory;
        firePropertyChange(PROPERTY_SAMPLE_CATEGORY, oldValue, sampleCategory);
    }

    public SampleCategory<?> getSampleCategory(SampleCategoryEnum sampleCategoryEnum) {
        switch (sampleCategoryEnum) {
            case sortedUnsorted:
                sampleCategory = getSortedUnsortedCategory();
                break;
            case size:
                sampleCategory = getSizeCategory();
                break;
            case sex:
                sampleCategory = getSexCategory();
                break;
            case maturity:
                sampleCategory = getMaturityCategory();
                break;
            case age:
                sampleCategory = getAgeCategory();
                break;
            default:
        }
        return sampleCategory;
    }

    public SampleCategory getFinestCategory() {
        SampleCategory finestCategory;
        if (ageCategory.isValid()) {
            finestCategory = ageCategory;

        } else if (maturityCategory.isValid()) {
            finestCategory = maturityCategory;

        } else if (sexCategory.isValid()) {
            finestCategory = sexCategory;

        } else if (sizeCategory.isValid()) {
            finestCategory = sizeCategory;

        } else {
            finestCategory = sortedUnsortedCategory;
        }
        return finestCategory;
    }

    //------------------------------------------------------------------------//
    //-- SortedUnsorted category                                            --//
    //------------------------------------------------------------------------//

    public SampleCategory<CaracteristicQualitativeValue> getSortedUnsortedCategory() {
        return sortedUnsortedCategory;
    }

    public void setSortedUnsortedCategory(SampleCategory<CaracteristicQualitativeValue> sortedUnsortedCategory) {
        Object oldCategory = getSortedUnsortedCategory();
        Object oldValue = getSortedUnsortedCategoryValue();
        Object oldWeight = getSortedUnsortedCategoryWeight();
        Object oldComputedWeight = getSortedUnsortedCategoryComputedWeight();
        this.sortedUnsortedCategory = sortedUnsortedCategory;
        firePropertyChange(PROPERTY_SORTED_UNSORTED_CATEGORY, oldCategory, sortedUnsortedCategory);
        firePropertyChange(PROPERTY_SORTED_UNSORTED_CATEGORY_VALUE, oldValue, getSortedUnsortedCategoryValue());
        firePropertyChange(PROPERTY_SORTED_UNSORTED_CATEGORY_WEIGHT, oldWeight, getSortedUnsortedCategoryWeight());
        firePropertyChange(PROPERTY_SORTED_UNSORTED_CATEGORY_COMPUTED_WEIGHT, oldComputedWeight, getSortedUnsortedCategoryComputedWeight());
    }

    public CaracteristicQualitativeValue getSortedUnsortedCategoryValue() {
        return sortedUnsortedCategory.getCategoryValue();
    }

    public Float getSortedUnsortedCategoryWeight() {
        return sortedUnsortedCategory.getCategoryWeight();
    }

    public void setSortedUnsortedCategoryWeight(Float sortedUnsortedCategoryWeight) {
        Object oldValue = getSortedUnsortedCategoryWeight();
        sortedUnsortedCategory.setCategoryWeight(sortedUnsortedCategoryWeight);
        firePropertyChange(PROPERTY_SORTED_UNSORTED_CATEGORY_WEIGHT, oldValue, sortedUnsortedCategoryWeight);
    }

    public Float getSortedUnsortedCategoryComputedWeight() {
        return sortedUnsortedCategory.getComputedWeight();
    }

    public void setSortedUnsortedCategoryComputedWeight(Float sortedUnsortedCategoryComputedWeight) {
        Object oldValue = getSortedUnsortedCategoryComputedWeight();
        sortedUnsortedCategory.setComputedWeight(sortedUnsortedCategoryComputedWeight);
        firePropertyChange(PROPERTY_SORTED_UNSORTED_CATEGORY_COMPUTED_WEIGHT, oldValue, sortedUnsortedCategoryComputedWeight);
    }

    //------------------------------------------------------------------------//
    //-- Size category                                                      --//
    //------------------------------------------------------------------------//

    public SampleCategory<CaracteristicQualitativeValue> getSizeCategory() {
        return sizeCategory;
    }

    public void setSizeCategory(SampleCategory<CaracteristicQualitativeValue> sizeCategory) {
        Object oldCategory = getSizeCategory();
        Object oldValue = getSizeCategoryValue();
        Object oldWeight = getSizeCategoryWeight();
        Object oldComputedWeight = getSizeCategoryComputedWeight();
        this.sizeCategory = sizeCategory;
        firePropertyChange(PROPERTY_SIZE_CATEGORY, oldCategory, sizeCategory);
        firePropertyChange(PROPERTY_SIZE_CATEGORY_VALUE, oldValue, getSizeCategoryValue());
        firePropertyChange(PROPERTY_SIZE_CATEGORY_WEIGHT, oldWeight, getSizeCategoryWeight());
        firePropertyChange(PROPERTY_SIZE_CATEGORY_COMPUTED_WEIGHT, oldComputedWeight, getSizeCategoryComputedWeight());
    }

    public CaracteristicQualitativeValue getSizeCategoryValue() {
        return sizeCategory.getCategoryValue();
    }

    public Float getSizeCategoryWeight() {
        return sizeCategory.getCategoryWeight();
    }

    public void setSizeCategoryWeight(Float sizeCategoryWeight) {
        Object oldValue = getSizeCategoryWeight();
        this.sizeCategory.setCategoryWeight(sizeCategoryWeight);
        firePropertyChange(PROPERTY_SIZE_CATEGORY_WEIGHT, oldValue, sizeCategoryWeight);
    }

    public Float getSizeCategoryComputedWeight() {
        return sizeCategory.getComputedWeight();
    }

    public void setSizeCategoryComputedWeight(Float sizeCategoryComputedWeight) {
        Object oldValue = getSizeCategoryComputedWeight();
        this.sizeCategory.setComputedWeight(sizeCategoryComputedWeight);
        firePropertyChange(PROPERTY_SIZE_CATEGORY_COMPUTED_WEIGHT, oldValue, sizeCategoryComputedWeight);
    }

    //------------------------------------------------------------------------//
    //-- Sex category                                                       --//
    //------------------------------------------------------------------------//

    public SampleCategory<CaracteristicQualitativeValue> getSexCategory() {
        return sexCategory;
    }

    public void setSexCategory(SampleCategory<CaracteristicQualitativeValue> sexCategory) {
        Object oldCategory = getSexCategory();
        Object oldValue = getSexCategoryValue();
        Object oldWeight = getSexCategoryWeight();
        Object oldComputedWeight = getSexCategoryComputedWeight();
        this.sexCategory = sexCategory;
        firePropertyChange(PROPERTY_SEX_CATEGORY, oldCategory, sexCategory);
        firePropertyChange(PROPERTY_SEX_CATEGORY_VALUE, oldValue, getSexCategoryValue());
        firePropertyChange(PROPERTY_SEX_CATEGORY_WEIGHT, oldWeight, getSexCategoryWeight());
        firePropertyChange(PROPERTY_SEX_CATEGORY_COMPUTED_WEIGHT, oldComputedWeight, getSexCategoryComputedWeight());
    }

    public CaracteristicQualitativeValue getSexCategoryValue() {
        return sexCategory.getCategoryValue();
    }

    public Float getSexCategoryWeight() {
        return sexCategory.getCategoryWeight();
    }

    public void setSexCategoryWeight(Float sexCategoryWeight) {
        Object oldValue = getSexCategoryWeight();
        this.sexCategory.setCategoryWeight(sexCategoryWeight);
        firePropertyChange(PROPERTY_SEX_CATEGORY_WEIGHT, oldValue, sexCategoryWeight);
    }

    public Float getSexCategoryComputedWeight() {
        return sexCategory.getComputedWeight();
    }

    public void setSexCategoryComputedWeight(Float sexCategoryComputedWeight) {
        Object oldValue = getSexCategoryComputedWeight();
        this.sexCategory.setComputedWeight(sexCategoryComputedWeight);
        firePropertyChange(PROPERTY_SEX_CATEGORY_COMPUTED_WEIGHT, oldValue, sexCategoryComputedWeight);
    }

    //------------------------------------------------------------------------//
    //-- Maturity category                                                  --//
    //------------------------------------------------------------------------//

    public SampleCategory<CaracteristicQualitativeValue> getMaturityCategory() {
        return maturityCategory;
    }

    public void setMaturityCategory(SampleCategory<CaracteristicQualitativeValue> maturityCategory) {
        Object oldCategory = getMaturityCategory();
        Object oldValue = getMaturityCategoryValue();
        Object oldWeight = getMaturityCategoryWeight();
        Object oldComputedWeight = getMaturityCategoryComputedWeight();
        this.maturityCategory = maturityCategory;
        firePropertyChange(PROPERTY_MATURITY_CATEGORY, oldCategory, maturityCategory);
        firePropertyChange(PROPERTY_MATURITY_CATEGORY_VALUE, oldValue, getMaturityCategoryValue());
        firePropertyChange(PROPERTY_MATURITY_CATEGORY_WEIGHT, oldWeight, getMaturityCategoryWeight());
        firePropertyChange(PROPERTY_MATURITY_CATEGORY_COMPUTED_WEIGHT, oldComputedWeight, getMaturityCategoryComputedWeight());
    }

    public CaracteristicQualitativeValue getMaturityCategoryValue() {
        return maturityCategory.getCategoryValue();
    }

    public Float getMaturityCategoryWeight() {
        return maturityCategory.getCategoryWeight();
    }

    public void setMaturityCategoryWeight(Float maturityCategoryWeight) {
        Object oldValue = getMaturityCategoryWeight();
        this.maturityCategory.setCategoryWeight(maturityCategoryWeight);
        firePropertyChange(PROPERTY_MATURITY_CATEGORY_WEIGHT, oldValue, maturityCategoryWeight);
    }

    public Float getMaturityCategoryComputedWeight() {
        return maturityCategory.getComputedWeight();
    }

    public void setMaturityCategoryComputedWeight(Float maturityCategoryComputedWeight) {
        Object oldValue = getMaturityCategoryComputedWeight();
        this.maturityCategory.setComputedWeight(maturityCategoryComputedWeight);
        firePropertyChange(PROPERTY_MATURITY_CATEGORY_COMPUTED_WEIGHT, oldValue, maturityCategoryComputedWeight);
    }

    //------------------------------------------------------------------------//
    //-- Age category                                                       --//
    //------------------------------------------------------------------------//

    public SampleCategory<Float> getAgeCategory() {
        return ageCategory;
    }

    public void setAgeCategory(SampleCategory<Float> ageCategory) {
        Object oldCategory = getAgeCategory();
        Object oldValue = getAgeCategoryValue();
        Object oldWeight = getAgeCategoryWeight();
        Object oldComputedWeight = getAgeCategoryComputedWeight();
        this.ageCategory = ageCategory;
        firePropertyChange(PROPERTY_AGE_CATEGORY, oldCategory, ageCategory);
        firePropertyChange(PROPERTY_AGE_CATEGORY_VALUE, oldValue, getAgeCategoryValue());
        firePropertyChange(PROPERTY_AGE_CATEGORY_WEIGHT, oldWeight, getAgeCategoryWeight());
        firePropertyChange(PROPERTY_AGE_CATEGORY_COMPUTED_WEIGHT, oldComputedWeight, getAgeCategoryComputedWeight());
    }

    public Float getAgeCategoryValue() {
        return ageCategory.getCategoryValue();
    }

    public Float getAgeCategoryWeight() {
        return ageCategory.getCategoryWeight();
    }

    public void setAgeCategoryWeight(Float ageCategoryWeight) {
        Object oldValue = getAgeCategoryWeight();
        ageCategory.setCategoryWeight(ageCategoryWeight);
        firePropertyChange(PROPERTY_AGE_CATEGORY_WEIGHT, oldValue, ageCategoryWeight);
    }

    public Float getAgeCategoryComputedWeight() {
        return ageCategory.getComputedWeight();
    }

    public void setAgeCategoryComputedWeight(Float ageCategoryComputedWeight) {
        Object oldValue = getAgeCategoryComputedWeight();
        ageCategory.setComputedWeight(ageCategoryComputedWeight);
        firePropertyChange(PROPERTY_AGE_CATEGORY_COMPUTED_WEIGHT, oldValue, ageCategoryComputedWeight);
    }

    //------------------------------------------------------------------------//
    //-- AttachmentModelAware                                               --//
    //------------------------------------------------------------------------//

    @Override
    public AttachementObjectTypeEnum getObjectType() {
        return AttachementObjectTypeEnum.SAMPLE;
    }

    @Override
    public Integer getObjectId() {
        return getIdAsInt();
    }

    @Override
    public List<Attachment> getAttachment() {
        return attachment;
    }

    @Override
    public void addAllAttachment(Collection<Attachment> attachments) {
        this.attachment.addAll(attachments);
        firePropertyChange(PROPERTY_ATTACHMENT, null, getAttachment());
    }

    @Override
    public void addAttachment(Attachment attachment) {
        this.attachment.add(attachment);
        firePropertyChange(PROPERTY_ATTACHMENT, null, getAttachment());
    }

    @Override
    public void removeAllAttachment(Collection<Attachment> attachments) {
        this.attachment.removeAll(attachments);
        firePropertyChange(PROPERTY_ATTACHMENT, null, getAttachment());
    }

    @Override
    public void removeAttachment(Attachment attachment) {
        this.attachment.remove(attachment);
        firePropertyChange(PROPERTY_ATTACHMENT, null, getAttachment());
    }

    //------------------------------------------------------------------------//
    //-- Other properties                                                   --//
    //------------------------------------------------------------------------//

    public List<SpeciesBatchRowModel> getChildBatch() {
        return childBatch;
    }

    public void setChildBatch(List<SpeciesBatchRowModel> childBatch) {
        this.childBatch = childBatch;
        // force to propagate child changes
        firePropertyChange(PROPERTY_CHILD_BATCH, null, childBatch);
        firePropertyChange(PROPERTY_BATCH_LEAF, null, isBatchLeaf());
    }

    public boolean isBatchLeaf() {
        return CollectionUtils.isEmpty(childBatch);
    }

    public boolean isBatchRoot() {
        return getParentBatch() == null;
    }

    public List<SpeciesFrequencyRowModel> getFrequency() {
        return frequency;
    }

    public void setFrequency(List<SpeciesFrequencyRowModel> frequency) {
        this.frequency = frequency;
        // force to propagate frequencies changes
        firePropertyChange(PROPERTY_FREQUENCY, null, frequency);
    }

    public TuttiComputedOrNotData<Integer> getComputedOrNotNumber() {
        return computedOrNotNumber;
    }

    public void setComputedOrNotNumber(TuttiComputedOrNotData<Integer> computedOrNotNumber) {
        this.computedOrNotNumber = computedOrNotNumber;
    }

    public TuttiComputedOrNotData<Float> getComputedOrNotWeight() {
        return computedOrNotWeight;
    }

    public void setComputedOrNotWeight(TuttiComputedOrNotData<Float> computedOrNotWeight) {
        this.computedOrNotWeight = computedOrNotWeight;
    }


    public void collectShell(Set<SpeciesBatchRowModel> collectedRows) {

        if (!isBatchLeaf()) {

            for (SpeciesBatchRowModel batchChild : getChildBatch()) {
                collectedRows.add(batchChild);
                batchChild.collectShell(collectedRows);
            }
        }
    }

    public SpeciesBatchRowModel getFirstAncestor(SampleCategory<?> entrySampleCategory) {
        SpeciesBatchRowModel result = this;
        if (getParentBatch() != null) {
            SpeciesBatchRowModel parentBatch = getParentBatch();
            SampleCategory<?> parentSampleCategory = parentBatch.getSampleCategory(entrySampleCategory.getCategoryType());
            if (ObjectUtils.equals(entrySampleCategory, parentSampleCategory)) {

                result = parentBatch.getFirstAncestor(entrySampleCategory);
            }
        }
        return result;
    }
}
