package fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.split;

/*
 * #%L
 * Tutti :: UI
 * $Id: SplitBenthosBatchUIHandler.java 896 2013-04-30 14:12:06Z kmorin $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-2.1/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/catches/benthos/split/SplitBenthosBatchUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import fr.ifremer.tutti.persistence.entities.data.SampleCategory;
import fr.ifremer.tutti.persistence.entities.data.SampleCategoryEnum;
import fr.ifremer.tutti.persistence.entities.referential.Caracteristic;
import fr.ifremer.tutti.persistence.entities.referential.CaracteristicQualitativeValue;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUIHandler;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.BenthosBatchRowModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.BenthosBatchUI;
import fr.ifremer.tutti.ui.swing.util.Cancelable;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import fr.ifremer.tutti.ui.swing.util.TuttiUI;
import fr.ifremer.tutti.ui.swing.util.table.AbstractTuttiTableUIHandler;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.table.DefaultTableColumnModelExt;

import javax.swing.DefaultComboBoxModel;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.List;

/**
 * Handler of {@link SplitBenthosBatchUI}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.3
 */
public class SplitBenthosBatchUIHandler
        extends AbstractTuttiTableUIHandler<SplitBenthosBatchRowModel, SplitBenthosBatchUIModel, SplitBenthosBatchUI> implements Cancelable {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(SplitBenthosBatchUIHandler.class);

    /**
     * Parent UI.
     *
     * @since 0.3
     */
    private final BenthosBatchUI parentUi;

    public SplitBenthosBatchUIHandler(EditCatchesUI parentUi,
                                      SplitBenthosBatchUI ui) {
        super(parentUi.getHandler().getContext(), ui,
              SplitBenthosBatchRowModel.PROPERTY_CATEGORY_VALUE,
              SplitBenthosBatchRowModel.PROPERTY_WEIGHT);
        this.parentUi = parentUi.getBenthosTabContent();
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiTableUIHandler methods                                --//
    //------------------------------------------------------------------------//

    @Override
    public SplitBenthosBatchTableModel getTableModel() {
        return (SplitBenthosBatchTableModel) getTable().getModel();
    }

    @Override
    public JXTable getTable() {
        return ui.getTable();
    }

    @Override
    protected boolean isRowValid(SplitBenthosBatchRowModel row) {
        return row.isValid();
    }

    @Override
    protected void saveSelectedRowIfRequired(TuttiBeanMonitor<SplitBenthosBatchRowModel> rowMonitor,
                                             SplitBenthosBatchRowModel row) {
        if (rowMonitor.wasModified()) {

            if (row.isValid()) {
                if (log.isInfoEnabled()) {
                    log.info("Change row that was modified and valid");
                }
            }

            rowMonitor.clearModified();
        }
    }

    @Override
    protected void onRowModified(int rowIndex,
                                 SplitBenthosBatchRowModel row,
                                 String propertyName,
                                 Object oldValue,
                                 Object newValue) {

        recomputeRowValidState(row);

        if (SplitBenthosBatchRowModel.PROPERTY_WEIGHT.equals(propertyName)) {

            // Need to recompute the sample weight
            computeSampleWeight(row);
        }
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    public SwingValidator<SplitBenthosBatchUIModel> getValidator() {
        return ui.getValidator();
    }

    @Override
    public void beforeInitUI() {

        SplitBenthosBatchUIModel model = new SplitBenthosBatchUIModel();

        ui.setContextValue(model);
    }

    @Override
    public void afterInitUI() {

        initUI(ui);

        SplitBenthosBatchUIModel model = getModel();

        // when category changed, remove selected category
        model.addPropertyChangeListener(SplitBenthosBatchUIModel.PROPERTY_CATEGORY, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {

                SplitBenthosBatchUIModel source =
                        (SplitBenthosBatchUIModel) evt.getSource();

                // unselect previous selected category
                source.setSelectedCategory(null);

                // fill comboBox with new list
                List<SampleCategoryEnum> data = (List<SampleCategoryEnum>) evt.getNewValue();
                ui.getCategoryComboBox().setModel(new DefaultComboBoxModel(data.toArray()));
            }
        });

        // when selected category changed, regenerate the table model + add inside some default rows
        model.addPropertyChangeListener(SplitBenthosBatchUIModel.PROPERTY_SELECTED_CATEGORY, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {

                SplitBenthosBatchUIModel source =
                        (SplitBenthosBatchUIModel) evt.getSource();

                // when selected category change, sample total weight is reset
                source.setSampleWeight(null);

                SampleCategoryEnum newValue =
                        (SampleCategoryEnum) evt.getNewValue();
                generateTableModel(newValue);
            }
        });

        ui.getCategoryComboBox().setRenderer(newListCellRender(SampleCategoryEnum.class));


        ui.getCategoryComboBox().addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {
                JComboBox comboBox = (JComboBox) e.getSource();
                getModel().setSelectedCategory((SampleCategoryEnum) comboBox.getSelectedItem());
            }
        });

        generateTableModel(null);

        initTable(getTable());

        listenValidatorValid(ui.getValidator(), model);
    }

    @Override
    protected JComponent getComponentToFocus() {
        return getUI().getCategoryComboBox();
    }

    @Override
    public void onCloseUI() {
        if (log.isDebugEnabled()) {
            log.debug("closing: " + ui);
        }

        // evict model from validator
        ui.getValidator().setBean(null);

        // when canceling always invalid model
        getModel().setValid(false);
        getModel().setSelectedCategory(null);

        EditCatchesUI parent = SwingUtil.getParentContainer(ui, EditCatchesUI.class);
        parent.getHandler().setBenthosSelectedCard(EditCatchesUIHandler.MAIN_CARD);
    }

    //------------------------------------------------------------------------//
    //-- Cancelable methods                                                 --//
    //------------------------------------------------------------------------//

    @Override
    public void cancel() {

        if (log.isInfoEnabled()) {
            log.info("Cancel UI " + ui);
        }
        closeUI(ui);
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    public String decorateSpecies(Species object) {
        String result = object == null ? "" : super.decorate(object);
        return result;
    }

    public void editBatch(BenthosBatchRowModel batch) {

        // get possible the last used
        List<SampleCategoryEnum> categories =
                Lists.newArrayList(SampleCategoryEnum.values());

        if (batch != null) {
            //TODO Use the samplingOrder (+ the one from the table model)

            SampleCategory<?> lastCategory = batch.getSampleCategory();

            Preconditions.checkNotNull(
                    lastCategory,
                    "Can't split a species batch with no sample category.");

            if (batch.getSortedUnsortedCategory().isValid()) {
                categories.remove(SampleCategoryEnum.sortedUnsorted);
            }

            if (batch.getSizeCategory().isValid()) {
                categories.remove(SampleCategoryEnum.size);
                categories.remove(SampleCategoryEnum.sortedUnsorted);
            }

            if (batch.getSexCategory().isValid()) {
                categories.remove(SampleCategoryEnum.sex);
                categories.remove(SampleCategoryEnum.size);
                categories.remove(SampleCategoryEnum.sortedUnsorted);
            }

            if (batch.getMaturityCategory().isValid()) {
                categories.remove(SampleCategoryEnum.maturity);
                categories.remove(SampleCategoryEnum.sex);
                categories.remove(SampleCategoryEnum.size);
                categories.remove(SampleCategoryEnum.sortedUnsorted);
            }

            if (batch.getAgeCategory().isValid()) {
                categories.remove(SampleCategoryEnum.age);
                categories.remove(SampleCategoryEnum.maturity);
                categories.remove(SampleCategoryEnum.sex);
                categories.remove(SampleCategoryEnum.size);
                categories.remove(SampleCategoryEnum.sortedUnsorted);
            }
        }

        SplitBenthosBatchUIModel model = getModel();

        // connect model to validator
        ui.getValidator().setBean(model);

        model.setSampleWeight(null);
        model.setCategory(categories);
        if (CollectionUtils.isNotEmpty(categories)) {
            model.setSelectedCategory(categories.get(0));
        }

        // keep batch (will be used to push back editing entry)
        model.setBatch(batch);
    }

    public void save() {

        if (log.isInfoEnabled()) {
            log.info("Save UI " + ui);
        }

        EditCatchesUI parent = SwingUtil.getParentContainer(ui, EditCatchesUI.class);
        parent.getBenthosTabContent().getHandler().splitBatch(getModel());

        // close dialog
        closeUI(ui);
    }

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected void computeSampleWeight(SplitBenthosBatchRowModel row) {

        if (log.isInfoEnabled()) {
            log.info("Will recompute sample weight from row: " + row);
        }

        Float result = null;
        List<SplitBenthosBatchRowModel> rows = getTableModel().getRows();
        for (SplitBenthosBatchRowModel rowModel : rows) {
            Float weight = rowModel.getWeight();
            if (weight != null) {
                if (result == null) {
                    result = 0f;
                }
                result += weight;
            }
        }
        getModel().setSampleWeight(result);
    }

    protected void generateTableModel(SampleCategoryEnum category) {

        // when generate a new table model, then reset previous rows from model
        getModel().setRows(null);

        Caracteristic data = null;

        DefaultTableColumnModelExt columnModel =
                new DefaultTableColumnModelExt();

        { // Selection

            addBooleanColumnToModel(columnModel, SplitBenthosBatchTableModel.SELECTED, getTable());
        }

        boolean editableCategoryValue = false;
        if (category != null) {

            switch (category) {

                case sortedUnsorted:
                    data = persistenceService.getSortedUnsortedCaracteristic();
                    break;
                case size:
                    data = persistenceService.getSizeCategoryCaracteristic();
                    break;
                case sex:
                    data = persistenceService.getSexCaracteristic();
                    break;
                case maturity:
                    data = persistenceService.getMaturityCaracteristic();
                    break;
                case age:
                    editableCategoryValue = true;
                    addFloatColumnToModel(columnModel,
                                          SplitBenthosBatchTableModel.EDITABLE_CATEGORY_VALUE,
                                          TuttiUI.DECIMAL1_PATTERN);
                    break;
            }

            if (data != null) {

                if (log.isDebugEnabled()) {
                    log.debug("Got " + data.sizeQualitativeValue() + " qualitative data to add");
                }
                addColumnToModel(columnModel,
                                 null,
                                 newTableCellRender(CaracteristicQualitativeValue.class),
                                 SplitBenthosBatchTableModel.READ_ONLY_CATEGORY_VALUE);
            }
            { // Weight

                addFloatColumnToModel(columnModel,
                                      SplitBenthosBatchTableModel.WEIGHT,
                                      TuttiUI.DECIMAL3_PATTERN);
            }
        }

        // create table model
        SplitBenthosBatchTableModel tableModel =
                new SplitBenthosBatchTableModel(columnModel,
                                                getModel(),
                                                editableCategoryValue);

        JXTable table = getTable();

        // remove all listener on tables we could add before
        uninstallTableSaveOnRowChangedSelectionListener();
        uninstallTableKeyListener();

        if (log.isDebugEnabled()) {
            log.debug("Install new table model " + tableModel);
        }
        table.setModel(tableModel);
        table.setColumnModel(columnModel);

        // install table listeners
        installTableSaveOnRowChangedSelectionListener();
        installTableKeyListener(columnModel, table);

        // fill datas

        List<SplitBenthosBatchRowModel> rows = Lists.newArrayList();

        if (data != null) {

            // add a row for each qualitive value
            for (CaracteristicQualitativeValue qualitativeValue : data.getQualitativeValue()) {
                if (log.isDebugEnabled()) {
                    log.debug("Add QV: " + qualitativeValue);
                }
                SplitBenthosBatchRowModel newRow = tableModel.createNewRow();
                newRow.setCategoryValue(qualitativeValue);
                rows.add(newRow);
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Will add " + rows.size() + " rows in table model " +
                      "(can add a first empty row? " + editableCategoryValue + ").");
        }

        getModel().setRows(rows);
    }

}
