package fr.ifremer.tutti.ui.swing.config;

/*
 * #%L
 * Tutti :: Application
 * $Id: TuttiApplicationConfig.java 881 2013-04-29 10:22:47Z kmorin $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-2.1/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/config/TuttiApplicationConfig.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Charsets;
import fr.ifremer.tutti.TuttiTechnicalException;
import fr.ifremer.tutti.service.config.TuttiServiceConfig;
import fr.ifremer.tutti.service.config.TuttiServiceConfigOption;
import fr.ifremer.tutti.ui.swing.util.editor.CoordinateEditorType;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.builder.ReflectionToStringBuilder;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.Version;
import org.nuiton.util.config.ApplicationConfig;
import org.nuiton.util.config.ApplicationConfigHelper;
import org.nuiton.util.config.ApplicationConfigProvider;
import org.nuiton.util.config.ArgumentsParserException;

import javax.swing.KeyStroke;
import java.awt.Color;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.List;
import java.util.Locale;
import java.util.Set;

import static org.nuiton.i18n.I18n._;

/**
 * Tutti application config.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class TuttiApplicationConfig {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(TuttiApplicationConfig.class);

    /**
     * Delegate application config object containing configuration.
     *
     * @since 0.1
     */
    protected final ApplicationConfig applicationConfig;

    /**
     * Service configuration to pass to service context.
     *
     * @since 0.1
     */
    protected final TuttiServiceConfig serviceConfig;

    protected final String[] optionKeyToNotSave;

    public TuttiApplicationConfig(String file, String... args) {

        applicationConfig = new ApplicationConfig();
        applicationConfig.setEncoding(Charsets.UTF_8.name());

        // get all config providers
        Set<ApplicationConfigProvider> providers =
                ApplicationConfigHelper.getProviders(null,
                                                     null,
                                                     null,
                                                     true);

        // load all default options
        ApplicationConfigHelper.loadAllDefaultOption(applicationConfig,
                                                     providers);

        // get all transient and final option keys
        Set<String> optionToSkip =
                ApplicationConfigHelper.getTransientOptionKeys(providers);

        if (log.isDebugEnabled()) {
            log.debug("Option that won't be saved: " + optionToSkip);
        }
        optionKeyToNotSave = optionToSkip.toArray(new String[optionToSkip.size()]);

        applicationConfig.setConfigFileName(file);

        try {
            applicationConfig.parse(args);

        } catch (ArgumentsParserException e) {
            throw new TuttiTechnicalException(_("tutti.config.parse.error"), e);
        }

        //TODO Review this, this is very dirty to do this...
        File tuttiBasedir = applicationConfig.getOptionAsFile(
                TuttiServiceConfigOption.BASEDIR.getKey());

        if (tuttiBasedir == null) {
            tuttiBasedir = new File("");
        }
        if (!tuttiBasedir.isAbsolute()) {
            tuttiBasedir = new File(tuttiBasedir.getAbsolutePath());
        }
        if (tuttiBasedir.getName().equals("..")) {
            tuttiBasedir = tuttiBasedir.getParentFile().getParentFile();
        }
        if (tuttiBasedir.getName().equals(".")) {
            tuttiBasedir = tuttiBasedir.getParentFile();
        }
        if (log.isInfoEnabled()) {
            log.info("Application basedir: " + tuttiBasedir);
        }
        applicationConfig.setOption(
                TuttiServiceConfigOption.BASEDIR.getKey(),
                tuttiBasedir.getAbsolutePath());

        serviceConfig = new TuttiServiceConfig(applicationConfig);
    }

    public ApplicationConfig getApplicationConfig() {
        return applicationConfig;
    }

    public TuttiServiceConfig getServiceConfig() {
        return serviceConfig;
    }

    protected File configFile;

    protected File getConfigFile() {
        if (configFile == null) {
            File dir = getTuttiBasedir();
            if (dir == null || !dir.exists() || !isFullLaunchMode()) {
                dir = new File(applicationConfig.getUserConfigDirectory());
            }
            configFile = new File(dir, applicationConfig.getConfigFileName());
        }
        return configFile;
    }

    public void save() {

        File file = getConfigFile();
        if (log.isInfoEnabled()) {
            log.info("Save configuration at: " + file);
        }
        try {
            applicationConfig.save(file, false, optionKeyToNotSave);
        } catch (IOException e) {
            throw new TuttiTechnicalException(_("tutti.config.save.error", file), e);
        }
    }

    @Override
    public String toString() {
        return ReflectionToStringBuilder.toString(this);
    }

    public boolean isFullLaunchMode() {
        return "full".equals(getTuttiLaunchMode());
    }

    public String getTuttiLaunchMode() {
        return applicationConfig.getOption(TuttiApplicationConfigOption.TUTTI_LAUNCH_MODE.getKey());
    }

    /** @return {@link TuttiApplicationConfigOption#UI_CONFIG_FILE} value */
    public File getUIConfigFile() {
        return applicationConfig.getOptionAsFile(TuttiApplicationConfigOption.UI_CONFIG_FILE.getKey());
    }

    public Float getDifferenceRateBetweenSortedAndTotalWeights() {
        return applicationConfig.getOptionAsFloat(TuttiApplicationConfigOption.TOTAL_SORTED_WEIGHTS_DIFFERENCE_RATE.getKey());
    }

    public String getTremieCarousselVesselId() {
        return applicationConfig.getOption(TuttiApplicationConfigOption.TREMIE_CAROUSSEL_VESSEL_ID.getKey());
    }

    public boolean isAutoPopupNumberEditor() {
        return applicationConfig.getOptionAsBoolean(TuttiApplicationConfigOption.AUTO_POPUP_NUMBER_EDITOR.getKey());
    }

    public boolean isShowNumberEditorButton() {
        return applicationConfig.getOptionAsBoolean(TuttiApplicationConfigOption.SHOW_NUMBER_EDITOR_BUTTON.getKey());
    }

    public CoordinateEditorType getCoordinateEditorType() {
        String value = applicationConfig.getOption(TuttiApplicationConfigOption.COORDINATE_EDITOR_TYPE.getKey());
        return CoordinateEditorType.valueOf(value);
    }

    public String getProgramId() {
        String result = applicationConfig.getOption(TuttiApplicationConfigOption.PROGRAM_ID.getKey());
        return StringUtils.isBlank(result) ? null : result;
    }

    public String getCruiseId() {
        String result = applicationConfig.getOption(TuttiApplicationConfigOption.CRUISE_ID.getKey());
        return StringUtils.isBlank(result) ? null : result;
    }

    public String getProtocolId() {
        String result = applicationConfig.getOption(TuttiApplicationConfigOption.PROTOCOL_ID.getKey());
        return StringUtils.isBlank(result) ? null : result;
    }

    public Color getColorRowInvalid() {
        return applicationConfig.getOptionAsColor(TuttiApplicationConfigOption.COLOR_ROW_INVALID.getKey());
    }

    public Color getColorRowReadOnly() {
        return applicationConfig.getOptionAsColor(TuttiApplicationConfigOption.COLOR_ROW_READ_ONLY.getKey());
    }

    public Color getColorCellWithValue() {
        return applicationConfig.getOptionAsColor(TuttiApplicationConfigOption.COLOR_CELL_WITH_VALUE.getKey());
    }

    public Color getColorRowToConfirm() {
        return applicationConfig.getOptionAsColor(TuttiApplicationConfigOption.COLOR_ROW_TO_CONFIRM.getKey());
    }

    public Color getColorComputedWeights() {
        return applicationConfig.getOptionAsColor(TuttiApplicationConfigOption.COLOR_COMPUTED_WEIGHTS.getKey());
    }

    public Color getColorBlockingLayer() {
        return applicationConfig.getOptionAsColor(TuttiApplicationConfigOption.COLOR_BLOCKING_LAYER.getKey());
    }

    public KeyStroke getShortcutClosePopup() {
        return applicationConfig.getOptionAsKeyStroke(TuttiApplicationConfigOption.SHORTCUT_CLOSE_POPUP.getKey());
    }

    public KeyStroke getShortCut(String actionName) {
        KeyStroke result = applicationConfig.getOptionAsKeyStroke(
                "tutti.ui." + actionName);
        return result;
    }

    public String getDateFormat() {
        return applicationConfig.getOption(TuttiApplicationConfigOption.DATE_FORMAT.getKey());
    }

    public Version getVersion() {
        return serviceConfig.getVersion();
    }

    public int getInceptionYear() {
        return serviceConfig.getInceptionYear();
    }

    public URL getSiteUrl() {
        return serviceConfig.getSiteUrl();
    }

    public String getOrganizationName() {
        return serviceConfig.getOrganizationName();
    }

    public File getTmpDirectory() {
        return serviceConfig.getTmpDirectory();
    }

    public File getNewTmpDirectory(String name) {
        return new File(getTmpDirectory(), name + "_" + System.nanoTime());
    }

    public File getDataDirectory() {
        return serviceConfig.getDataDirectory();
    }

    public File getI18nDirectory() {
        return applicationConfig.getOptionAsFile(
                TuttiApplicationConfigOption.TUTTI_I18N_DIRECTORY.getKey());
    }

    public Locale getI18nLocale() {
        return applicationConfig.getOptionAsLocale(
                TuttiApplicationConfigOption.TUTTI_I18N_LOCALE.getKey());
    }

    public void setI18nLocale(Locale locale) {
        applicationConfig.setOption(TuttiApplicationConfigOption.TUTTI_I18N_LOCALE.getKey(), locale.toString());
    }

    public File getHelpDirectory() {
        return applicationConfig.getOptionAsFile(
                TuttiApplicationConfigOption.TUTTI_HELP_DIRECTORY.getKey());
    }

    public File getHelpDirectoryWithLocale() {
        File result = getHelpDirectory();
        return new File(result, getI18nLocale().getLanguage());
    }

    public void setProgramId(String programId) {
        if (programId == null) {
            programId = "";
        }
        applicationConfig.setOption(TuttiApplicationConfigOption.PROGRAM_ID.getKey(), programId);
    }

    public void setCruiseId(String cruiseId) {
        if (cruiseId == null) {
            cruiseId = "";
        }
        applicationConfig.setOption(TuttiApplicationConfigOption.CRUISE_ID.getKey(), cruiseId);
    }

    public void setProtocolId(String protocolId) {
        if (protocolId == null) {
            protocolId = "";
        }
        applicationConfig.setOption(TuttiApplicationConfigOption.PROTOCOL_ID.getKey(), protocolId);
    }

    public File getTuttiBasedir() {
        return serviceConfig.getBasedir();
    }

    public String getUpdateUrl() {
        return applicationConfig.getOption(TuttiApplicationConfigOption.TUTTI_UPDATE_URL.getKey());
    }

    public String getAdminPassword() {
        return applicationConfig.getOption(TuttiApplicationConfigOption.ADMIN_PASSWORD.getKey());
    }
}
