package fr.ifremer.tutti.ui.swing.util.action;

/*
 * #%L
 * Tutti :: UI
 * $Id: AbstractTuttiAction.java 618 2013-03-14 12:47:49Z kmorin $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.1/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/util/action/AbstractTuttiAction.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import fr.ifremer.tutti.persistence.ProgressionModel;
import fr.ifremer.tutti.ui.swing.TuttiDataContext;
import fr.ifremer.tutti.ui.swing.TuttiUIContext;
import fr.ifremer.tutti.ui.swing.config.TuttiApplicationConfig;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiUIHandler;
import fr.ifremer.tutti.ui.swing.util.TuttiUI;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.beans.AbstractBean;
import org.nuiton.util.decorator.Decorator;

import javax.swing.SwingUtilities;

/**
 * Tutti base action.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.0
 */
public abstract class AbstractTuttiAction<M extends AbstractBean, UI extends TuttiUI<M, ?>, H extends AbstractTuttiUIHandler<M, UI>>
        extends AbstractBean {

    private static final Log log = LogFactory.getLog(AbstractTuttiAction.class);

    public static final String PROPERTY_DONE = "done";

    protected final H handler;

    protected String actionDescription;

    protected final boolean hideBody;

    protected final Object lock = new Object();

    protected abstract void doAction() throws Exception;

    protected AbstractTuttiAction(H handler, boolean hideBody) {
        this.handler = handler;
        this.hideBody = hideBody;
    }

    protected boolean prepareAction() throws Exception {
        // by default nothing to prepare
        return true;
    }

    protected void releaseAction() {
        // by default nothing to clean
        firePropertyChange(PROPERTY_DONE, null, true);
    }

    public void postSuccessAction() {
        // by default nothing to do after action
    }

    public void postFailedAction(Throwable error) {
        // by default nothing to do after action
    }

    public static void runAction(AbstractTuttiAction action) {
        Throwable error = null;
        try {
            action.doAction();
            action.postSuccessAction();
        } catch (Throwable e) {
            error = e;
            throw TuttiActionException.propagateError(action, e);
        } finally {
            if (error != null) {
                action.postFailedAction(error);
            }
            action.releaseAction();
        }
    }

    public H getHandler() {
        return handler;
    }

    public M getModel() {
        return handler.getModel();
    }

    public final UI getUI() {
        return handler.getUI();
    }

    public TuttiUIContext getContext() {
        return handler.getContext();
    }

    public String getActionDescription() {
        return actionDescription;
    }

    public void setActionDescription(String actionDescription) {
        this.actionDescription = actionDescription;
    }

    protected void setProgressionModel(ProgressionModel progressionModel) {
        getContext().getActionUI().getModel().setProgressionModel(progressionModel);
    }

    protected ProgressionModel getProgressionModel() {
        return getContext().getActionUI().getModel().getProgressionModel();
    }

    public boolean isHideBody() {
        return hideBody;
    }

    public TuttiDataContext getDataContext() {
        return getContext().getDataContext();
    }

    protected TuttiApplicationConfig getConfig() {
        return getContext().getConfig();
    }

    protected void sendMessage(String message) {
        getContext().showInformationMessage(message);
    }

    protected <O> Decorator<O> getDecorator(Class<O> type, String name) {
        Decorator<O> decorator = handler.getDecorator(type, name);
        Preconditions.checkNotNull(decorator);
        return decorator;
    }

    protected String decorate(Object object) {
        return getDecorator(object.getClass(), null).toString(object);
    }

    protected String decorate(Object object, String context) {
        return getDecorator(object.getClass(), context).toString(object);
    }

    //FIXME-TC Is this working ?
    protected void test(final Runnable r) throws InterruptedException {
        SwingUtilities.invokeLater(new Runnable() {

            public void run() {
                System.out.println("run");
//                getContext().getActionUI().pause();
                r.run();
//                getContext().getActionUI().resume();

                System.out.println("Deblock ?");
                log.info("Deblock ?");
                unblock();
            }
        });

        log.info("Block ?");
        System.out.println("Block ?");
        block();

        System.out.println("Block !");
        log.info("Block !");
    }

    protected void block() {
        try {
            synchronized (this) {
                lock.wait();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    protected void unblock() {
        try {
            synchronized (this) {
                lock.notifyAll();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}
