package fr.ifremer.tutti.ui.swing.content;

/*
 * #%L
 * Tutti :: UI
 * $Id: MainUIHandler.java 642 2013-03-18 11:18:53Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.1/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/MainUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import fr.ifremer.tutti.persistence.entities.data.Cruise;
import fr.ifremer.tutti.persistence.entities.data.Program;
import fr.ifremer.tutti.persistence.entities.protocol.TuttiProtocol;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.ui.swing.RunTutti;
import fr.ifremer.tutti.ui.swing.TuttiScreen;
import fr.ifremer.tutti.ui.swing.TuttiUIContext;
import fr.ifremer.tutti.ui.swing.config.TuttiApplicationConfig;
import fr.ifremer.tutti.ui.swing.content.config.TuttiConfigUI;
import fr.ifremer.tutti.ui.swing.content.cruise.EditCruiseUI;
import fr.ifremer.tutti.ui.swing.content.cruise.EditCruiseUIHandler;
import fr.ifremer.tutti.ui.swing.content.db.DbManagerUI;
import fr.ifremer.tutti.ui.swing.content.db.OpenDbAction;
import fr.ifremer.tutti.ui.swing.content.home.SelectCruiseUI;
import fr.ifremer.tutti.ui.swing.content.operation.FishingOperationsUI;
import fr.ifremer.tutti.ui.swing.content.program.EditProgramUI;
import fr.ifremer.tutti.ui.swing.content.program.EditProgramUIHandler;
import fr.ifremer.tutti.ui.swing.content.protocol.EditProtocolUI;
import fr.ifremer.tutti.ui.swing.content.protocol.EditProtocolUIHandler;
import fr.ifremer.tutti.ui.swing.content.referential.ManageTemporaryReferentialUI;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiUIHandler;
import fr.ifremer.tutti.ui.swing.util.CloseableUI;
import fr.ifremer.tutti.ui.swing.util.RemoveablePropertyChangeListener;
import fr.ifremer.tutti.ui.swing.util.TuttiUI;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;
import fr.ifremer.tutti.ui.swing.util.action.TuttiActionUI;
import fr.ifremer.tutti.ui.swing.util.action.TuttiUIAction;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.swing.AboutPanel;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.i18n.I18n;
import org.nuiton.util.config.ApplicationConfig;
import org.nuiton.widget.SwingSession;

import javax.swing.Icon;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JEditorPane;
import javax.swing.JLabel;
import javax.swing.JScrollPane;
import javax.swing.JToolBar;
import java.awt.Cursor;
import java.beans.PropertyChangeEvent;
import java.net.URL;
import java.util.Calendar;
import java.util.Locale;

import static org.nuiton.i18n.I18n._;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class MainUIHandler extends AbstractTuttiUIHandler<TuttiUIContext, MainUI> {

    /** Logger. */
    private static final Log log = LogFactory.getLog(MainUIHandler.class);

    protected JComponent currentBody;

    protected PersistenceService persistenceService;

    protected MainUIHandler(TuttiUIContext context, MainUI ui) {
        super(context, ui);
        this.persistenceService = context.getPersistenceService();
        context.setMainUI(ui);
        context.setActionUI(new TuttiActionUI(ui, context));
    }

    public MainUIHandler(TuttiUIContext context) {
        super(context, null);
        this.persistenceService = null;
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    public void beforeInitUI() {
        context.addPropertyChangeListener(new RemoveablePropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                String propertyName = evt.getPropertyName();
                if (TuttiUIContext.PROPERTIES_TO_SAVE.contains(propertyName)) {

                    // change the ui title
                    changeTitle();

                } else if (propertyName.equals(TuttiUIContext.PROPERTY_SCREEN)) {
                    setScreen((TuttiScreen) evt.getNewValue());
                }
            }
        });
        ui.setContextValue(ui, MainUI.class.getName());

        // ecoute des changements de l'état busy
        context.addPropertyChangeListener(TuttiUIContext.PROPERTY_BUSY, new RemoveablePropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                Boolean newvalue = (Boolean) evt.getNewValue();
                updateBusyState(newvalue != null && newvalue);
            }
        });

        // ecoute des changements de l'état busy
        context.addPropertyChangeListener(TuttiUIContext.PROPERTY_HIDE_BODY, new RemoveablePropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                Boolean newvalue = (Boolean) evt.getNewValue();
                getUI().getBody().setVisible(newvalue != null && newvalue);
            }
        });
    }

    protected void updateBusyState(boolean busy) {
        if (busy) {
            // ui bloquee
            if (log.isDebugEnabled()) {
                log.debug("block ui in busy mode");
            }
            ui.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
        } else {
            // ui debloquee
            if (log.isDebugEnabled()) {
                log.debug("unblock ui in none busy mode");
            }
            ui.setCursor(Cursor.getDefaultCursor());
        }
    }

    @Override
    public void afterInitUI() {

        initUI(ui);

        // installation layer de blocage en mode busy
        SwingUtil.setLayerUI(ui.getBody(), ui.getBusyBlockLayerUI());

        //FIXME-TC See why this binding is not setted ?
        ui.applyDataBinding(MainUI.BINDING_MENU_CHANGE_LOCALE_FR_ENABLED);
        ui.applyDataBinding(MainUI.BINDING_MENU_CHANGE_LOCALE_UK_ENABLED);

        // Init SwingSession
        SwingSession swingSession = context.getSwingSession();
        swingSession.add(ui);
        swingSession.save();

        //SwingUtil.getLayer(ui.getBody()).setUI(new BetaLayerUI());

        changeTitle();

        JToolBar bar = ui.getBottomBar();
        ui.getStatus().addWidget(bar, 0);

        if (context.isDbLoaded()) {

            // db already opened (happens when reloading ui)
            // just go to select cruise screen
            TuttiUIAction<SelectCruiseAction> uiAction = createUIAction(null, SelectCruiseAction.class);
            SelectCruiseAction logicAction = uiAction.getLogicAction();
            logicAction.setSkipCheckCurrentScreen(true);
            logicAction.setActionDescription(ui.getMenuActionSelectCruise().getToolTipText());
            uiAction.actionPerformed(null);

        } else {

            if (context.isDbExist()) {

                // open tutti db (using a fake button to have simple api)
                TuttiUIAction<OpenDbAction> uiAction = createUIAction(null, OpenDbAction.class);
                OpenDbAction logicAction = uiAction.getLogicAction();
                logicAction.setSkipCheckCurrentScreen(true);
                uiAction.actionPerformed(null);
            } else {

                // clean db context
                context.clearDbContext();

                // go to manage db screen (to install db)
                context.setScreen(TuttiScreen.MANAGE_DB);
            }
        }
    }

    @Override
    public void onCloseUI() {

        // remove any screen
        context.setScreen(null);
        context.removeMessageNotifier(this);

        TuttiUIUtil.closeBean(context);
        TuttiUIUtil.closeUI(ui);
        ui.setVisible(false);
        ui.dispose();
    }

    @Override
    public SwingValidator<TuttiUIContext> getValidator() {
        return null;
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    public void reloadUI() {

        //close ui
        getUI().getHandler().onCloseUI();

        // restart ui
        RunTutti.startTutti(getContext(), false);
    }

    public boolean acceptLocale(Locale l, String expected) {
        return l != null && l.toString().equals(expected);
    }

    public void changeLocale(Locale locale) {

        // change locale (and save configuration)
        getModel().setLocale(locale);

        // change i18n locale
        I18n.setDefaultLocale(getConfig().getI18nLocale());

        // reload decorator service (TODO Check if this is necessary)
        getContext().reloadDecoratorService();

        // close reload
        reloadUI();
    }

    public void showAbout() {

        ApplicationConfig applicationConfig =
                context.getConfig().getApplicationConfig();

        String iconPath = applicationConfig.getOption("application.icon.path");
        String name = "tutti";
        String licensePath = "META-INF/" + name + "-LICENSE.txt";
        String thirdPartyPath = "META-INF/" + name + "-THIRD-PARTY.txt";

        AboutPanel about = new AboutPanel();
        about.setTitle(_("tutti.about.title"));
        about.setAboutText(_("tutti.about.message"));

        TuttiApplicationConfig config = context.getConfig();
        int currentYear = Calendar.getInstance().get(Calendar.YEAR);
        int inceptionYear = config.getInceptionYear();
        String years;
        if (currentYear != inceptionYear) {
            years = inceptionYear + "-" + currentYear;
        } else {
            years = inceptionYear + "";
        }

        about.setBottomText(_("tutti.about.bottomText",
                              config.getOrganizationName(),
                              years,
                              config.getVersion()));
        about.setIconPath(iconPath);
        about.setLicenseFile(licensePath);
        about.setThirdpartyFile(thirdPartyPath);
        about.buildTopPanel();

        JScrollPane component = new JScrollPane();
        JEditorPane translateArea = new JEditorPane();
        translateArea.setContentType("text/html");
        translateArea.setEditable(false);
        if (translateArea.getFont() != null) {
            translateArea.setFont(translateArea.getFont().deriveFont((float) 11));
        }
        translateArea.setBorder(null);
        String text = _("tutti.about.translate.content");
        translateArea.setText(text);
        component.getViewport().add(translateArea);
        about.getTabs().add(_("tutti.about.translate.title"), component);
        about.init();
        about.showInDialog(ui, true);

        // register on swing session
        context.getSwingSession().add(about);
    }

    public void gotoSite() {
        TuttiApplicationConfig config = context.getConfig();

        URL siteURL = config.getSiteUrl();

        if (log.isDebugEnabled()) {
            log.debug("goto " + siteURL);
        }
        TuttiUIUtil.openLink(siteURL);
    }

    public void showHelp() {
        getModel().showHelp(ui, ui.getBroker(), null);
    }

    @Override
    public final void showInformationMessage(String message) {
        ui.getStatus().setStatus("<html><body>" + message + "</body></html>");
    }

    public void registerValidator(SwingValidator validator) {
        ui.getValidatorMessageWidget().registerValidator(validator);
    }

    @Override
    public void clearValidators() {
        ui.getValidatorMessageWidget().clearValidators();
    }

    public boolean quitCurrentScreen() {

        boolean canClose;
        if (context.getScreen() == null) {

            // no screen, surely can quit
            canClose = true;
            if (log.isWarnEnabled()) {
                log.warn("==================================================");
                log.warn("No screen, Should then skipCheckCurrent in action.");
                log.warn("==================================================");
            }
        } else {
            TuttiUI<?, ?> body = (TuttiUI<?, ?>) currentBody;
            Preconditions.checkNotNull(currentBody);
            AbstractTuttiUIHandler<?, ?> handler = body.getHandler();
            if (handler instanceof CloseableUI) {

                // try to quit UI
                canClose = ((CloseableUI) handler).quitUI();
            } else {

                // can always close ui
                canClose = true;
            }
        }
        return canClose;
    }

    public AbstractTuttiUIHandler getCurrentHandler() {
        TuttiUI<?, ?> body = (TuttiUI<?, ?>) currentBody;
        return body.getHandler();
    }

    /**
     * Reload the persistence service and set the new ne into the handler.
     *
     * @since 1.0
     */
    public void reloadPersistenceService() {
        this.persistenceService = context.reloadPersistenceService();
    }

    /**
     * Get the persistence service instance used by the handler.
     *
     * @return the persistence service instance used by the handler
     * @since 1.0
     */
    public PersistenceService getPersistenceService() {
        return persistenceService;
    }

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected void setScreen(TuttiScreen screen) {

        // close current body (if any)
        if (currentBody != null) {
            TuttiUI<?, ?> body = (TuttiUI<?, ?>) currentBody;
            body.getHandler().onCloseUI();

            context.getSwingSession().save();

            ui.getBody().remove(currentBody);

            currentBody = null;
        }

        if (screen != null) {

            // load new body

            JComponent screenUI;
            JToolBar rightDecoration = null;
            String screenTitle;

            Icon icon;
            switch (screen) {
                case CONFIG:

                    screenUI = new TuttiConfigUI(ui);
                    screenTitle = _("tutti.config.title");
                    icon = ui.getMenuFileConfiguration().getIcon();
                    break;

                default:
                case MANAGE_DB:

                    screenUI = new DbManagerUI(ui);
                    screenTitle = _("tutti.dbMabager.title");
                    icon = ui.getMenuActionManageDb().getIcon();
                    break;

                case SELECT_CRUISE:

                    screenUI = new SelectCruiseUI(ui);
                    screenTitle = _("tutti.selectCruise.title");
                    icon = ui.getMenuActionSelectCruise().getIcon();
                    break;

                case EDIT_PROGRAM:

                    screenTitle = EditProgramUIHandler.getTitle(
                            context.isProgramFilled());

                    screenUI = new EditProgramUI(ui);
                    icon = ui.getMenuActionEditProgram().getIcon();
                    break;

                case EDIT_CRUISE:

                    screenTitle = EditCruiseUIHandler.getTitle(
                            context.isCruiseFilled());

                    screenUI = new EditCruiseUI(ui);
                    rightDecoration = ((EditCruiseUI) screenUI).getTopToolBar();
                    icon = ui.getMenuActionEditCruise().getIcon();
                    break;

                case EDIT_PROTOCOL:

                    screenTitle = EditProtocolUIHandler.getTitle(
                            context.isProtocolFilled());

                    screenUI = new EditProtocolUI(ui);
                    icon = ui.getMenuActionEditProtocol().getIcon();
                    break;

                case EDIT_FISHING_OPERATION:

                    String validationContext = context.getValidationContext();

                    if (TuttiUIContext.VALIDATION_CONTEXT_EDIT.equals(validationContext)) {

                        // edit catches
                        screenTitle = _("tutti.fishingOperations.title.edit.operations", getSelectedCruiseTitle());
                        icon = ui.getMenuActionEditCatches().getIcon();
                    } else {

                        // validate catches
                        screenTitle = _("tutti.fishingOperations.title.validate.operations", getSelectedCruiseTitle());
                        icon = ui.getMenuActionValidateCatches().getIcon();
                    }
                    screenUI = new FishingOperationsUI(ui);

                    break;

                case IMPORT_TEMPORARY_REFERENTIAL:
                    screenTitle = _("tutti.manageTemporaryReferential.title");
                    screenUI = new ManageTemporaryReferentialUI(ui);
                    icon = ui.getMenuImportTemporaryReferential().getIcon();
                    break;
            }

            JButton showHelp = ui.getShowHelp();
            if (rightDecoration == null) {
                rightDecoration = new JToolBar();
                rightDecoration.setFloatable(false);
                rightDecoration.setOpaque(false);
                rightDecoration.setBorderPainted(false);
            } else {
                rightDecoration.remove(showHelp);
            }
            rightDecoration.add(showHelp, 0);
            this.currentBody = screenUI;
            context.getSwingSession().add(currentBody);
            ui.getBody().setTitle(screenTitle);
            ui.getBody().add(currentBody);
            ui.getBody().setLeftDecoration(new JLabel(icon));
            ui.getBody().setRightDecoration(rightDecoration);

        }
    }

    public void changeTitle() {

        String title = getSelectedCruiseTitle();

        ui.setTitle("Tutti - v " + getConfig().getVersion() + " [ " + title + " ]");
    }

    protected String getSelectedCruiseTitle() {


        String title;

        if (context.isDbLoaded()) {

            if (context.isProgramFilled()) {

                // selected program

                Program program = getDataContext().getProgram();

                title = _("tutti.main.title.selectedProgram", program.getName()) + " / ";

                if (context.isCruiseFilled()) {

                    // selected cruise
                    Cruise cruise;
                    cruise = getDataContext().getCruise();

                    if (cruise != null) {
                        title += _("tutti.main.title.selectedCruise", cruise.getName());
                    }
                } else {

                    // no selected cruise

                    title += _("tutti.main.title.noSelectedCruise");
                }

                title += " / ";

                if (context.isProtocolFilled()) {

                    // selected protocol
                    TuttiProtocol protocol = getDataContext().getProtocol();

                    title += _("tutti.main.title.selectedProtocol", protocol.getName());
                } else {

                    // no selected protocol

                    title += _("tutti.main.title.noSelectedProtocol");
                }

            } else {

                // no program selected (so neither cruise)

                title = _("tutti.main.title.noSelectedProgram");

            }
        } else {

            // no db loaded

            title = _("tutti.main.title.nodb");
        }

        return title;
    }

    public void setBodyTitle(String title) {
        ui.getBody().setTitle(title);
    }
}
