package fr.ifremer.tutti.ui.swing;

/*
 * #%L
 * Tutti :: UI
 * $Id: TuttiApplicationUpdaterCallBack.java 431 2013-02-15 20:01:16Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.0/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/TuttiApplicationUpdaterCallBack.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import fr.ifremer.tutti.persistence.ProgressionModel;
import fr.ifremer.tutti.persistence.service.synchro.ReferentialSynchronizeResult;
import fr.ifremer.tutti.service.TuttiTechnicalException;
import fr.ifremer.tutti.service.referential.TuttiReferentialSynchronizeService;
import fr.ifremer.tutti.ui.swing.util.action.TuttiActionUI;
import org.apache.commons.io.FileUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.ApplicationUpdater;
import org.nuiton.util.Version;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Map;

/**
 * CallBack to update jre, application, i18n or db.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.0
 */
public class TuttiApplicationUpdaterCallBack implements ApplicationUpdater.ApplicationUpdaterCallback {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(TuttiApplicationUpdaterCallBack.class);

    public enum UpdateType {
        JRE,
        TUTTI,
        I18N,
        DB
    }

    protected final TuttiUIContext context;

    protected List<UpdateType> types;

    protected ProgressionModel progressionModel;

    protected boolean dbUpdated;

    public TuttiApplicationUpdaterCallBack(TuttiUIContext context, ProgressionModel progressionModel) {
        this.context = context;
        this.progressionModel = progressionModel;
    }

    public void setTypes(UpdateType... types) {
        this.types = Lists.newArrayList(types);
    }

    public boolean isDbUpdated() {
        return dbUpdated;
    }

    @Override
    public Map<String, ApplicationUpdater.ApplicationInfo> updateToDo(Map<String, ApplicationUpdater.ApplicationInfo> appToUpdate) {
        Map<String, ApplicationUpdater.ApplicationInfo> result = Maps.newHashMap();

        for (UpdateType type : types) {
            ApplicationUpdater.ApplicationInfo info = getInfo(type, appToUpdate);
            if (info != null) {
                result.put(info.name, info);
                if (UpdateType.DB.equals(type)) {
                    // add auth
                    info.setAuthentication(
                            "t" + "u" + "t" + "t" + "i" + "d" + "b",
                            new char[]{'3', 'R', 'A', 'C', 'e', 'v', '2', 'N', 'w', '8', 'u', 'E', 'D', 'E', 'v', 's', 'y', '0', 'v', 'J'}
                    );
                }
            }
        }
        if (types.contains(UpdateType.DB)) {

            ApplicationUpdater.ApplicationInfo info = getInfo(UpdateType.DB, result);
            if (info == null) {
                dbUpdated = false;
            } else {
                dbUpdated = true;
            }
        }
        return result;
    }

    public void startUpdate(ApplicationUpdater.ApplicationInfo info) {
        if (UpdateType.DB.name().toLowerCase().equals(info.name)) {

            if (Version.VZERO.toString().equals(info.oldVersion)) {

                progressionModel.setMessage("Téléchargement et installation de la base (version " + info.newVersion + ")");
            } else {
                progressionModel.setMessage("Téléchargement et mise à jour de la base (version " + info.newVersion + ")");
            }
        }
        if (UpdateType.JRE.name().toLowerCase().equals(info.name)) {
            progressionModel.setMessage("Téléchargement et installation d'une nouvelle JRE (version " + info.newVersion + ")");
        }

        if (UpdateType.TUTTI.name().toLowerCase().equals(info.name)) {
            progressionModel.setMessage("Téléchargement et installation d'une nouvelle version de Tutti (version " + info.newVersion + ")");
        }

        if (UpdateType.I18N.name().toLowerCase().equals(info.name)) {
            progressionModel.setMessage("Téléchargement et installation d'une nouveau version I18N (version " + info.newVersion + ")");
        }
    }

    @Override
    public void updateDone(Map<String, ApplicationUpdater.ApplicationInfo> appToUpdate,
                           Map<String, Exception> appUpdateError) {

        boolean doRestart = updateDoneJre(appToUpdate, appUpdateError);

        doRestart |= updateDoneTutti(appToUpdate, appUpdateError);
        doRestart |= updateDoneI18n(appToUpdate, appUpdateError);

        updateDoneDb(appToUpdate, appUpdateError);

        if (doRestart) {
            System.exit(RunTutti.UPATE_EXIT_CODE);
        }
    }

    @Override
    public void aborted(String propertiesURL, Exception eee) {
        if (log.isErrorEnabled()) {
            log.error("Could not update from " + propertiesURL, eee);
        }
    }

    protected boolean updateDoneJre(Map<String, ApplicationUpdater.ApplicationInfo> appToUpdate,
                                    Map<String, Exception> appUpdateError) {
        boolean doRestart = false;
        Exception error = getError(UpdateType.JRE, appUpdateError);
        if (error != null) {

            // something bad while updating jre
            if (log.isErrorEnabled()) {
                log.error("Could not update jre", error);
            }
        } else {
            ApplicationUpdater.ApplicationInfo info = getInfo(UpdateType.JRE, appToUpdate);
            if (info != null) {

                if (log.isInfoEnabled()) {
                    log.info(String.format(
                            "A jre update was downloaded (oldVersion: %s, newVersion: %s), will restart application to use it",
                            info.oldVersion, info.newVersion));
                }
                doRestart = true;
            }
        }
        return doRestart;
    }

    protected boolean updateDoneTutti(Map<String, ApplicationUpdater.ApplicationInfo> appToUpdate,
                                      Map<String, Exception> appUpdateError) {
        boolean doRestart = false;
        Exception error = getError(UpdateType.TUTTI, appUpdateError);
        if (error != null) {

            // something bad while updating application
            if (log.isErrorEnabled()) {
                log.error("Could not update tutti", error);
            }
        } else {
            ApplicationUpdater.ApplicationInfo info = getInfo(UpdateType.TUTTI, appToUpdate);
            if (info != null) {

                if (log.isInfoEnabled()) {
                    log.info(String.format(
                            "A tutti update was downloaded (oldVersion: %s, newVersion: %s), will restart application to use it",
                            info.oldVersion, info.newVersion));
                }
                doRestart = true;
            }
        }
        return doRestart;
    }

    protected boolean updateDoneI18n(Map<String, ApplicationUpdater.ApplicationInfo> appToUpdate,
                                     Map<String, Exception> appUpdateError) {
        boolean doRestart = false;
        Exception error = getError(UpdateType.I18N, appUpdateError);
        if (error != null) {

            // something bad while updating i18n
            if (log.isErrorEnabled()) {
                log.error("Could not update i18n", error);
            }
        } else {
            ApplicationUpdater.ApplicationInfo info = getInfo(UpdateType.I18N, appToUpdate);
            if (info != null) {

                if (log.isInfoEnabled()) {
                    log.info(String.format(
                            "A i18n update was downloaded (oldVersion: %s, newVersion: %s), will restart application to use it",
                            info.oldVersion, info.newVersion));
                }
                doRestart = true;
            }
        }
        return doRestart;
    }

    protected void updateDoneDb(Map<String, ApplicationUpdater.ApplicationInfo> appToUpdate,
                                Map<String, Exception> appUpdateError) {
        Exception error = getError(UpdateType.DB, appUpdateError);
        if (error != null) {

            // something bad while updating db
            if (log.isErrorEnabled()) {
                log.error("Could not update db", error);
            }
        } else {
            ApplicationUpdater.ApplicationInfo info = getInfo(UpdateType.DB, appToUpdate);
            if (info != null) {

                if (log.isInfoEnabled()) {
                    log.info(String.format(
                            "A db update was downloaded (oldVersion: %s, newVersion: %s), will process it.",
                            info.oldVersion, info.newVersion));
                }
                if (Version.VZERO.toString().equals(info.oldVersion)) {

                    // first database, just copy it to correct directory

                    prepareFirstDatabase(info);
                } else {

                    // launch a referential synchronize operation
                    synchronizetDatabase(info);
                }
            }
        }
    }

    protected ApplicationUpdater.ApplicationInfo getInfo(UpdateType type,
                                                         Map<String, ApplicationUpdater.ApplicationInfo> appToUpdate) {
        return appToUpdate.get(type.name().toLowerCase());
    }

    protected Exception getError(UpdateType type,
                                 Map<String, Exception> appUpdateError) {
        return appUpdateError.get(type.name().toLowerCase());
    }

    protected File getDbDirectory(ApplicationUpdater.ApplicationInfo info) {
        File[] sources = info.destDir.listFiles();
        Preconditions.checkState(
                sources != null && sources.length == 1,
                "Downloaded db should contains one directory at " + info.destDir);
        File result = sources[0];
        return result;
    }

    protected void prepareFirstDatabase(ApplicationUpdater.ApplicationInfo info) {
        if (log.isInfoEnabled()) {
            log.info("First time database was downloaded at version: " + info.newVersion);
        }
        File source = getDbDirectory(info);
        File target = context.getConfig().getServiceConfig().getPersistenceConfig().getDbDirectory();
        if (log.isInfoEnabled()) {
            log.info("Copy from " + source + " to " + target);
        }
        try {
            FileUtils.copyDirectory(source, target);
        } catch (IOException e) {
            throw new RuntimeException("Could not copy directory content from " + source + " to " + target, e);
        }
        try {
            FileUtils.deleteDirectory(source.getParentFile());
        } catch (IOException e) {
            throw new RuntimeException("Could not delete temporary directory " + target, e);
        }
    }

    protected void synchronizetDatabase(ApplicationUpdater.ApplicationInfo info) {
        if (log.isInfoEnabled()) {
            log.info(String.format("A database update was downloaded (oldVersion: %s, newVersion: %s), will launch a referential synchronize operation ", info.oldVersion, info.newVersion));
        }
        TuttiReferentialSynchronizeService service = context.getService(TuttiReferentialSynchronizeService.class);
        ReferentialSynchronizeResult result = new ReferentialSynchronizeResult();
        File dbDirectory = getDbDirectory(info);

        TuttiActionUI actionUI = context.getActionUI();
        actionUI.getModel().setProgressionModel(result.getProgressionModel());
        service.prepare(dbDirectory, result);

        if (!result.isSuccess()) {
            throw new TuttiTechnicalException("Could not prepare synchro", result.getError());
        }

        service.synchronize(dbDirectory, result);
    }
}
