package fr.ifremer.tutti.ui.swing.util.editor;

/*
 * #%L
 * Tutti :: UI
 * $Id: LongTextCellComponent.java 512 2013-03-01 20:37:07Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.0.3/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/util/editor/LongTextCellComponent.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiBeanUIModel;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;
import fr.ifremer.tutti.ui.swing.util.table.AbstractSelectTableAction;
import fr.ifremer.tutti.ui.swing.util.table.AbstractTuttiTableModel;
import fr.ifremer.tutti.ui.swing.util.table.ColumnIdentifier;
import jaxx.runtime.JAXXUtil;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.swing.ComponentMover;
import jaxx.runtime.swing.ComponentResizer;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.swing.AbstractCellEditor;
import javax.swing.BorderFactory;
import javax.swing.JDialog;
import javax.swing.JTable;
import javax.swing.border.LineBorder;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;
import java.awt.Color;
import java.awt.Component;
import java.awt.Frame;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import static org.nuiton.i18n.I18n._;

/**
 * Component to edit a cell in a popup with a text area
 *
 * @author kmorin
 * @since 0.2
 */
public class LongTextCellComponent extends DefaultTableCellRenderer {

    /** Logger. */
    private static final Log log = LogFactory.getLog(LongTextCellComponent.class);

    private static final long serialVersionUID = 1L;

    public LongTextCellComponent() {
        setHorizontalAlignment(CENTER);
        setIcon(SwingUtil.createActionIcon("edit-comment"));
    }

    @Override
    protected void setValue(Object value) {
        // do nothing
    }

    public static TableCellRenderer newRender(String noneText) {
        return new LongTextCellRenderer(noneText);
    }

    public static TableCellEditor newEditor(LongTextEditorUI ui) {
        return new LongTextCellEditor(ui);
    }


    public static class LongTextCellEditor extends AbstractCellEditor implements TableCellEditor {

        public static final int DEFAULT_EDITOR_WIDTH = 400;

        public static final int DEFAULT_EDITOR_HEIGHT = 200;

        private static final long serialVersionUID = 1L;

        protected final LongTextCellComponent component;

        protected final LongTextEditorUI ui;

        protected Frame frame;

        protected JTable table;

        protected AbstractTuttiTableModel<AbstractTuttiBeanUIModel> tableModel;

        protected ColumnIdentifier<AbstractTuttiBeanUIModel> columnIdentifier;

        protected AbstractTuttiBeanUIModel editRow;

        protected Integer rowIndex;

        protected Integer columnIndex;

        public LongTextCellEditor(LongTextEditorUI ui) {
            this.ui = ui;

            component = new LongTextCellComponent();
            component.setBorder(new LineBorder(Color.BLACK));
            component.addKeyListener(new KeyAdapter() {
                @Override
                public void keyReleased(KeyEvent e) {
                    if (e.getKeyCode() == KeyEvent.VK_ENTER ||
                        e.getKeyCode() == KeyEvent.VK_SPACE) {
                        e.consume();
                        startEdit();
                    }
                }
            });

            component.addMouseListener(new MouseAdapter() {
                @Override
                public void mouseClicked(MouseEvent e) {
                    e.consume();
                    startEdit();
                }
            });
        }

        protected void startEdit() {
            if (frame == null) {
                frame = SwingUtil.getParentContainer(ui, Frame.class);
            }

            ui.setBorder(BorderFactory.createTitledBorder(_(columnIdentifier.getHeaderI18nKey())));
            ui.setBean(editRow);
            ui.setProperty(columnIdentifier.getPropertyName());

            JDialog editor = new JDialog(frame, true);
            editor.setUndecorated(true);
            editor.add(ui);
            editor.setSize(DEFAULT_EDITOR_WIDTH, DEFAULT_EDITOR_HEIGHT);
            editor.setResizable(true);
            ComponentResizer cr = new ComponentResizer();
            cr.registerComponent(editor);
            ComponentMover cm = new ComponentMover();
            cm.setDragInsets(cr.getDragInsets());
            cm.registerComponent(editor);

            editor.addWindowListener(new WindowAdapter() {

                @Override
                public void windowClosed(WindowEvent e) {
                    Component ui = (Component) e.getSource();
                    if (log.isInfoEnabled()) {
                        log.info("Destroy ui " + ui);
                    }
                    JAXXUtil.destroy(ui);
                }
            });

            // Computes the location of bottom left corner of the cell
            Component comp = component;
            int x = 0;
            int y = component.getHeight();
            while (comp != null) {
                x += comp.getX();
                y += comp.getY();
                comp = comp.getParent();
            }
            // if the editor is too big on the right,
            // then align its right side to the right side of the cell
            if (x + editor.getWidth() > frame.getX() + frame.getWidth()) {
                x = x - editor.getWidth() + component.getWidth();
            }
            editor.setLocation(x, y);
            editor.setVisible(true);

            int r = rowIndex;
            int c = columnIndex;

            // stop edition
            stopCellEditing();

            // reselect this cell
            AbstractSelectTableAction.doSelectCell(table, r, c);
            table.requestFocus();
        }

        @Override
        public Component getTableCellEditorComponent(JTable table,
                                                     Object value,
                                                     boolean isSelected,
                                                     int row,
                                                     int column) {
            if (tableModel == null) {
                tableModel = (AbstractTuttiTableModel<AbstractTuttiBeanUIModel>) table.getModel();
                this.table = table;
                columnIdentifier = tableModel.getPropertyName(column);
            }

            rowIndex = row;
            columnIndex = column;

            editRow = tableModel.getEntry(row);

            return component;
        }

        @Override
        public Object getCellEditorValue() {

            Preconditions.checkNotNull(editRow, "No editRow found in editor.");

            String propertyName = columnIdentifier.getPropertyName();
            Object result = TuttiUIUtil.getProperty(editRow, propertyName);
            if (log.isInfoEnabled()) {
                log.info("editor value (" + propertyName + "): " + result);
            }

            return result;
        }

        @Override
        public boolean stopCellEditing() {
            boolean b = super.stopCellEditing();
            if (b) {
                rowIndex = null;
                editRow = null;
                columnIndex = null;
            }
            return b;
        }

        @Override
        public void cancelCellEditing() {
            super.cancelCellEditing();
            rowIndex = null;
            columnIndex = null;
            editRow = null;
        }
    }

    public static class LongTextCellRenderer implements TableCellRenderer {

        protected final LongTextCellComponent component;

        protected String propertyName;

        public static final String TEXT_PATTERN = "<html><body>%s</body></html>";

        private final String noneText;

        public LongTextCellRenderer(String noneText) {
            component = new LongTextCellComponent();
            this.noneText = noneText;
        }

        @Override
        public Component getTableCellRendererComponent(JTable table,
                                                       Object value,
                                                       boolean isSelected,
                                                       boolean hasFocus,
                                                       int row,
                                                       int column) {

            LongTextCellComponent result =
                    (LongTextCellComponent) component.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);

            if (table != null) {
                String textValue;
                if (StringUtils.isBlank((String) value)) {
                    if (propertyName == null) {
                        AbstractTuttiTableModel tableModel =
                                (AbstractTuttiTableModel) table.getModel();
                        ColumnIdentifier columnIdentifier =
                                tableModel.getPropertyName(column);
                        propertyName = columnIdentifier.getPropertyName();
                    }
                    // use HTML to show the tooltip in italic
                    textValue = "<i>" + _(noneText) + "</i>";

                } else {
                    // use html to display the tooltip on several lines
                    textValue = String.valueOf(value).replace("\n", "<br/>");
                }

                String text = String.format(TEXT_PATTERN, textValue);

                boolean editable = table.isCellEditable(row, column);
                result.setEnabled(editable);
                result.setToolTipText(text);
            }

            return result;
        }
    }
}
