package fr.ifremer.tutti.ui.swing.util;

/*
 * #%L
 * Tutti :: UI
 * $Id: AbstractTuttiUIHandler.java 537 2013-03-05 12:28:40Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.0.2/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/util/AbstractTuttiUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import fr.ifremer.tutti.persistence.entities.IdAware;
import fr.ifremer.tutti.service.DecoratorService;
import fr.ifremer.tutti.ui.swing.TuttiDataContext;
import fr.ifremer.tutti.ui.swing.TuttiScreen;
import fr.ifremer.tutti.ui.swing.TuttiUIContext;
import fr.ifremer.tutti.ui.swing.config.TuttiApplicationConfig;
import fr.ifremer.tutti.ui.swing.content.AbstractMainUITuttiAction;
import fr.ifremer.tutti.ui.swing.content.MainUI;
import fr.ifremer.tutti.ui.swing.content.MainUIHandler;
import fr.ifremer.tutti.ui.swing.util.action.AbstractTuttiAction;
import fr.ifremer.tutti.ui.swing.util.action.TuttiUIAction;
import fr.ifremer.tutti.ui.swing.util.attachment.ButtonAttachmentEditor;
import fr.ifremer.tutti.ui.swing.util.editor.SimpleTimeEditor;
import jaxx.runtime.JAXXUtil;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.swing.editor.NumberEditor;
import jaxx.runtime.swing.editor.bean.BeanComboBox;
import jaxx.runtime.swing.editor.bean.BeanDoubleList;
import jaxx.runtime.swing.renderer.DecoratorListCellRenderer;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.reflect.ConstructorUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXDatePicker;
import org.nuiton.util.decorator.Decorator;
import org.nuiton.util.decorator.JXPathDecorator;
import org.nuiton.validator.bean.simple.SimpleBeanValidator;

import javax.swing.AbstractAction;
import javax.swing.AbstractButton;
import javax.swing.Action;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JRootPane;
import javax.swing.JTextField;
import javax.swing.KeyStroke;
import javax.swing.ListCellRenderer;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.text.JTextComponent;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Frame;
import java.awt.event.ActionEvent;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.ItemEvent;
import java.awt.event.KeyEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.Serializable;
import java.text.ParseException;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static org.nuiton.i18n.I18n._;

/**
 * Contract of any UI handler.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public abstract class AbstractTuttiUIHandler<M, UI extends TuttiUI<M, ?>> implements UIMessageNotifier {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(AbstractTuttiUIHandler.class);

    public abstract void beforeInitUI();

    public abstract void afterInitUI();

    public abstract void onCloseUI();

    public abstract SwingValidator<M> getValidator();

    /**
     * Global application context.
     *
     * @since 0.1
     */
    protected final TuttiUIContext context;

    /**
     * UI handled.
     *
     * @since 0.1
     */
    protected final UI ui;

    protected AbstractTuttiUIHandler(TuttiUIContext context, UI ui) {
        this.context = context;
        this.ui = ui;
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    public DefaultComboBoxModel newComboModel(Object... items) {
        return new DefaultComboBoxModel(items);
    }

    public DefaultComboBoxModel newComboActionModel(Class<?>... actionNames) {
        List<Object> items = Lists.newArrayListWithCapacity(actionNames.length);
        for (Class<?> actionName : actionNames) {
            Action action = createAction((Class<AbstractTuttiAction>) actionName);
            items.add(action);
        }
        return newComboModel(items.toArray());
    }

    public final M getModel() {
        return ui.getModel();
    }

    public final UI getUI() {
        return ui;
    }

    @Override
    public void showInformationMessage(String message) {
        context.showInformationMessage(message);
    }

    public TuttiUIContext getContext() {
        return context;
    }

    public TuttiDataContext getDataContext() {
        return getContext().getDataContext();
    }

    public TuttiApplicationConfig getConfig() {
        return context.getConfig();
    }

    /**
     * Can the UI be closed? It is useful whe the user wants to exit the current
     * screen but the model is modified: we can then ask the user if he wants to
     * save or not.
     *
     * @param nextScreen the next screen to display
     * @return {@code true} if UI can be closed, {@code false} otherwise.
     */
    public boolean canCloseUI(TuttiScreen nextScreen) {
        return true;
    }

    public void setText(KeyEvent event, String property) {
        JTextComponent field = (JTextComponent) event.getSource();
        String value = field.getText();
        TuttiUIUtil.setProperty(getModel(), property, value);
    }

    public void setBoolean(ItemEvent event, String property) {
        boolean value = event.getStateChange() == ItemEvent.SELECTED;
        TuttiUIUtil.setProperty(getModel(), property, value);
    }

    public void setDate(ActionEvent event, String property) {
        JXDatePicker field = (JXDatePicker) event.getSource();
        Date value = field.getDate();
        TuttiUIUtil.setProperty(getModel(), property, value);
    }

    public void selectListData(ListSelectionEvent event, String property) {
        if (!event.getValueIsAdjusting()) {
            JList list = (JList) event.getSource();
            ListSelectionModel selectionModel = list.getSelectionModel();

            selectionModel.setValueIsAdjusting(true);
            try {
                List selectedList = Lists.newLinkedList();

                for (int index : list.getSelectedIndices()) {
                    Object o = list.getModel().getElementAt(index);
                    selectedList.add(o);
                }
                TuttiUIUtil.setProperty(getModel(), property, selectedList);
            } finally {
                selectionModel.setValueIsAdjusting(false);
            }
        }
    }

    public void openDialog(TuttiUI ui,
                           TuttiUI dialogContent,
                           String title, Dimension dim) {
        Frame frame = SwingUtil.getParentContainer(ui, Frame.class);

        JDialog result = new JDialog(frame, true);
        result.setTitle(title);
        result.add((Component) dialogContent);
        result.setResizable(true);

        result.setSize(dim);

        final AbstractTuttiUIHandler handler = dialogContent.getHandler();

        if (handler instanceof Cancelable) {

            // add a auto-close action
            JRootPane rootPane = result.getRootPane();

            KeyStroke shortcutClosePopup = getConfig().getShortcutClosePopup();

            rootPane.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(
                    shortcutClosePopup, "close");
            rootPane.getActionMap().put("close", new AbstractAction() {
                private static final long serialVersionUID = 1L;

                @Override
                public void actionPerformed(ActionEvent e) {
                    ((Cancelable) handler).cancel();
                }
            });
        }

        result.addWindowListener(new WindowAdapter() {

            @Override
            public void windowClosed(WindowEvent e) {
                Component ui = (Component) e.getSource();
                if (log.isInfoEnabled()) {
                    log.info("Destroy ui " + ui);
                }
                JAXXUtil.destroy(ui);
            }
        });
        SwingUtil.center(frame, result);
        result.setVisible(true);
    }

    public void closeDialog(TuttiUI ui) {
        SwingUtil.getParentContainer(ui, JDialog.class).setVisible(false);
    }

    public int askSaveBeforeLeaving(String message) {
        int result = JOptionPane.showConfirmDialog(
                getContext().getMainUI(),
                message,
                _("tutti.dialog.askSaveBeforeLeaving.title"),
                JOptionPane.YES_NO_CANCEL_OPTION);
        return result;
    }

    public boolean askCancelEditBeforeLeaving(String message) {
        boolean result = TuttiUIUtil.askQuestion(
                getContext().getMainUI(),
                message,
                _("tutti.dialog.askCancelEditBeforeLeaving.title"));
        return result;
    }
    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected void initUI(TuttiUI ui) {

        for (Map.Entry<String, Object> entry : ui.get$objectMap().entrySet()) {
            Object component = entry.getValue();
            if (component instanceof NumberEditor) {
                initNumberEditor((NumberEditor) component);

            } else if (component instanceof JXDatePicker) {
                initDatePicker((JXDatePicker) component);

            } else if (component instanceof SimpleTimeEditor) {
                initTimeEditor((SimpleTimeEditor) component);

            } else if (component instanceof ButtonAttachmentEditor) {
                initButtonAttachmentEditor((ButtonAttachmentEditor) component);

            } else if (component instanceof JLabel) {
                JLabel jLabel = (JLabel) component;
                Boolean strongStyle = (Boolean) jLabel.getClientProperty("strongStyle");
                Boolean italicStyle = (Boolean) jLabel.getClientProperty("italicStyle");
                boolean addHtml = strongStyle != null && strongStyle || italicStyle != null && italicStyle;
                if (addHtml) {
                    String text = jLabel.getText();
                    if (strongStyle != null && strongStyle) {
                        text = "<strong>" + text + "</strong>";
                    }
                    if (italicStyle != null && italicStyle) {
                        text = "<em>" + text + "</em>";
                    }
                    jLabel.setText("<html>" + text + "</html>");
                }

            } else if (component instanceof JTextField) {
                JTextField jTextField = (JTextField) component;
                Boolean computed = (Boolean) jTextField.getClientProperty("computed");
                if (computed != null && computed) {
                    Font font = jTextField.getFont().deriveFont(Font.ITALIC);
                    jTextField.setFont(font);
                    jTextField.setEditable(!computed);
                    jTextField.setEnabled(!computed);
                    jTextField.setDisabledTextColor(getConfig().getColorComputedWeights());
                }

            } else if (component instanceof AbstractButton) {
                AbstractButton abstractButton = (AbstractButton) component;
                Class<AbstractTuttiAction> actionName = (Class<AbstractTuttiAction>) abstractButton.getClientProperty("tuttiAction");
                if (actionName != null) {
                    initAction(abstractButton, actionName);
                }
            }
        }
    }

    protected void initButtonAttachmentEditor(ButtonAttachmentEditor component) {

        component.init();
    }

    protected <A extends AbstractTuttiAction> void initAction(AbstractButton abstractButton,
                                                              Class<A> actionName) {

        Action action = createAction(actionName);
        abstractButton.setAction(action);
    }

    protected <A extends AbstractTuttiAction> TuttiUIAction<A> createAction(Class<A> actionName) {
        TuttiUIAction result = null;
        if (actionName != null) {
            try {

                AbstractTuttiUIHandler handler = this;

                if (AbstractMainUITuttiAction.class.isAssignableFrom(actionName) &&
                    getContext().getMainUI() != null) {
                    handler = getContext().getMainUI().getHandler();
                }

                // create action
                AbstractTuttiAction action = ConstructorUtils.invokeConstructor(actionName, handler);

                // create ui action
                result = new TuttiUIAction(action);

            } catch (Exception e) {
                throw new RuntimeException(
                        "Could not instanciate action " + actionName, e);
            }
        }
        return result;
    }

    protected void doAction(AbstractButton button, ActionEvent event) {
        button.getAction().actionPerformed(event);
    }

    protected void registerValidators(SwingValidator... validators) {
        MainUI main = context.getMainUI();
        Preconditions.checkNotNull(
                main, "No mainUI registred in application context");
        MainUIHandler handler = main.getHandler();
        handler.clearValidators();
        for (SwingValidator validator : validators) {
            handler.registerValidator(validator);
        }
    }

    public void clearValidators() {
        MainUI main = context.getMainUI();
        Preconditions.checkNotNull(
                main, "No mainUI registred in application context");
        MainUIHandler handler = main.getHandler();
        handler.clearValidators();
    }


    /**
     * Prépare un component de choix d'entités pour un type d'entité donné et
     * pour un service de persistance donné.
     *
     * @param comboBox le component graphique à initialiser
     */
    protected <E extends Serializable> void initBeanComboBox(
            BeanComboBox<E> comboBox,
            List<E> data,
            E selectedData) {

        initBeanComboBox(comboBox, data, selectedData, null);
    }

    protected <E extends Serializable> void initBeanComboBox(
            BeanComboBox<E> comboBox,
            List<E> data,
            E selectedData,
            String decoratorContext) {

        Preconditions.checkNotNull(comboBox, "No comboBox!");

        Class<E> beanType = comboBox.getBeanType();

        Preconditions.checkNotNull(beanType, "No beanType on the combobox!");

        Decorator<E> decorator = getDecorator(beanType, decoratorContext);

        if (data == null) {
            data = Lists.newArrayList();
        }

        if (log.isInfoEnabled()) {
            log.info("entity comboBox list [" + beanType.getName() + "] : " +
                     (data == null ? 0 : data.size()));
        }

        // add data list to combo box
        comboBox.init((JXPathDecorator<E>) decorator, data);

        comboBox.setSelectedItem(selectedData);

        if (log.isDebugEnabled()) {
            log.debug("combo [" + beanType.getName() + "] : " +
                      comboBox.getData().size());
        }
    }

    /**
     * Prépare un component de choix d'entités pour un type d'entité donné et
     * pour un service de persistance donné.
     *
     * @param list         le component graphique à initialiser
     * @param data         la liste des données à mettre dans la liste de gauche
     * @param selectedData la liste des données à mettre dans la liste de droite
     */
    protected <E extends IdAware> void initBeanList(
            BeanDoubleList<E> list,
            List<E> data,
            List<E> selectedData) {

        Preconditions.checkNotNull(list, "No list!");

        Class<E> beanType = list.getBeanType();
        Preconditions.checkNotNull(beanType, "No beanType on the double list!");

        DecoratorService decoratorService =
                context.getDecoratorService();
        Decorator<E> decorator = decoratorService.getDecoratorByType(beanType);

        if (log.isInfoEnabled()) {
            log.info("entity list [" + beanType.getName() + "] : " +
                     (data == null ? 0 : data.size()));
        }

        // add data list to combo box
        list.init((JXPathDecorator<E>) decorator, data, selectedData);

        if (log.isDebugEnabled()) {
            log.debug("Jlist [" + beanType.getName() + "] : " +
                      list.getUniverseList().getModel().getSize());
        }
    }

    protected void initNumberEditor(NumberEditor editor) {
        if (log.isDebugEnabled()) {
            log.debug("init number editor " + editor.getName());
        }
        editor.init();

        // Force binding if value is already in model
        Number model = editor.getModel();
        if (model != null) {
            editor.setModel(null);
            editor.setModel(model);
        }
    }

    protected void initTimeEditor(SimpleTimeEditor editor) {
        if (log.isDebugEnabled()) {
            log.debug("init time editor " + editor.getName() +
                      " for property " + editor.getModel().getProperty());
        }
        editor.init();
    }

    protected void initDatePicker(final JXDatePicker picker) {

        if (log.isDebugEnabled()) {
            log.debug("disable JXDatePicker editor" + picker.getName());
        }
        String dateFormat = getConfig().getDateFormat();
        picker.setFormats(dateFormat);
        picker.getEditor().addFocusListener(new FocusAdapter() {

            @Override
            public void focusLost(FocusEvent e) {
                try {
                    picker.commitEdit();

                } catch (ParseException ex) {
                    if (log.isDebugEnabled()) {
                        log.debug("format error", ex);
                    }
                }
            }

        });
    }

    public <O> Decorator<O> getDecorator(Class<O> type, String name) {
        DecoratorService decoratorService =
                context.getDecoratorService();

        Preconditions.checkNotNull(type);

        Decorator decorator = decoratorService.getDecoratorByType(type, name);
        if (decorator == null) {

            if (DecoratorService.LabelAware.class.isAssignableFrom(type)) {
                decorator = getDecorator(DecoratorService.LabelAware.class, null);
            }
        }
        Preconditions.checkNotNull(decorator);
        return decorator;
    }

    protected String decorate(Object object) {
        String result = "";
        if (object != null) {
            getDecorator(object.getClass(), null).toString(object);
        }
        return result;
    }

    protected <O> ListCellRenderer newListCellRender(Class<O> type) {

        return newListCellRender(type, null);
    }

    protected <O> ListCellRenderer newListCellRender(Class<O> type, String name) {

        Decorator<O> decorator = getDecorator(type, name);
        return newListCellRender(decorator);
    }

    protected <O> ListCellRenderer newListCellRender(Decorator<O> decorator) {

        Preconditions.checkNotNull(decorator);

        ListCellRenderer result = new DecoratorListCellRenderer(decorator);
        return result;
    }

    protected void listenValidatorValid(SimpleBeanValidator validator,
                                        final AbstractTuttiBeanUIModel model) {
        validator.addPropertyChangeListener(SimpleBeanValidator.VALID_PROPERTY, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                if (log.isDebugEnabled()) {
                    log.debug("Model [" + model +
                              "] pass to valid state [" +
                              evt.getNewValue() + "]");
                }
                model.setValid((Boolean) evt.getNewValue());
            }
        });
    }

    protected void listModelIsModify(AbstractTuttiBeanUIModel model) {
        model.addPropertyChangeListener(new PropertyChangeListener() {

            final Set<String> excludeProperties = getPropertiesToIgnore();

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                if (!excludeProperties.contains(evt.getPropertyName())) {
                    ((AbstractTuttiBeanUIModel) evt.getSource()).setModify(true);
                }
            }
        });
    }

    protected Set<String> getPropertiesToIgnore() {
        return Sets.newHashSet(
                AbstractTuttiBeanUIModel.PROPERTY_MODIFY,
                AbstractTuttiBeanUIModel.PROPERTY_VALID);
    }

    public <B> void selectFirstInCombo(BeanComboBox<B> combo) {
        List<B> data = combo.getData();
        B selectedItem = null;
        if (CollectionUtils.isNotEmpty(data)) {
            selectedItem = data.get(0);
        }
        combo.setSelectedItem(selectedItem);
    }

    protected void closeUI(TuttiUI ui) {
        ui.getHandler().onCloseUI();
    }

    protected <B> void changeValidatorContext(String newContext,
                                              SwingValidator<B> validator) {
        B bean = validator.getBean();
        validator.setContext(newContext);
        validator.setBean(bean);
    }
}
