package fr.ifremer.tutti.ui.swing.content.operation.fishing.hydrology;

/*
 * #%L
 * Tutti :: UI
 * $Id: HydrologyTabUIHandler.java 195 2013-01-14 17:46:19Z kmorin $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-0.3/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/fishing/hydrology/HydrologyTabUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import fr.ifremer.tutti.persistence.entities.CaracteristicMap;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.data.FishingOperation;
import fr.ifremer.tutti.persistence.entities.protocol.TuttiProtocol;
import fr.ifremer.tutti.persistence.entities.referential.Caracteristic;
import fr.ifremer.tutti.service.DecoratorService;
import fr.ifremer.tutti.ui.swing.content.operation.EditFishingOperationUI;
import fr.ifremer.tutti.ui.swing.util.HydrologicCaracteristicUtil;
import fr.ifremer.tutti.ui.swing.util.HydrologicCaracteristicUtil.Type;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import fr.ifremer.tutti.ui.swing.util.editor.CaracteristicValueEditor;
import fr.ifremer.tutti.ui.swing.util.editor.CaracteristicValueRenderer;
import fr.ifremer.tutti.ui.swing.util.table.AbstractTuttiTableUIHandler;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.Serializable;
import java.util.List;
import java.util.Map;
import java.util.Set;
import jaxx.runtime.swing.editor.bean.BeanComboBox;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.table.DefaultTableColumnModelExt;

/**
 * @author kmorin
 * @since 0.3
 */
public class HydrologyTabUIHandler extends AbstractTuttiTableUIHandler<HydrologyRowModel, HydrologyTabUIModel> {

    private final static Log log =
            LogFactory.getLog(HydrologyTabUIHandler.class);

    /**
     * UI.
     *
     * @since 0.3
     */
    protected final HydrologyTabUI ui;

    protected Map<Caracteristic, Map<Type, Caracteristic>> availableGlobalCaracteristics;

    public HydrologyTabUIHandler(EditFishingOperationUI parentUi,
                                 HydrologyTabUI ui) {
        super(parentUi.getHandler().getContext(),
              HydrologyRowModel.PROPERTY_GEAR_SHOOTING_START_VALUE,
              HydrologyRowModel.PROPERTY_AVERAGE_VALUE,
              HydrologyRowModel.PROPERTY_GEAR_SHOOTING_END_VALUE
        );
        this.ui = ui;
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiTableUIHandler methods                                --//
    //------------------------------------------------------------------------//

    @Override
    protected JXTable getTable() {
        return ui.getHydrologyTable();
    }

    @Override
    protected HydrologyTableModel getTableModel() {
        return (HydrologyTableModel) getTable().getModel();
    }

    @Override
    protected boolean isRowValid(HydrologyRowModel row) {
        //TODO
        return row.getGearShootingStartValue() != null ||
               row.getGearShootingEndValue() != null ||
               row.getAverageValue() != null;
    }


    @Override
    protected void onRowModified(int rowIndex,
                                 HydrologyRowModel row,
                                 String propertyName,
                                 Object oldValue,
                                 Object newValue) {
        recomputeRowValidState(row);
        super.onRowModified(rowIndex, row, propertyName, oldValue, newValue);
    }

    @Override
    protected void saveSelectedRowIfRequired(TuttiBeanMonitor<HydrologyRowModel> rowMonitor,
                                             HydrologyRowModel row) {

        if (row.isValid()) {
            // there is a valid bean attached to the monitor

            if (rowMonitor.wasModified()) {

                // monitored bean was modified, save it
                if (log.isInfoEnabled()) {
                    log.info("Row " + row + " was modified, will save it");
                }

                showInformationMessage(
                        "[ Trait - Hydro ] " +
                        "Sauvegarde des modifications de " + row + ".");

                saveRow(row);

                // clear modified flag on the monitor
                rowMonitor.clearModified();
                getModel().setModify(false);
            }
        }
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    protected HydrologyTabUIModel getModel() {
        return ui.getModel();
    }

    @Override
    public void beforeInitUI() {
        HydrologyTabUIModel model = new HydrologyTabUIModel();
        ui.setContextValue(model);

        model.addPropertyChangeListener(HydrologyTabUIModel.PROPERTY_AVAILABLE_CARACTERISTICS, new PropertyChangeListener() {

            public void propertyChange(PropertyChangeEvent evt) {
                List<Caracteristic> hydrologyPmfm = (List<Caracteristic>) evt.getNewValue();
                availableGlobalCaracteristics = Maps.newHashMap();
                if (hydrologyPmfm != null) {
                    for (Caracteristic caracteristic : hydrologyPmfm) {
                        String name = HydrologicCaracteristicUtil.getGlobalName(caracteristic.getName());
                        Caracteristic globalCaracteristic = HydrologicCaracteristicUtil.createGlobalCaracteristic(name, caracteristic);
                        Map<Type, Caracteristic> values = availableGlobalCaracteristics.get(globalCaracteristic);
                        if (values == null) {
                            values = Maps.newHashMap();
                            availableGlobalCaracteristics.put(globalCaracteristic, values);
                        }
                        Type type = HydrologicCaracteristicUtil.getTypeOfCaracteristic(caracteristic);
                        values.put(type, caracteristic);
                    }
                }
            }
        });
    }

    @Override
    public void afterInitUI() {
        initUI(ui);

        initBeanComboBox(ui.getNewRowKey(),
                         Lists.<Caracteristic>newArrayList(), null);

        JXTable table = getTable();

        // create table column model
        DefaultTableColumnModelExt columnModel =
                new DefaultTableColumnModelExt();

        {

            addColumnToModel(columnModel,
                             null,
                             newTableCellRender(Caracteristic.class, DecoratorService.CARACTERISTIC_WITH_UNIT),
                             HydrologyTableModel.KEY);
        }

        {

            addColumnToModel(columnModel,
                             new CaracteristicValueEditor(context),
                             new CaracteristicValueRenderer(context),
                             HydrologyTableModel.GEAR_SHOOTING_START_VALUE);
        }

        {

            addColumnToModel(columnModel,
                             new CaracteristicValueEditor(context),
                             new CaracteristicValueRenderer(context),
                             HydrologyTableModel.GEAR_SHOOTING_END_VALUE);
        }

        {

            addColumnToModel(columnModel,
                             new CaracteristicValueEditor(context),
                             new CaracteristicValueRenderer(context),
                             HydrologyTableModel.AVERAGE_VALUE);
        }

        // create table model
        HydrologyTableModel tableModel =
                new HydrologyTableModel(columnModel);

        table.setModel(tableModel);
        table.setColumnModel(columnModel);
        initTable(table);
    }

    @Override
    public void onCloseUI() {
        if (log.isInfoEnabled()) {
            log.info("closing: " + ui);
        }
    }

    @Override
    protected void onAfterSelectedRowChanged(int oldRowIndex,
                                             HydrologyRowModel oldRow,
                                             int newRowIndex,
                                             HydrologyRowModel newRow) {
        
        super.onAfterSelectedRowChanged(oldRowIndex, oldRow, newRowIndex, newRow);
        getModel().setRemoveCaracteristicEnabled(newRow != null);
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    /** Adds a row with the parameter selected in the combo box */
    public void addRow() {
        BeanComboBox<Caracteristic> keyCombo = ui.getNewRowKey();
        Caracteristic selectedItem = (Caracteristic) keyCombo.getSelectedItem();
        
        HydrologyRowModel row = getTableModel().createNewRow();
        row.setKey(selectedItem);
        getModel().getRows().add(row);
        
        HydrologyTableModel tableModel = getTableModel();
        int rowIndex = tableModel.getRowCount() - 1;
        tableModel.fireTableRowsInserted(rowIndex, rowIndex);

        keyCombo.removeItem(selectedItem);
        selectFirstInCombo(keyCombo);
        
        HydrologyTabUIModel model = getModel();
        model.setModify(true);
        model.setValid(row.isValid());
    }

    /** Resets the table with the data from the database */
    public void reset(FishingOperation fishingOperation) {
        HydrologyTableModel tableModel = getTableModel();
        HydrologyTabUIModel model = getModel();
        model.setFishingOperation(fishingOperation);
        
        CaracteristicMap hydrologyCaracteristics = fishingOperation.getHydrologyCaracteristics();
        if (hydrologyCaracteristics == null) {
            hydrologyCaracteristics = new CaracteristicMap();
        }
        
        TuttiProtocol protocol = ui.getContextValue(TuttiProtocol.class);
        List<String> gearPmfmIds = Lists.newArrayList();
        if (protocol != null) {
            List<String> protocolHydrologyPmfmId = protocol.getHydrologyPmfmId();
            if (protocolHydrologyPmfmId != null) {
                gearPmfmIds.addAll(protocolHydrologyPmfmId);
            }
        }
        
        List<HydrologyRowModel> rows = Lists.newArrayList();
        List<Caracteristic> caracteristicList = Lists.newArrayList();
        List<Caracteristic> selectedCaracteristics = Lists.newArrayList(hydrologyCaracteristics.keySet());
        
        List<Caracteristic> availableCaracteristics = model.getAvailableCaracteristics();
        for (String id : gearPmfmIds) {
            Caracteristic caracteristic = TuttiEntities.findById(availableCaracteristics, id);
            if (!selectedCaracteristics.contains(caracteristic)) {
                selectedCaracteristics.add(caracteristic);
            }
        }
        
        Set<Caracteristic> globalCaracteristics = availableGlobalCaracteristics.keySet();

        for (Caracteristic availableCaracteristic : globalCaracteristics) {
            Map<Type, Caracteristic> caracteristics = availableGlobalCaracteristics.get(availableCaracteristic);

            //should add the global caracteristic to the combo or to the table?
            boolean selectedCaracteristic = false;
            for (Caracteristic caracteristic : caracteristics.values()) {
                selectedCaracteristic |= selectedCaracteristics.contains(caracteristic);
            }

            if (selectedCaracteristic) {
                HydrologyRowModel row = tableModel.createNewRow();
                row.setKey(availableCaracteristic);
                row.setGearShootingStartValue(hydrologyCaracteristics.get(caracteristics.get(Type.START)));
                row.setGearShootingEndValue(hydrologyCaracteristics.get(caracteristics.get(Type.END)));
                row.setAverageValue(hydrologyCaracteristics.get(caracteristics.get(Type.AVERAGE)));
                rows.add(row);

            } else {
                caracteristicList.add(availableCaracteristic);
            }
        }

        model.setRows(rows);

        ui.getNewRowKey().setData(caracteristicList);
        selectFirstInCombo(ui.getNewRowKey());
        model.setModify(false);
    }
    
    public void removeCaracteristic() {
        int rowIndex = getTable().getSelectedRow();

        Preconditions.checkState(rowIndex != -1,
                                 "Cant remove caracteristic if no caracteristic selected");
        
        HydrologyRowModel row = getTableModel().getEntry(rowIndex);
        
        FishingOperation fishingOperation = getModel().getFishingOperation();
        Preconditions.checkNotNull(fishingOperation);

        Map<Type, Caracteristic> caracteristics = availableGlobalCaracteristics.get(row.getKey());
        for (Type type : caracteristics.keySet()) {
            Caracteristic caracteristic = caracteristics.get(type);
            fishingOperation.getHydrologyCaracteristics().remove(caracteristic);
        }
        
        if (TuttiEntities.isNew(fishingOperation)) {

            fishingOperation = persistenceService.createFishingOperation(fishingOperation);
        } else {
            persistenceService.saveFishingOperation(fishingOperation);
        }
        
        //add the row in the combo
        BeanComboBox keyCombo = ui.getNewRowKey();
        keyCombo.addItem(row.getKey());
        selectFirstInCombo(keyCombo);
        
        // remove the row from the model
        getModel().getRows().remove(rowIndex);

        // refresh all the table
        getTableModel().fireTableRowsDeleted(rowIndex, rowIndex);
        
    }
    
    //------------------------------------------------------------------------//
    //-- Protected methods                                                  --//
    //------------------------------------------------------------------------//
    
    protected void saveRow(HydrologyRowModel row) {

        FishingOperation fishingOperation = getModel().getFishingOperation();
        Preconditions.checkNotNull(fishingOperation);

        if (fishingOperation.getHydrologyCaracteristics() == null) {
            fishingOperation.setHydrologyCaracteristics(new CaracteristicMap());
        }
        Map<Type, Caracteristic> caracteristics = availableGlobalCaracteristics.get(row.getKey());
        CaracteristicMap map = new CaracteristicMap();
        for (Type type : caracteristics.keySet()) {
            Serializable value = null;
            switch (type) {
                case START:
                    value = row.getGearShootingStartValue();
                    break;
                case END:
                    value = row.getGearShootingEndValue();
                    break;
                case AVERAGE:
                    value = row.getAverageValue();
            }
            map.put(caracteristics.get(type), value);
        }
        fishingOperation.getHydrologyCaracteristics().putAll(map);
        
        if (TuttiEntities.isNew(fishingOperation)) {

            fishingOperation = persistenceService.createFishingOperation(fishingOperation);
        } else {
            persistenceService.saveFishingOperation(fishingOperation);
        }
    }

}
