package fr.ifremer.tutti.ui.swing.content.operation;

/*
 * #%L
 * Tutti :: UI
 * $Id: FishingOperationsUIHandler.java 201 2013-01-15 11:07:25Z kmorin $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-0.3/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/FishingOperationsUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.data.Cruise;
import fr.ifremer.tutti.persistence.entities.data.FishingOperation;
import fr.ifremer.tutti.persistence.entities.protocol.TuttiProtocol;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.ui.swing.AbstractTuttiUIHandler;
import fr.ifremer.tutti.ui.swing.MainUI;
import fr.ifremer.tutti.ui.swing.TuttiUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUIModel;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiTabContainerUIHandler;
import fr.ifremer.tutti.ui.swing.util.TabHandler;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.decorator.Decorator;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Date;
import java.util.List;
import javax.swing.DefaultSingleSelectionModel;
import javax.swing.JTabbedPane;

import static org.nuiton.i18n.I18n._;

/**
 * Handler of UI {@link FishingOperationsUI}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class FishingOperationsUIHandler extends AbstractTuttiTabContainerUIHandler<FishingOperationsUIModel> {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(FishingOperationsUIHandler.class);

    private final FishingOperationsUI ui;

    private final PersistenceService persistenceService;

    public FishingOperationsUIHandler(TuttiUI parentUI,
                                      FishingOperationsUI ui) {
        super(parentUI.getHandler().getContext());
        this.ui = ui;
        persistenceService = context.getService(PersistenceService.class);
    }

    @Override
    public void beforeInitUI() {

        if (log.isInfoEnabled()) {
            log.info("for " + ui);
        }
        FishingOperationsUIModel model = new FishingOperationsUIModel();

        EditCatchesUIModel catchesUIModel = new EditCatchesUIModel();

        String cruiseId = context.getCruiseId();

        // load existing cruise
        Cruise cruise = persistenceService.getCruise(cruiseId);
        model.setCruise(cruise);

        if (getContext().isProtocolFilled()) {

            // load existing protocol

            TuttiProtocol protocol =
                    persistenceService.getProtocol(getContext().getProtocolId());

            if (log.isInfoEnabled()) {
                log.info("Loading existing protocol: " + protocol);
            }
            ui.setContextValue(protocol);
        } else {

            // remove any previous existing protocol
            ui.removeContextValue(TuttiProtocol.class);
        }

        List<FishingOperation> fishingOperations =
                persistenceService.getAllFishingOperation(cruiseId);
        model.setFishingOperation(fishingOperations);

        if (log.isInfoEnabled()) {
            log.info("Loaded " + fishingOperations.size() +
                     " fishingOperation(s).");
        }

        ui.setContextValue(model);
        ui.setContextValue(catchesUIModel);
    }

    @Override
    public void afterInitUI() {

        FishingOperationsUIModel model = getModel();

        initUI(ui);

        List<FishingOperation> fishingOperations = model.getFishingOperation();

        initBeanComboBox(ui.getFishingOperationComboBox(),
                         fishingOperations,
                         model.getSelectedFishingOperation());

        model.addPropertyChangeListener(FishingOperationsUIModel.PROPERTY_SELECTED_FISHING_OPERATION, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                selectFishingOperation((FishingOperation) evt.getNewValue());
            }
        });

        model.addPropertyChangeListener(FishingOperationsUIModel.PROPERTY_FISHING_OPERATION, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                ui.getFishingOperationComboBox().setData(null);
                ui.getFishingOperationComboBox().setData((List<FishingOperation>) evt.getNewValue());
            }
        });

        if (!fishingOperations.isEmpty()) {

            model.setSelectedFishingOperation(fishingOperations.get(0));
        }
    }

    @Override
    public void onCloseUI() {

        // ui will be saved so we do not want to keep selected tab indexes
        ui.getTabPane().setSelectedIndex(0);
        ui.getFishingOperationTabContent().getFishingOperationTabPane().setSelectedIndex(0);

        // when quitting ui, let's de-select fishingOperation (will save any changes)
        selectFishingOperation(null);
        
        closeUI(ui.getFishingOperationTabContent());
        closeUI(ui.getCatchesTabContent());    
    }

    @Override
    protected FishingOperationsUIModel getModel() {
        return ui.getModel();
    }
    
    @Override
    protected JTabbedPane getTabPanel() {
        return ui.getTabPane();
    }

    public void createNewFishingOperation() {

        FishingOperationsUIModel model = getModel();
        
        // deselect selected fishingOperation
        model.setSelectedFishingOperation(null);

        // use a new empty fishingOperation
        FishingOperation newFishingOperation = new FishingOperation();
        newFishingOperation.setCruise(model.getCruise());

        // by default use the current day with no time information
        Date currentDate = new Date();
        currentDate = DateUtils.setHours(currentDate, 0);
        currentDate = DateUtils.setMinutes(currentDate, 0);

        newFishingOperation.setDate(currentDate);
        newFishingOperation.setGearShootingStartDate(currentDate);
        newFishingOperation.setGearShootingEndDate(currentDate);
        
        selectFishingOperation(newFishingOperation);
    }

    public void selectFishingOperation(FishingOperation fishingOperation) {

        if (log.isInfoEnabled()) {
            log.info("New selected fishingOperation: " + fishingOperation);
        }
        // back to fishingOperation tab
        ui.getTabPane().setSelectedIndex(0);

        // back to general tab of fishingOperation tabs
        ui.getFishingOperationTabContent().getFishingOperationTabPane().setSelectedIndex(0);

        if (fishingOperation != null) {
            Decorator<FishingOperation> decorator =
                    getDecorator(FishingOperation.class, null);

            String fishingOperationText =
                    _("tutti.label.traitReminder",
                      decorator.toString(fishingOperation));

            // propagate fishingOperation to his tabs
            ui.getFishingOperationTabContent().getHandler().selectFishingOperation(fishingOperation);

            ui.getCatchesTabContent().getHandler().selectFishingOperation(fishingOperation,
                                                                          fishingOperationText);

            // repaint tabs
            ui.getTabPane().repaint();
            
        } else {
            ui.getFishingOperationTabContent().getHandler().clearFishingOperation();
        }
    }
    
    public void closeCurrentFishingOperation() {
        ui.getFishingOperationTabContent().getHandler().selectFishingOperation(null);
        ui.getCatchesTabContent().getHandler().selectFishingOperation(null, "");
    }

    public void saveFishingOperation(FishingOperation toSave) {

        // persist the fishingOperation

        boolean create = toSave.getId() == null;
        FishingOperationsUIModel model = getModel();
        FishingOperation savedFishingOperation;

        if (create) {
            savedFishingOperation = persistenceService.createFishingOperation(toSave);
            model.addFishingOperation(savedFishingOperation);
            model.setSelectedFishingOperation(savedFishingOperation);
            
        } else {
            savedFishingOperation = persistenceService.saveFishingOperation(toSave);
            
            // add the saved fishingOperation to fishingOperation list
            List<FishingOperation> data = model.getFishingOperation();

            FishingOperation existingFishingOperation =
                    TuttiEntities.findById(data, savedFishingOperation.getId());
            model.updateFishingOperation(existingFishingOperation, savedFishingOperation);
        }
    }
}
