package fr.ifremer.tutti.ui.swing.content.cruise;

/*
 * #%L
 * Tutti :: UI
 * $Id: EditCruiseUIHandler.java 184 2013-01-13 14:58:21Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-0.3/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/cruise/EditCruiseUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Lists;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.data.Cruise;
import fr.ifremer.tutti.persistence.entities.data.Program;
import fr.ifremer.tutti.persistence.entities.referential.Gear;
import fr.ifremer.tutti.persistence.entities.referential.Person;
import fr.ifremer.tutti.persistence.entities.referential.Vessel;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.ui.swing.AbstractTuttiUIHandler;
import fr.ifremer.tutti.ui.swing.MainUI;
import fr.ifremer.tutti.ui.swing.TuttiScreen;
import fr.ifremer.tutti.ui.swing.TuttiUI;
import fr.ifremer.tutti.ui.swing.TuttiUIContext;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.List;
import jaxx.runtime.JAXXUtil;
import jaxx.runtime.context.JAXXContextEntryDef;
import jaxx.runtime.validator.swing.SwingValidator;
import jaxx.runtime.validator.swing.SwingValidatorMessageTableRenderer;
import jaxx.runtime.validator.swing.SwingValidatorUtil;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Handler of UI {@link EditCruiseUI}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class EditCruiseUIHandler extends AbstractTuttiUIHandler<EditCruiseUIModel> {

    /** Logger. */
    private static final Log log = LogFactory.getLog(EditCruiseUIHandler.class);

    /**
     * UI.
     *
     * @since 0.1
     */
    private final EditCruiseUI ui;

    /**
     * Persistence service.
     *
     * @since 0.1
     */
    private final PersistenceService persistenceService;

    public EditCruiseUIHandler(TuttiUI parentUi, EditCruiseUI ui) {
        super(parentUi.getHandler().getContext());
        this.ui = ui;
        persistenceService = context.getService(PersistenceService.class);
    }

    @Override
    public void beforeInitUI() {

        EditCruiseUIModel model = new EditCruiseUIModel();

        // get vessels

        List<Vessel> fishingVessels = persistenceService.getAllFishingVessel();
        FISHING_VESSELS_CONTEXT_ENTRY.setContextValue(ui, fishingVessels);

        List<Vessel> scientificVessels = persistenceService.getAllScientificVessel();
        SCIENTIFIC_VESSELS_CONTEXT_ENTRY.setContextValue(ui, scientificVessels);

        // get gears

        List<Gear> scientificGears = persistenceService.getAllScientificGear();
        SCIENTIFIC_GEARS_CONTEXT_ENTRY.setContextValue(ui, scientificGears);

        List<Gear> fishingGears = persistenceService.getAllFishingGear();
        FISHING_GEARS_CONTEXT_ENTRY.setContextValue(ui, fishingGears);

        String cruiseId = context.getCruiseId();
        if (cruiseId == null) {

            // create new cruise
            Program program =
                    persistenceService.getProgram(context.getProgramId());
            model.setProgram(program);
            model.setPoche(1);

            if (log.isDebugEnabled()) {
                log.debug("Will create a new cruise from program: " + program);
            }
        } else {

            // load existing program
            Cruise cruise =
                    persistenceService.getCruise(cruiseId);

            model.fromBean(cruise);
            if (log.isDebugEnabled()) {
                log.debug("Will edit cruise: " + cruiseId);
            }
        }
        listModelIsModify(model);
        ui.setContextValue(model);
    }

    JAXXContextEntryDef<List<Vessel>> SCIENTIFIC_VESSELS_CONTEXT_ENTRY = JAXXUtil.newListContextEntryDef("scientificVessel");

    JAXXContextEntryDef<List<Vessel>> FISHING_VESSELS_CONTEXT_ENTRY = JAXXUtil.newListContextEntryDef("fishingVessel");

    JAXXContextEntryDef<List<Gear>> SCIENTIFIC_GEARS_CONTEXT_ENTRY = JAXXUtil.newListContextEntryDef("scientificGear");

    JAXXContextEntryDef<List<Gear>> FISHING_GEARS_CONTEXT_ENTRY = JAXXUtil.newListContextEntryDef("fishingGear");

    protected List<Vessel> getVesselList(VesselTypeEnum vesselType) {
        List<Vessel> result = Lists.newArrayList();
        switch (vesselType) {

            case ALL:
                result.addAll(FISHING_VESSELS_CONTEXT_ENTRY.getContextValue(ui));
                result.addAll(SCIENTIFIC_VESSELS_CONTEXT_ENTRY.getContextValue(ui));
                break;
            case SCIENTIFIC:
                result.addAll(SCIENTIFIC_VESSELS_CONTEXT_ENTRY.getContextValue(ui));
                break;
            case FISHING:
                result.addAll(FISHING_VESSELS_CONTEXT_ENTRY.getContextValue(ui));
                break;
        }
        return result;
    }

    protected List<Gear> getGearList(VesselTypeEnum vesselType) {
        List<Gear> result = Lists.newArrayList();
        switch (vesselType) {

            case ALL:
                result.addAll(SCIENTIFIC_GEARS_CONTEXT_ENTRY.getContextValue(ui));
                result.addAll(FISHING_GEARS_CONTEXT_ENTRY.getContextValue(ui));
                break;
            case SCIENTIFIC:
                result.addAll(SCIENTIFIC_GEARS_CONTEXT_ENTRY.getContextValue(ui));
                break;
            case FISHING:
                result.addAll(FISHING_GEARS_CONTEXT_ENTRY.getContextValue(ui));
                break;
        }
        return result;
    }

    @Override
    public void afterInitUI() {

        initUI(ui);

        List<Person> users = persistenceService.getAllPerson();

        EditCruiseUIModel model = getModel();

        model.setVesselType(VesselTypeEnum.ALL);
        
        // Change vessel list and gear list when vessel type changes
        model.addPropertyChangeListener(EditCruiseUIModel.PROPERTY_VESSEL_TYPE, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {

                EditCruiseUIModel source = (EditCruiseUIModel) evt.getSource();

                List<Vessel> selectedVessel = source.getVessel();
                List<Gear> selectedGear = source.getGear();

                VesselTypeEnum vesselType = (VesselTypeEnum) evt.getNewValue();

                List<Vessel> newVesselList = getVesselList(vesselType);
                List<Gear> newGearList = getGearList(vesselType);

                // compute new selected vessel list
                List<Vessel> newSelectedVessel = Lists.newArrayList();
                if (selectedVessel!=null) {
                    newSelectedVessel.addAll(selectedVessel);
                }
                newSelectedVessel.retainAll(newVesselList);

                // compute new selected gear list
                List<Gear> newSelectedGear = Lists.newArrayList();
                if (selectedGear!=null) {
                    newSelectedGear.addAll(selectedGear);
                }
                newSelectedGear.retainAll(newGearList);

                // reinit vessel list
                ui.getVesselList().getModel().setUniverse(newVesselList);
                ui.getVesselList().getModel().setSelected(newSelectedVessel);

                // reinit gear list
                ui.getGearList().getModel().setUniverse(newGearList);
                ui.getGearList().getModel().setSelected(newSelectedGear);

            }
        });

        initBeanComboBox(ui.getSurveyComboBox(),
                         persistenceService.getAllProgram(),
                         model.getProgram());

        initBeanComboBox(ui.getCountryComboBox(),
                         persistenceService.getAllCountry(),
                         model.getCountry());

        initBeanList(ui.getVesselList(),
                     getVesselList(VesselTypeEnum.ALL),
                     model.getVessel());

        initBeanList(ui.getGearList(),
                     getGearList(VesselTypeEnum.ALL),
                     model.getGear());

        initBeanList(ui.getHeadOfMissionList(),
                     users,
                     model.getHeadOfMission());

        initBeanList(ui.getHeadOfSortRoomList(),
                     users,
                     model.getHeadOfSortRoom());

        SwingValidator validator = ui.getValidator();
        listenValidatorValid(validator, model);

        // if new fishingOperation can already cancel his creation
        model.setModify(model.isCreate());
        
        MainUI main = ui.getContextValue(MainUI.class, MainUI.class.getName());
        main.clearValidators();
        main.registerValidator(validator);

    }

    @Override
    public void onCloseUI() {
        if (log.isInfoEnabled()) {
            log.info("closing: " + ui);
        }
        MainUI main = ui.getContextValue(MainUI.class, MainUI.class.getName());
        main.clearValidators();
    }

    @Override
    protected EditCruiseUIModel getModel() {
        return ui.getModel();
    }

    public void cancel() {

        context.setScreen(TuttiScreen.SELECT_CRUISE);
    }

    public void save() {

        EditCruiseUIModel model = getModel();

        Cruise bean = model.toBean();

        Cruise saved;
        if (TuttiEntities.isNew(bean)) {

            saved = persistenceService.createCruise(bean);
        } else {
            saved = persistenceService.saveCruise(bean);
        }

        context.setProgramId(saved.getProgram().getId());
        context.setCruiseId(saved.getId());

        context.setScreen(TuttiScreen.SELECT_CRUISE);
    }

    public static final String cruiseNameFormat = "%1$s_%2$s";

    public void generateCampaignName() {

        EditCruiseUIModel model = getModel();
        Program program = model.getProgram();
        Integer year = model.getYear();
        String name = String.format(cruiseNameFormat, program.getName(), year);

        String surveyPart = model.getSurveyPart();
        if (StringUtils.isNotEmpty(surveyPart)) {
            name += "_" + surveyPart;
        }
        model.setName(name);
    }

}
