package fr.ifremer.tutti.ui.swing.content.operation.species;

/*
 * #%L
 * Tutti :: UI
 * $Id: SpeciesBatchUIHandler.java 82 2012-12-17 09:18:52Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-0.2/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/species/SpeciesBatchUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.data.Attachment;
import fr.ifremer.tutti.persistence.entities.data.FishingOperation;
import fr.ifremer.tutti.persistence.entities.data.SpeciesBatch;
import fr.ifremer.tutti.persistence.entities.data.SpeciesBatchFrequency;
import fr.ifremer.tutti.persistence.entities.referential.Sex;
import fr.ifremer.tutti.persistence.entities.referential.SortedUnsortedCategory;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.persistence.entities.referential.WeightCategory;
import fr.ifremer.tutti.service.DecoratorService;
import fr.ifremer.tutti.ui.swing.TuttiUI;
import fr.ifremer.tutti.ui.swing.content.operation.AbstractTuttiBatchTableUIHandler;
import fr.ifremer.tutti.ui.swing.content.operation.FishingOperationsUI;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import fr.ifremer.tutti.ui.swing.util.editor.AttachmentCellComponent;
import fr.ifremer.tutti.ui.swing.util.editor.LongTextCellComponent;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.table.DefaultTableColumnModelExt;
import org.nuiton.util.decorator.Decorator;

import javax.swing.table.TableColumnModel;
import java.util.List;
import java.util.Set;

import static org.nuiton.i18n.I18n._;
import static org.nuiton.i18n.I18n.n_;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class SpeciesBatchUIHandler extends AbstractTuttiBatchTableUIHandler<SpeciesBatchRowModel, SpeciesBatchUIModel> {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(SpeciesBatchUIHandler.class);

    public static final Set<String> RECOMPUTE_TOTAL_WEIGHT = Sets.newHashSet(
            SpeciesBatchRowModel.PROPERTY_SORTED_UNSORTED_CATEGORY,
            SpeciesBatchRowModel.PROPERTY_WEIGHT);

    public static final Set<String> SAMPLING_PROPERTIES = Sets.newHashSet(
            SpeciesBatchRowModel.PROPERTY_SPECIES,
            SpeciesBatchRowModel.PROPERTY_SORTED_UNSORTED_CATEGORY,
            SpeciesBatchRowModel.PROPERTY_WEIGHT_CATEGORY,
            SpeciesBatchRowModel.PROPERTY_SEX,
            SpeciesBatchRowModel.PROPERTY_MATURITY,
            SpeciesBatchRowModel.PROPERTY_AGE);

    /**
     * UI.
     *
     * @since 0.2
     */
    private final SpeciesBatchUI ui;

    public SpeciesBatchUIHandler(FishingOperationsUI parentUi,
                                 SpeciesBatchUI ui) {
        super(parentUi,
              new String[]{
                      SpeciesBatchUIModel.PROPERTY_SAMPLE_VRAC_WEIGHT,
                      SpeciesBatchUIModel.PROPERTY_TOTAL_HORS_VRAC_WEIGHT,
                      SpeciesBatchUIModel.PROPERTY_TOTAL_VRAC_WEIGHT,
                      SpeciesBatchUIModel.PROPERTY_TOTAL_WEIGHT
              },
              SpeciesBatchRowModel.PROPERTY_SPECIES_TO_CONFIRM,
              SpeciesBatchRowModel.PROPERTY_SPECIES,
              SpeciesBatchRowModel.PROPERTY_SORTED_UNSORTED_CATEGORY,
              SpeciesBatchRowModel.PROPERTY_WEIGHT_CATEGORY,
              SpeciesBatchRowModel.PROPERTY_SEX,
              SpeciesBatchRowModel.PROPERTY_MATURITY,
              SpeciesBatchRowModel.PROPERTY_AGE,
              SpeciesBatchRowModel.PROPERTY_WEIGHT,
              SpeciesBatchRowModel.PROPERTY_COMMENT,
              SpeciesBatchRowModel.PROPERTY_ATTACHMENTS,
              SpeciesBatchRowModel.PROPERTY_FREQUENCY);
        this.ui = ui;
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiTableUIHandler methods                                --//
    //------------------------------------------------------------------------//

    @Override
    protected SpeciesBatchUIModel getModel() {
        return ui.getModel();
    }

    @Override
    protected SpeciesBatchTableModel getTableModel() {
        return (SpeciesBatchTableModel) getTable().getModel();
    }

    @Override
    protected JXTable getTable() {
        return ui.getTable();
    }

    @Override
    protected FishingOperation getFishingOperation() {
        return getModel().getFishingOperation();
    }

    @Override
    protected void onModelRowsChanged(List<SpeciesBatchRowModel> rows) {
        super.onModelRowsChanged(rows);

        // build the new sampling tree from the new rows to edit
        getModel().getSamplingTreeModel().populate(rows);
    }

    @Override
    protected String[] getRowPropertiesToIgnore() {
        return new String[]{
                SpeciesBatchRowModel.PROPERTY_SAMPLE_WEIGHT,
                SpeciesBatchRowModel.PROPERTY_SAMPLING_RATIO,
                SpeciesBatchRowModel.PROPERTY_FREQUENCY};
    }

    @Override
    protected TableColumnModel createTableColumnModel() {

        DefaultTableColumnModelExt columnModel =
                new DefaultTableColumnModelExt();

        {
            // Species to confirm column

            addBooleanColumnToModel(columnModel,
                                    SpeciesBatchTableModel.SPECIES_TO_CONFIRM,
                                    getTable());
        }

        List<Species> allSpecies = persistenceService.getAllSpecies();

        {
            // Species (by code) column

            Decorator<Species> decorator = getDecorator(
                    Species.class, DecoratorService.SPECIES_BY_CODE);

            addComboDataColumnToModel(columnModel,
                                      SpeciesBatchTableModel.SPECIES_BY_CODE,
                                      decorator, allSpecies);
        }

        {
            // Species (by genusCode) column

            Decorator<Species> decorator = getDecorator(
                    Species.class, DecoratorService.SPECIES_BY_GENUS);

            addComboDataColumnToModel(columnModel,
                                      SpeciesBatchTableModel.SPECIES_BY_GENUS_CODE,
                                      decorator, allSpecies);

        }

        {

            // SortedUnsortedCategory column

            Decorator<SortedUnsortedCategory> decorator =
                    getDecorator(SortedUnsortedCategory.class, null);

            List<SortedUnsortedCategory> data =
                    persistenceService.getAllSortedUnsortedCategory();

            addComboDataColumnToModel(columnModel,
                                      SpeciesBatchTableModel.SORTED_UNSORTED_CATEGORY,
                                      decorator, data);
        }

        {

            // WeightCategory column

            Decorator<WeightCategory> decorator =
                    getDecorator(WeightCategory.class,
                                 DecoratorService.BY_NAME);

            List<WeightCategory> data =
                    persistenceService.getAllWeightCategory();

            addComboDataColumnToModel(columnModel,
                                      SpeciesBatchTableModel.WEIGHT_CATEGORY,
                                      decorator, data);
        }

        {

            // Sex column

            Decorator<Sex> decorator =
                    getDecorator(Sex.class, DecoratorService.BY_NAME);

            List<Sex> data = persistenceService.getAllSex();

            addComboDataColumnToModel(columnModel,
                                      SpeciesBatchTableModel.SEX,
                                      decorator, data);
        }

        {

            // Maturity column

            addFloatColumnToModel(columnModel,
                                  SpeciesBatchTableModel.MATURITY,
                                  TuttiUI.DECIMAL3_PATTERN);
        }

        {

            // Age column

            addFloatColumnToModel(columnModel,
                                  SpeciesBatchTableModel.AGE,
                                  TuttiUI.DECIMAL3_PATTERN);
        }

        { // Weight column

            addFloatColumnToModel(columnModel,
                                  SpeciesBatchTableModel.WEIGHT,
                                  TuttiUI.DECIMAL3_PATTERN);
        }

        { // Computed weight column (from frequencies)

            addColumnToModel(columnModel,
                             FrequencyCellComponent.newEditor(ui.getFrequencyEditor()),
                             FrequencyCellComponent.newRender(),
                             SpeciesBatchTableModel.COMPUTED_WEIGHT);
        }

        { // Computed number column (from frequencies)

            addColumnToModel(columnModel,
                             FrequencyCellComponent.newEditor(ui.getFrequencyEditor()),
                             FrequencyCellComponent.newRender(),
                             SpeciesBatchTableModel.COMPUTED_NUMBER);
        }

        { // Sample weight column

            addColumnToModel(columnModel,
                             SpeciesBatchTableModel.SAMPLE_WEIGHT);
        }

        { // SamplingRatio column

            addColumnToModel(columnModel,
                             SpeciesBatchTableModel.SAMPLING_RATIO);
        }

        { // Comment column

            addColumnToModel(columnModel,
                             LongTextCellComponent.newEditor(ui.getLongTextEditor()),
                             LongTextCellComponent.newRender(n_("tutti.tooltip.comment.none")),
                             SpeciesBatchTableModel.COMMENT);
        }

        { // File column

            addColumnToModel(columnModel,
                             AttachmentCellComponent.newEditor(ui.getAttachmentEditor()),
                             AttachmentCellComponent.newRender(
                                     getDecorator(Attachment.class, null),
                                     n_("tutti.tooltip.attachment.none")),
                             SpeciesBatchTableModel.ATTACHMENTS);
        }
        return columnModel;
    }

    @Override
    protected void onRowModified(SpeciesBatchRowModel row,
                                 String propertyName,
                                 Object oldValue,
                                 Object newValue) {

        if (RECOMPUTE_TOTAL_WEIGHT.contains(propertyName)) {

            // Need to recompute totalHorsVracWeight
            recomputeTotalHorsVrac();
        }

        SpeciesBatchUIModel model = getModel();

        SpeciesBatchTreeModel samplingTreeModel = model.getSamplingTreeModel();

        if (SAMPLING_PROPERTIES.contains(propertyName)) {

            // Need to rebuilt this row sampling tree path (and then recompute
            // old super - samplingRatio and new super - samplingRatio)

            // old node of the previous sampling def for this row
            // and remove it from any cache
            SpeciesBatchTreeNode oldNode = samplingTreeModel.removeNodeFromCache(row);

            boolean rowWasValid = row.isValid();
            if (oldNode != null && rowWasValid) {

                // remove this row from his super sampling
                recomputeSuperSamplingRatio(samplingTreeModel, oldNode);
            }

            // get new sampling node
            SpeciesBatchTreeNode newNode = samplingTreeModel.getSamplingNode(row);

            // check this row is valid
            boolean rowValid = samplingTreeModel.isValid(row, newNode);

            // push this state back to the row
            row.setValid(rowValid);

            if (rowValid) {

                // can add it to his super-sampling
                recomputeSuperSamplingRatio(samplingTreeModel, newNode);
            } else {

                if (rowWasValid) {

                    row.setSampleWeight(null);
                    row.setSamplingRatio(null);

                    getTableModel().updateSamplingRatio(Sets.newHashSet(row));
                }
            }
        }

        if (SpeciesBatchRowModel.PROPERTY_WEIGHT.equals(propertyName)) {

            // Need to recompute the super - samplingRatio

            SpeciesBatchTreeNode node = samplingTreeModel.getSamplingNode(row);

            boolean rowWasValid = row.isValid();

            // check this row is valid
            boolean rowValid = samplingTreeModel.isValid(row, node);

            // push this state back to the row
            row.setValid(rowValid);

            if (rowValid) {
                recomputeSuperSamplingRatio(samplingTreeModel, node);
            } else {
                if (rowWasValid) {

                    // must remove this row from his super-sampling
                    recomputeSuperSamplingRatio(samplingTreeModel, node);

                    row.setSampleWeight(null);
                    row.setSamplingRatio(null);

                    getTableModel().updateSamplingRatio(Sets.newHashSet(row));
                }
            }
        }
    }

    @Override
    protected void onRowValidStateChanged(SpeciesBatchRowModel row,
                                          Boolean oldValue,
                                          Boolean newValue) {
        int rowIndex = getTableModel().getRowIndex(row);

        if (rowIndex > -1) {
            getTableModel().fireTableRowsUpdated(rowIndex, rowIndex);
        }
    }

    @Override
    protected void onRowModifyStateChanged(SpeciesBatchRowModel row,
                                           Boolean oldValue,
                                           Boolean newValue) {
    }

    @Override
    protected void saveSelectedRowIfRequired() {

        TuttiBeanMonitor<SpeciesBatchRowModel> rowMonitor = getRowMonitor();

        SpeciesBatchRowModel bean = rowMonitor.getBean();
        if (bean != null) {

            if (bean.isValid()) {
                // there is a valid bean attached to the monitor

                if (rowMonitor.wasModified()) {

                    // monitored bean was modified, save it
                    if (log.isInfoEnabled()) {
                        log.info("Row " + bean + " was modified, will save it");
                    }

                    saveRow(bean);

                    // clear modified flag on the monitor
                    rowMonitor.clearModified();
                }
            } else {

                // row is not valid can not save it

                SpeciesBatch catchBean = bean.toBean();

                if (!TuttiEntities.isNew(catchBean)) {

                    // remove this
                    persistenceService.deleteSpeciesBatch(catchBean.getId());
                }
            }
        }
    }

    @Override
    protected boolean isRowValid(SpeciesBatchRowModel row) {

        SpeciesBatchUIModel model = getModel();

        SpeciesBatchTreeModel samplingTreeModel = model.getSamplingTreeModel();
        SpeciesBatchTreeNode node = samplingTreeModel.getSamplingNode(row);
        boolean rowValid = samplingTreeModel.isValid(row, node);
        return rowValid;
    }

    @Override
    protected void saveRow(SpeciesBatchRowModel row) {

        SpeciesBatch catchBean = row.toBean();

        FishingOperation fishingOperation = getModel().getFishingOperation();
        catchBean.setFishingOperation(fishingOperation);
        if (log.isInfoEnabled()) {
            log.info("Selected fishingOperation: " + fishingOperation.getId());
        }

        if (TuttiEntities.isNew(catchBean)) {

            catchBean = persistenceService.createSpeciesBatch(catchBean);
            row.setId(catchBean.getId());
        } else {
            persistenceService.saveSpeciesBatch(catchBean);
        }

        List<SpeciesFrequencyRowModel> frequencyRows = row.getFrequency();

        List<SpeciesBatchFrequency> frequency =
                SpeciesFrequencyRowModel.toBeans(frequencyRows, catchBean);

        if (log.isInfoEnabled()) {
            log.info("Will save " + frequency.size() + " frequencies.");
        }
        frequency = persistenceService.saveSpeciesBatchFrequency(
                catchBean.getId(), frequency);

        // push it back to row model
        frequencyRows = SpeciesFrequencyRowModel.fromBeans(frequency);
        row.setFrequency(frequencyRows);
    }

    @Override
    public void selectFishingOperation(FishingOperation bean) {

        JXTable table = ui.getTable();

        if (table.isEditing()) {

            // make sure to stop editor
            table.editingCanceled(null);
        }

        if (fishingOperationMonitor.wasModified()) {

            // previous fishingOperation was modified, let's save it
            SpeciesBatchUIModel beanToSave = fishingOperationMonitor.getBean();

            FishingOperation fishingOperation = beanToSave.toBean();

            if (log.isInfoEnabled()) {
                log.info("FishingOperation " + fishingOperation.getId() +
                         " was modified, will save it.");
            }

            persistenceService.saveFishingOperation(fishingOperation);
        }

        // make sure selection is empty (will remove bean from monitor)
        table.clearSelection();

        boolean empty = bean == null;

        SpeciesBatchUIModel model = getModel();

        List<SpeciesBatchRowModel> rows;

        String fishingoperationText;

        if (empty) {
            rows = null;
            bean = new FishingOperation();
            model.setFishingOperation(null);
            fishingoperationText = "";
        } else {

            Decorator<FishingOperation> decorator =
                    getDecorator(FishingOperation.class, null);

            fishingoperationText =
                    _("tutti.label.traitReminder", decorator.toString(bean));

            if (log.isInfoEnabled()) {
                log.info("Get species batch for fishingOperation: " +
                         bean.getId() + " - " + fishingoperationText);
            }
            rows = Lists.newArrayList();

            if (!TuttiEntities.isNew(bean)) {
                List<SpeciesBatch> catches =
                        persistenceService.getAllSpeciesBatch(bean.getId());
                for (SpeciesBatch aBatch : catches) {
                    List<SpeciesBatchFrequency> frequencies =
                            persistenceService.getAllSpeciesBatchFrequency(aBatch.getId());
                    SpeciesBatchRowModel entry =
                            new SpeciesBatchRowModel(aBatch, frequencies);
                    rows.add(entry);
                }
            }
            model.setFishingOperation(bean);
        }

        model.fromBean(bean);
        model.setRows(rows);

        fishingOperationMonitor.clearModified();

        ui.getFishingOperationReminderLabel().setText(fishingoperationText);
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    public void beforeInitUI() {

        if (log.isInfoEnabled()) {
            log.info("beforeInit: " + ui);
        }

        SpeciesBatchUIModel model = new SpeciesBatchUIModel();
        ui.setContextValue(model);

        fishingOperationMonitor.setBean(model);
    }

    @Override
    public void afterInitUI() {

        if (log.isInfoEnabled()) {
            log.info("afterInit: " + ui);
        }

        initUI(ui);

        //TODO Will come from protocol
        List<String> samplingOrder = Lists.newArrayList(
                SpeciesBatchRowModel.PROPERTY_SPECIES,
                SpeciesBatchRowModel.PROPERTY_SORTED_UNSORTED_CATEGORY,
                SpeciesBatchRowModel.PROPERTY_WEIGHT_CATEGORY,
                SpeciesBatchRowModel.PROPERTY_SEX,
                SpeciesBatchRowModel.PROPERTY_MATURITY,
                SpeciesBatchRowModel.PROPERTY_AGE
        );

        getModel().setSamplingOrder(samplingOrder);

        JXTable table = getTable();

        // create table column model
        TableColumnModel columnModel = createTableColumnModel();

        // create table model
        SpeciesBatchTableModel tableModel =
                new SpeciesBatchTableModel(columnModel);

        table.setModel(tableModel);
        table.setColumnModel(columnModel);

        initBatchTable(table, columnModel, tableModel);
    }

    @Override
    public void onCloseUI() {
        if (log.isInfoEnabled()) {
            log.info("closing: " + ui);
        }
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected void recomputeSuperSamplingRatio(SpeciesBatchTreeModel samplingTreeModel,
                                               SpeciesBatchTreeNode node) {

        SpeciesBatchTreeNode superSamplingNode = node.getParent();

        Preconditions.checkNotNull(superSamplingNode,
                                   "Super sampling node can't be null");

        SpeciesBatchTreeModel.SamplingContext samplingContext =
                samplingTreeModel.createSamplingContext(superSamplingNode);

        float samplingTotalWeight = samplingContext.getTotalWeight();
        Float superSamplingTotalWeight;
        SpeciesBatchRowModel superSamplingRow =
                samplingContext.getSuperSamplingRow();
        if (superSamplingRow == null) {

            // Use directly the batch total weight (means no super-sampling)

            superSamplingTotalWeight = getModel().getTotalWeight();
        } else {
            superSamplingTotalWeight = superSamplingRow.getWeight();
        }

        if (log.isInfoEnabled()) {
            log.info("Super sampling total weight: " +
                     superSamplingTotalWeight);
        }

        Float samplingRatio = null;
        if (superSamplingTotalWeight != null) {
            samplingRatio = samplingTotalWeight / superSamplingTotalWeight;
        }

        if (log.isInfoEnabled()) {
            log.info("Sampling ratio: " + samplingRatio);
        }

        samplingContext.applyNewSampleValues(samplingTotalWeight,
                                             samplingRatio);

        getTableModel().updateSamplingRatio(samplingContext.getSamplingRows());
    }

    protected void recomputeTotalHorsVrac() {

        // recompute total hors vrac
        Float totalHorsVrac = 0f;
        Float totalVrac = 0f;

        for (SpeciesBatchRowModel batch : getModel().getRows()) {
            SortedUnsortedCategory vracHorsVrac = batch.getSortedUnsortedCategory();

            if (vracHorsVrac == null) {

                // can't sum anything
            } else {
                boolean unsorted = "unsorted".equals(vracHorsVrac.getName());
                if (unsorted) {
                    Float weight = batch.getWeight();
                    if (weight != null) {
                        totalHorsVrac += weight;
                    }
                } else {
                    Float weight = batch.getWeight();
                    if (weight != null) {
                        totalVrac += weight;
                    }
                }
            }
        }
        if (log.isInfoEnabled()) {
            log.info("New total vrac / hors vrac: " +
                     totalVrac + " / " + totalHorsVrac);
        }
        getModel().setTotalHorsVracWeight(totalHorsVrac);
        //TODO Should we also set the total vrac weight ?
//        getModel().setTotalVracWeight(totalVrac);
    }
}
