package fr.ifremer.tutti.ui.swing.content.home;

/*
 * #%L
 * Tutti :: UI
 * $Id: SelectCruiseUIHandler.java 55 2012-12-13 01:57:28Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-0.2/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/home/SelectCruiseUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Lists;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.data.Program;
import fr.ifremer.tutti.persistence.entities.data.Cruise;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.ui.swing.AbstractTuttiUIHandler;
import fr.ifremer.tutti.ui.swing.TuttiScreen;
import fr.ifremer.tutti.ui.swing.TuttiUIContext;
import jaxx.runtime.swing.editor.bean.BeanComboBox;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.List;

/**
 * Main ui content to select cruise.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class SelectCruiseUIHandler extends AbstractTuttiUIHandler<SelectCruiseUIModel> {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(SelectCruiseUIHandler.class);

    protected final PersistenceService persistenceService;

    protected final SelectCruiseUI ui;

    public SelectCruiseUIHandler(TuttiUIContext context, SelectCruiseUI ui) {
        super(context);
        this.ui = ui;
        persistenceService = context.getService(PersistenceService.class);
    }

    @Override
    protected SelectCruiseUIModel getModel() {
        return ui.getModel();
    }

    @Override
    public void beforeInitUI() {
        SelectCruiseUIModel model = new SelectCruiseUIModel();

        List<Program> allPrograms = persistenceService.getAllProgram();

        model.setPrograms(allPrograms);

        if (allPrograms.isEmpty()) {
            // do nothing

            if (log.isDebugEnabled()) {
                log.debug("No program found.");
            }
        } else {

            // get selected program (if any)

            String programId = context.getProgramId();
            Program selectedProgram = null;
            List<Cruise> allCruises = null;
            if (programId != null) {
                selectedProgram = TuttiEntities.findById(allPrograms, programId);
                allCruises = persistenceService.getAllCruise(programId);
            }

            model.setProgram(selectedProgram);
            model.setCruises(allCruises);

            if (CollectionUtils.isEmpty(allCruises)) {

                // nothing to select
            } else {
                String cruiseId = context.getCruiseId();

                Cruise selectedCampaign = null;
                if (cruiseId != null) {
                    selectedCampaign =
                            TuttiEntities.findById(allCruises, cruiseId);
                }
                model.setCruise(selectedCampaign);
            }
        }
        ui.setContextValue(model);
    }

    @Override
    public void afterInitUI() {

        SelectCruiseUIModel model = ui.getModel();

        initBeanComboBox(ui.getProgramCombobox(),
                         model.getPrograms(),
                         model.getProgram());

        initBeanComboBox(ui.getCruiseCombobox(),
                         model.getCruises(),
                         model.getCruise());

        model.addPropertyChangeListener(SelectCruiseUIModel.PROPERTY_PROGRAM, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                Program newValue = (Program) evt.getNewValue();
                boolean noProgram = newValue == null;
                context.setProgramId(noProgram ? null : newValue.getId());
                List<Cruise> campaigns;
                if (noProgram) {
                    campaigns = Lists.newArrayList();
                } else {
                    campaigns = persistenceService.getAllCruise(newValue.getId());
                }
                SelectCruiseUIModel model = (SelectCruiseUIModel) evt.getSource();
                model.setCruises(campaigns);
                model.setCruise(null);
            }
        });

        model.addPropertyChangeListener(SelectCruiseUIModel.PROPERTY_CRUISES, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                // reload combo box
                BeanComboBox<Cruise> combobox = ui.getCruiseCombobox();
                List<Cruise> campaigns = (List<Cruise>) evt.getNewValue();
                combobox.setData(null);
                combobox.setData(campaigns);
            }
        });

        model.addPropertyChangeListener(SelectCruiseUIModel.PROPERTY_CRUISE, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                Cruise newValue = (Cruise) evt.getNewValue();
                context.setCruiseId(newValue == null ? null : newValue.getId());
            }
        });
    }

    @Override
    public void onCloseUI() {
        PropertyChangeListener[] listeners = ui.getModel().getPropertyChangeListeners();
        for (PropertyChangeListener listener : listeners) {
            getModel().removePropertyChangeListener(listener);
        }
    }

    public void showFishingOperations() {

        context.setScreen(TuttiScreen.EDIT_FISHING_OPERATION);
    }

    public void showSelectedProgram() {

        Program program = getModel().getProgram();
        context.setProgramId(program.getId());
        context.setScreen(TuttiScreen.EDIT_PROGRAM);
    }

    public void showNewProgram() {
        context.setProgramId(null);
        context.setCruiseId(null);
        context.setScreen(TuttiScreen.EDIT_PROGRAM);
    }

    public void showSelectedCruise() {
        Cruise cruise = getModel().getCruise();
        context.setCruiseId(cruise.getId());
        context.setScreen(TuttiScreen.EDIT_CRUISE);
    }

    public void showNewCruise() {
        context.setCruiseId(null);
        context.setScreen(TuttiScreen.EDIT_CRUISE);
    }
}
