package fr.ifremer.tutti.ui.swing.content.operation;

/*
 * #%L
 * Tutti :: UI
 * $Id: EditFishingOperationUIHandler.java 146 2013-01-02 18:51:13Z kmorin $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-0.2.5/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/EditFishingOperationUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Lists;
import fr.ifremer.tutti.persistence.entities.data.FishingOperation;
import fr.ifremer.tutti.persistence.entities.data.Program;
import fr.ifremer.tutti.persistence.entities.referential.FishingOperationLocation;
import fr.ifremer.tutti.persistence.entities.referential.Person;
import fr.ifremer.tutti.persistence.entities.referential.Zone;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.ui.swing.AbstractTuttiUIHandler;
import fr.ifremer.tutti.ui.swing.content.operation.fishing.environment.EnvironmentTabUI;
import fr.ifremer.tutti.ui.swing.content.operation.fishing.environment.EnvironmentTabUIModel;
import fr.ifremer.tutti.ui.swing.content.operation.fishing.gearshooting.GearShootingTabUI;
import fr.ifremer.tutti.ui.swing.content.operation.fishing.gearshooting.GearShootingTabUIModel;
import fr.ifremer.tutti.ui.swing.content.operation.fishing.hydrology.HydrologyTabUI;
import fr.ifremer.tutti.ui.swing.content.operation.fishing.hydrology.HydrologyTabUIModel;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import jaxx.runtime.swing.editor.bean.BeanDoubleListModel;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.swing.JLabel;
import javax.swing.JPanel;
import java.awt.BorderLayout;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.List;


/**
 * Handler for UI {@link EditFishingOperationUI}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class EditFishingOperationUIHandler extends AbstractTuttiUIHandler<EditFishingOperationUIModel> {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(EditFishingOperationUIHandler.class);

    private final EditFishingOperationUI ui;

    private final FishingOperationsUI parentUi;

    private final PersistenceService persistenceService;

    /**
     * To monitor changes on the incoming fishing operation.
     *
     * @since 0.3
     */
    private final TuttiBeanMonitor<EditFishingOperationUIModel> fishingOperationMonitor;

    public EditFishingOperationUIHandler(FishingOperationsUI parentUi,
                                         EditFishingOperationUI ui) {
        super(parentUi.getHandler().getContext());
        this.ui = ui;
        this.parentUi = parentUi;
        this.persistenceService = context.getService(PersistenceService.class);
        this.fishingOperationMonitor = new TuttiBeanMonitor<EditFishingOperationUIModel>(
                EditFishingOperationUIModel.PROPERTY_STATION_NUMBER,
                EditFishingOperationUIModel.PROPERTY_FISHING_OPERATION_NUMBER,
                EditFishingOperationUIModel.PROPERTY_DATE,
                EditFishingOperationUIModel.PROPERTY_STRATA,
                EditFishingOperationUIModel.PROPERTY_SUB_STRATA,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_START_LATITUDE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_START_LONGITUDE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_START_DATE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_END_LATITUDE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_END_LONGITUDE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_END_DATE,
                EditFishingOperationUIModel.PROPERTY_FISHING_OPERATION_RECTILIGNE,
                EditFishingOperationUIModel.PROPERTY_DISTANCE_CHALUTEE,
                EditFishingOperationUIModel.PROPERTY_DUREE,
                EditFishingOperationUIModel.PROPERTY_FISHING_OPERATION_VALID,
                EditFishingOperationUIModel.PROPERTY_LOCATION,
                EditFishingOperationUIModel.PROPERTY_COMMENT,
                EditFishingOperationUIModel.PROPERTY_SAISISSEUR);
    }

    @Override
    public void beforeInitUI() {

        EditFishingOperationUIModel model = new EditFishingOperationUIModel();
        model.addPropertyChangeListener(EditFishingOperationUIModel.PROPERTY_EMPTY, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                JPanel form = ui.getFishingOperationPane();
                JLabel noContentPane = ui.getNoTraitPane();

                Boolean empty = (Boolean) evt.getNewValue();
                ui.remove(form);
                ui.remove(noContentPane);
                if (empty) {
                    ui.add(noContentPane, BorderLayout.CENTER);
                } else {
                    ui.add(form, BorderLayout.CENTER);
                }
                ui.repaint();
            }
        });

        model.addPropertyChangeListener(EditFishingOperationUIModel.PROPERTY_STRATA, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                EditFishingOperationUIModel source = (EditFishingOperationUIModel) evt.getSource();
                FishingOperationLocation newStrata = (FishingOperationLocation) evt.getNewValue();
                onSelectedStrata(source, newStrata);
            }
        });

        model.addPropertyChangeListener(EditFishingOperationUIModel.PROPERTY_SUB_STRATA, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                EditFishingOperationUIModel source = (EditFishingOperationUIModel) evt.getSource();
                FishingOperationLocation newStrata = (FishingOperationLocation) evt.getNewValue();
                onSelectedSubStrata(source, newStrata);
            }
        });

//        listModelIsModify(model);

        ui.setContextValue(model);

        fishingOperationMonitor.setBean(model);
    }

    @Override
    public void afterInitUI() {

        ui.getFishingOperationValidPanel().remove(
                ui.getFishingOperationResetRadio());

        final EditFishingOperationUIModel model = getModel();

        initUI(ui);

        initBeanList(ui.getSaisisseurList(),
                     persistenceService.getAllPerson(),
                     model.getSaisisseur());
        final BeanDoubleListModel<Person> saisisseurModel = ui.getSaisisseurList().getModel();
//        saisisseurModel.getSelectedModel().addListDataListener(new ListDataListener() {
//
//            public void intervalAdded(ListDataEvent e) {
//                log.info(">>>>>>>>>> intervalAdded " + saisisseurModel.getSelected());
//                getModel().setSaisisseur(saisisseurModel.getSelected());
//            }
//
//            public void intervalRemoved(ListDataEvent e) {
//                log.info(">>>>>>>>>> intervalRemoved " + saisisseurModel.getSelected());
//                getModel().setSaisisseur(saisisseurModel.getSelected());
//            }
//
//            public void contentsChanged(ListDataEvent e) {
//                log.info(">>>>>>>>>> contentsChanged " + saisisseurModel.getSelected());
//                getModel().setSaisisseur(saisisseurModel.getSelected());
//            }
//        });

        String programId = context.getProgramId();

        Program program = persistenceService.getProgram(programId);
        Zone zone = program.getZone();

        FishingOperationLocation strata = model.getStrata();

        String strataId = strata == null ? null : strata.getId();

        FishingOperationLocation subStrata = model.getSubStrata();
        String substrataId = subStrata == null ? null : subStrata.getId();

        FishingOperationLocation location = model.getLocation();

        List<FishingOperationLocation> stratas =
                persistenceService.getAllFishingOperationStrata(zone.getId());

        List<FishingOperationLocation> subStratas;

        if (strataId == null) {
            subStratas = Lists.newArrayList();
        } else {

            // load substratas
            subStratas = persistenceService.getAllFishingOperationSubStrata(strataId);
        }

        List<FishingOperationLocation> locations;

        if (substrataId == null) {
            locations = Lists.newArrayList();
        } else {

            // load localites
            locations = persistenceService.getAllFishingOperationLocation(substrataId);
        }

        initBeanComboBox(ui.getStrataComboBox(), stratas, strata);

        initBeanComboBox(ui.getSubStrataComboBox(), subStratas, subStrata);

        initBeanComboBox(ui.getLocationComboBox(), locations, location);

        model.setEmpty(true);

        listenValidatorValid(ui.getValidator(), getModel());

        //init gear shooting
        GearShootingTabUIModel gearShootingModel =
                ui.getGearShootingTabContent().getModel();
        gearShootingModel.setAvailableCaracteristics(persistenceService.getAllFishingOperationGearCaracteristic());

        //init environment
        EnvironmentTabUIModel environmentModel =
                ui.getEnvironmentTabContent().getModel();
        environmentModel.setAvailableCaracteristics(persistenceService.getAllFishingOperationEnvironmentCaracteristic());

        //init hydrology
        HydrologyTabUIModel hydrologyModel = ui.getHydrologyTabContent().getModel();
        hydrologyModel.setAvailableCaracteristics(persistenceService.getAllFishingOperationHydrologicCaracteristic());
    }

    @Override
    public void onCloseUI() {

        if (log.isInfoEnabled()) {
            log.info("closing: " + ui);
        }

        closeUI(ui.getGearShootingTabContent());
        closeUI(ui.getEnvironmentTabContent());
        closeUI(ui.getHydrologyTabContent());
    }

    @Override
    protected EditFishingOperationUIModel getModel() {
        return ui.getModel();
    }

    public void selectFishingOperation(FishingOperation bean) {
        
        if (fishingOperationMonitor.wasModified()) {

            // previous fishingOperation was modified, let's save it
            EditFishingOperationUIModel beanToSave = fishingOperationMonitor.getBean();

            if (beanToSave.isEmpty()) {

                // user must use save button
                if (log.isWarnEnabled()) {
                    log.warn("Won't save new fishing operation, use explicit save button instead...");
                }
            } else {

                // save modified fishing operation
                FishingOperation toSave = beanToSave.toBean();

                showInformationMessage(
                        "[ Trait - Caractéristiques générales ] " +
                        "Sauvegarde des modifications de " + decorate(toSave) +
                        ".");

                if (log.isInfoEnabled()) {
                    log.info("FishingOperation " + toSave.getId() +
                             " was modified, will save it.");
                }

                fishingOperationMonitor.clearModified();

                // persist current fishingOperation
                persistenceService.saveFishingOperation(toSave);
                //parentUi.getHandler().saveFishingOperation(toSave);
            }
        }

        boolean empty = bean == null;

        EditFishingOperationUIModel model = getModel();

        if (empty) {

            bean = new FishingOperation();
        }

        FishingOperationLocation strata = bean.getStrata();
        FishingOperationLocation subStrata = bean.getSubStrata();
        FishingOperationLocation location = bean.getLocation();

        model.fromBean(bean);

        // to be sure saisisseurs are updated
//        model.setSaisisseur(Lists.newArrayList(bean.getSaisisseur()));

        // to be sure combo list will be reloaded
        model.setStrata(null);
        model.setSubStrata(null);
        model.setLocation(null);

        if (strata != null) {
            ui.getStrataComboBox().setSelectedItem(strata);
        }

        if (subStrata != null) {
            ui.getSubStrataComboBox().setSelectedItem(subStrata);
        }

        if (location != null) {
            ui.getLocationComboBox().setSelectedItem(location);
        }

        model.setFishingOperation(bean);

        // update saisissuer selection
        List<Person> saisisseur = model.getSaisisseur();
        if (log.isInfoEnabled()) {
            log.info("Selected saisisseurs: " + (saisisseur == null ? 0 : saisisseur.size()));
        }
        ui.getSaisisseurList().getModel().setSelected(saisisseur);

        // update model empty property
        model.setEmpty(empty);

        //reset gear shooting
        GearShootingTabUI gearShootingTab = ui.getGearShootingTabContent();
        gearShootingTab.getHandler().reset(bean);

        //reset environment
        EnvironmentTabUI environmentTab = ui.getEnvironmentTabContent();
        environmentTab.getHandler().reset(bean);

        //reset hydrology
        HydrologyTabUI hydrologyTab = ui.getHydrologyTabContent();
        hydrologyTab.getHandler().reset(bean);

        fishingOperationMonitor.clearModified();
    }

    public void cancel() {

        String id = getModel().getId();

        if (log.isInfoEnabled()) {
            log.info("Cancel edition for fishingOperation: " + id);
        }
        boolean newBean = StringUtils.isBlank(id);

        if (newBean) {

            // cancel to create a new fishingOperation
            parentUi.getHandler().selectFishingOperation(null);

        } else {

            // reselect in ui the fishingOperation
            FishingOperation selectedFishingOperation = parentUi.getModel().getSelectedFishingOperation();
            parentUi.getHandler().selectFishingOperation(selectedFishingOperation);
        }
    }

    public void save() {

        // get fishingOperation to save

        FishingOperation toSave = getModel().toBean();

        if (log.isInfoEnabled()) {
            log.info("Save edition for fishingOperation: " + toSave.getId());
        }

        // keep selected tab (to resotre it after save)
        int selectedIndex = ui.getFishingOperationTabPane().getSelectedIndex();

        // clean auto-save monitor to avoid re-entrant code
        fishingOperationMonitor.clearModified();

        // persist current fishingOperation
        parentUi.getHandler().saveFishingOperation(toSave);

        // reselect current tab
        ui.getFishingOperationTabPane().setSelectedIndex(selectedIndex);

//        getModel().setModify(false);
    }

    public void importCasino() {
        //TODO
    }

    public void setGearShootingStartLatitude(String text) {
        getModel().setGearShootingStartLatitude(Float.valueOf(text));
    }

    public void setGearShootingStartLongitude(String text) {
        getModel().setGearShootingStartLongitude(Float.valueOf(text));
    }

    public void setGearShootingEndLatitude(String text) {
        getModel().setGearShootingEndLatitude(Float.valueOf(text));
    }

    public void setGearShootingEndLongitude(String text) {
        getModel().setGearShootingEndLongitude(Float.valueOf(text));
    }

    protected void onSelectedStrata(EditFishingOperationUIModel model,
                                    FishingOperationLocation newStrata) {

        // reset substrata value
        model.setSubStrata(null);

        // reset sub strata combo
        ui.getSubStrataComboBox().setData(null);

        if (newStrata == null) {

            // reset strata, keep focus on it
            ui.getStrataComboBox().grabFocus();

        } else {

            // try to load substrata

            List<FishingOperationLocation> subStrata =
                    persistenceService.getAllFishingOperationSubStrata(newStrata.getId());

            ui.getSubStrataComboBox().setData(subStrata);

            if (CollectionUtils.isEmpty(subStrata)) {

                // try to load localite

                List<FishingOperationLocation> location =
                        persistenceService.getAllFishingOperationLocation(newStrata.getId());

                ui.getLocationComboBox().setData(location);

                ui.getLocationComboBox().grabFocus();
            }
        }
    }

    protected void onSelectedSubStrata(EditFishingOperationUIModel model,
                                       FishingOperationLocation newSubStrata) {
        // reset localite value
        model.setLocation(null);

        // reset localite combo
        ui.getLocationComboBox().setData(null);

        if (newSubStrata == null) {

            // reset substrata, keep focus on it
            ui.getSubStrataComboBox().grabFocus();
        } else {

            // try to load localite

            List<FishingOperationLocation> location =
                    persistenceService.getAllFishingOperationLocation(newSubStrata.getId());

            ui.getLocationComboBox().setData(location);
        }
    }
}
