/*
 * #%L
 * $Id: ResultHandler.java 4263 2015-06-11 09:01:47Z echatellier $
 * $HeadURL: http://svn.codelutin.com/isis-fish/trunk/src/main/java/fr/ifremer/isisfish/ui/result/ResultHandler.java $
 * %%
 * Copyright (C) 2011 - 2015 Ifremer, Codelutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.ui.result;

import static org.nuiton.i18n.I18n.t;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.beans.PropertyVetoException;
import java.io.File;
import java.util.List;

import javax.swing.JMenuItem;
import javax.swing.SwingUtilities;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.math.matrix.MatrixND;
import org.nuiton.math.matrix.viewer.MatrixViewerPanel;
import org.nuiton.math.matrix.viewer.renderer.MatrixChartRenderer;
import org.nuiton.math.matrix.viewer.renderer.MatrixPanelRenderer;
import org.nuiton.topia.TopiaException;
import org.nuiton.util.FileUtil;

import fr.ifremer.isisfish.IsisFishRuntimeException;
import fr.ifremer.isisfish.datastore.ExportStorage;
import fr.ifremer.isisfish.datastore.ResultStorage;
import fr.ifremer.isisfish.datastore.SimulationStorage;
import fr.ifremer.isisfish.datastore.StorageException;
import fr.ifremer.isisfish.entities.FisheryRegion;
import fr.ifremer.isisfish.export.ExportHelper;
import fr.ifremer.isisfish.logging.SimulationLoggerUtil;
import fr.ifremer.isisfish.simulator.launcher.SimulationJob;
import fr.ifremer.isisfish.simulator.launcher.SimulationService;
import fr.ifremer.isisfish.simulator.launcher.SimulationServiceListener;
import fr.ifremer.isisfish.ui.CommonHandler;
import fr.ifremer.isisfish.ui.models.common.GenericComboModel;

/**
 * Handler for result UIs.
 * 
 * @author chatellier
 * @version $Revision: 4263 $
 * 
 * Last update : $Date: 2015-06-11 11:01:47 +0200 (jeu., 11 juin 2015) $
 * By : $Author: echatellier $
 */
public class ResultHandler extends CommonHandler {

    private static Log log = LogFactory.getLog(ResultHandler.class);

    protected SimulationServiceListener listener = null;

    protected ResultUI resultUI;
    
    public ResultHandler(ResultUI resultUI) {
        this.resultUI = resultUI;
    }

    /**
     * Initialise la vue avec les liste des simulations disponible.
     */
    public void afterInit() {
        initSimulationList();

        // reference must stay allocated
        listener = new SimulationServiceListener() {
            @Override
            public void simulationStart(SimulationService simService, SimulationJob job) {
            }
            @Override
            public void simulationStop(SimulationService simService, final SimulationJob job) {
                final GenericComboModel<String> model = (GenericComboModel<String>)resultUI.getSimulationComboBox().getModel();
                SwingUtilities.invokeLater(new Runnable() {
                    public void run() {
                        model.addElement(job.getId());
                    }
                });
            }
            @Override
            public void clearJobDone(SimulationService simService) {
            }
        };
        SimulationService.getService().addSimulationServiceListener(listener);
    }

    /**
     * Reload simulation combo box model.
     */
    protected void initSimulationList() {
        List<String> value = SimulationStorage.getSimulationNames();
        
        //resultUI.getSimulationComboBox().setEditable(false);
        GenericComboModel<String> model = new GenericComboModel<String>(value);
        resultUI.getSimulationComboBox().setModel(model);
        resultUI.getSimulationComboBox().setSelectedItem(null);
    }

    /**
     * Open selected simulation item in new internal frame.
     */
    public void openNewSimulation() {
        String selected = (String)resultUI.getSimulationComboBox().getSelectedItem();
        SimulationStorage simulation = SimulationStorage.getSimulation(selected);

        ResultSimulationFrame internalFrame = new ResultSimulationFrame(resultUI);
        internalFrame.setTitle(simulation.getName());
        internalFrame.setSimulationStorage(simulation);
        init(internalFrame);
        internalFrame.setSize(800, 600);
        resultUI.getSimulationDesktopPane().add(internalFrame);
        try {
            internalFrame.setSelected(true);
        } catch (PropertyVetoException ex) {
            if (log.isWarnEnabled()) {
                log.warn("Can't auto selected internal frame", ex);
            }
        }
    }

    /**
     * Delete selected simulation.
     */
    public void deleteSimulation() {
        String selected = (String)resultUI.getSimulationComboBox().getSelectedItem();
        SimulationStorage simulation = SimulationStorage.getSimulation(selected);
        try {
            simulation.delete(false);
        } catch (StorageException ex) {
            throw new IsisFishRuntimeException("Can't delete simulation", ex);
        }
        initSimulationList();
    }

    /**
     * Display simulation log.
     */
    public void showLog() {
        String selected = (String)resultUI.getSimulationComboBox().getSelectedItem();
        try {
            SimulationLoggerUtil.showSimulationLogConsole(resultUI, selected);
        } catch (Exception ex) {
            throw new IsisFishRuntimeException("Can't display simulation log", ex);
        }
    }

    /**
     * Init ResultSimulationFrame.
     * 
     * @param resultSimulationFrame resultSimulationFrame
     */
    public void init(final ResultSimulationFrame resultSimulationFrame) {
        SimulationStorage simulation = resultSimulationFrame.getSimulationStorage();
        try {
            // init menu
            for (String exportName : ExportStorage.getExportNames()) {
                JMenuItem item = new JMenuItem(t(exportName));
                item.addActionListener(new ExportActionListener(simulation, exportName));
                resultSimulationFrame.getExportMenu().add(item);
            }

            resultSimulationFrame.topiaContext = simulation.getStorage().beginTransaction();
            final ResultStorage resultStorage = simulation.getResultStorage();
            FisheryRegion fisheryRegion = SimulationStorage.getFisheryRegion(resultSimulationFrame.topiaContext);

            // init viewer panel
            final MatrixViewerPanel matrixViewerPanel = new MatrixViewerPanel();
            matrixViewerPanel.addMatrixDimentionAction(new YearSumDimensionAction());
            matrixViewerPanel.addMatrixRenderer(new MatrixSummaryRenderer(simulation, resultStorage, resultSimulationFrame.topiaContext), true);
            matrixViewerPanel.addMatrixRenderer(new MatrixChartRenderer());
            matrixViewerPanel.addMatrixRenderer(new MatrixMapRenderer(fisheryRegion));
            matrixViewerPanel.addMatrixRenderer(new MatrixPanelRenderer());
            matrixViewerPanel.addMatrixFilter(new ResultMatrixFilter(resultStorage));
            matrixViewerPanel.addMatrixFilter(new SumByYearMatrixFilter());

            // init available results list
            GenericComboModel<String> model = new GenericComboModel<>(resultStorage.getResultName());
            resultSimulationFrame.getResultsComboBox().setModel(model);
            resultSimulationFrame.getResultsComboBox().addItemListener(new ItemListener() {
                @Override
                public void itemStateChanged(ItemEvent e) {
                    if (e.getStateChange() == ItemEvent.SELECTED) {
                        String selectedMatrixName = (String)e.getItem();
                        MatrixND matrix = resultStorage.getMatrix(selectedMatrixName, resultSimulationFrame.topiaContext);
                        matrixViewerPanel.setMatrix(matrix);
                    }
                }
            });

            // force summary display
            matrixViewerPanel.updateSelectedRenderingComponent();
            resultSimulationFrame.getMatrixViewerContainer().add(matrixViewerPanel, BorderLayout.CENTER);
            
            if (model.getSize() > 0) {
                // FIXME poussin 20140721 est-ce bien intelligent de forcer l'ouverture
                // du 1er resultat trouver. On prend du temps pour ca, alors que l'utilisateur
                // veut potentiellement regarder autre chose. Demander au utilisateur si on peut
                // ne pas ouvrir de resultat par defaut.
                model.setSelectedItem(model.getElementAt(0));
            }
        } catch (TopiaException ex) {
            throw new IsisFishRuntimeException("Can't open simulation", ex);
        } catch (StorageException ex) {
            throw new IsisFishRuntimeException("Can't open simulation", ex);
        }
    }

    protected class ExportActionListener implements ActionListener {

        protected SimulationStorage simulationStorage;

        protected String exportName;

        public ExportActionListener(SimulationStorage simulationStorage, String exportName) {
            this.simulationStorage = simulationStorage;
            this.exportName = exportName;
        }

        public void actionPerformed(ActionEvent e) {
            try {
                File file = FileUtil.getFile(".+\\.csv(\\.gz)?", t("isisfish.result.export.file"));
                if (file != null) {
                    // add csv extension if not set
                    if (!file.getName().matches(".+\\.csv(\\.gz)?")) {
                        file = new File(file.getAbsolutePath() + ".csv.gz");
                    }
                    ExportHelper.exportToFile(simulationStorage, exportName, file);
                }
            } catch(Exception eee) {
                if (log.isWarnEnabled()) {
                    log.warn("Erreur lors de l'export ", eee);
                }
            }
        }
    }
}
