/* *##% ToPIA - UI
 * Copyright (C) 2004 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*/

package org.nuiton.topia.generator;

import java.io.File;
import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.eugene.MonitorWriter;
import org.nuiton.eugene.StateModelGenerator;
import org.nuiton.eugene.GeneratorUtil;
import org.nuiton.eugene.models.state.StateModel;
import org.nuiton.eugene.models.state.StateModelComplexState;
import org.nuiton.eugene.models.state.StateModelSimpleState;
import org.nuiton.eugene.models.state.StateModelState;
import org.nuiton.eugene.models.state.StateModelStateChart;
import org.nuiton.eugene.models.state.StateModelTransition;

/**
 * Genere la navigation basee sur le framework Tapestry 5
 * a partir de la representation StateModel
 * 
 * @see org.nuiton.eugene.StateModelGenerator
 * @author chatellier
 * @version $Revision: 1668 $
 *
 * Last update : $Date: 2009-11-13 14:01:01 +0100 (Fri, 13 Nov 2009) $
 * By : $Author: fdesbois $
 */
public class TapestryWebGenerator extends StateModelGenerator {

	/** logger */
	private static final Log log = LogFactory.getLog(TapestryWebGenerator.class);

	/** Already asssigned use case states names */
	protected List<String> assignedUseCaseStateNames;

    protected StateModel model;

	/**
	 * Default Constructeur
	 */
	public TapestryWebGenerator() {
		assignedUseCaseStateNames = new ArrayList<String>();
	}

	/**
	 * Redefintion pour une generation specifique tapestry
	 * 
	 * @param stateModel Le modele d'état
	 * @param destDir le dossier de destination
	 */
	@Override
    public void applyTemplate(StateModel stateModel, File destDir) throws IOException {

        this.model = stateModel;

		// pour tous les diagramme du model
		for (StateModelStateChart chart : stateModel.getStateCharts()) {
			generate(chart,chart.getStates(),destDir);

			// generate use case engine
			generateUseCaseEngineFromModel(stateModel, chart,destDir);
		}
	}

	/**
	 * Appele par lutin generator pour tous les etats du model
	 * @param chart
	 * @return filename
	 */
	protected String getFilenameFromState(StateModelStateChart chart,StateModelState state) {
		return (getPackageFromState(chart) + '.' + getNameFromState(state)).replace('.',
				File.separatorChar) + ".java";
	}

	/**
	 * Get generated class name from state
	 * 
	 * Prefix it with "Abstract"
	 * 
	 * @param state state
	 * @return name
	 */
	protected String getNameFromState(StateModelState state) {
		return "Abstract" +  GeneratorUtil.toUpperCaseFirstLetter(state.getName());
	}

	/**
	 * Return specifique tapestry package name.
	 * 
	 * Detect "web" patern and replace it with "web.pages".
	 * 
	 * @param chart the chart
	 * @return a tapestry package name
	 */
	protected String getPackageFromState(StateModelStateChart chart) {
		return chart.getPackageName().replaceFirst("web", "web.pages");
	}

	/**
	 * Return specifique tapestry package name (base)
	 * 
	 * Detect "web.*" patern and replace it with "web.base"
	 * Component are placed into "base" directory 
	 * 
	 * @param chart the chart
	 * @return a tapestry package name
	 */
	protected String getPackageFromComponents(StateModelStateChart chart) {
		return chart.getPackageName().replaceFirst("web.*", "web.base");
	}

	/**
	 * Generate a collection of states
	 * 
	 * @param chart the chart
	 * @param states states collection
	 * @throws IOException 
	 */
	protected void generate(StateModelStateChart chart, Collection<StateModelState> states, File destDir) throws IOException {

		// et tous les états de ces diagrammes
		for (Object oState : states.toArray()) {
			generateFromState(chart,(StateModelState)oState,destDir);
		}
	}

	/**
	 * Generate a state.
	 * 
	 * This state can be complexe, so the method is recusively called.
	 * 
	 * @param chart the parent chart
	 * @param state the current state
	 * @throws IOException
	 */
	protected void generateFromState(StateModelStateChart chart, StateModelState state, File destDir) throws IOException {

		// complexe
		if(state.isComplex()) {
			StateModelComplexState complexeState = (StateModelComplexState)state;
			generate(chart,complexeState.getStates(), destDir);
		}
		// simple
		else {

			StateModelSimpleState simpleState = (StateModelSimpleState)state;

			// les etat initiaux et finaux ne donne pas
			// pas de generation de fichier
			if(!simpleState.isFinal() && !simpleState.isInitial()) {

				String filename = getFilenameFromState(chart, simpleState);
				File outputFile = getDestinationFile(destDir, filename);
				if (getOverwrite() || !isNewerThanSource(outputFile)) {
					try {
						StringWriter out = new StringWriter();
						MonitorWriter monitorOut = new MonitorWriter(out);
						generateFromSimpleState(monitorOut, chart, simpleState);
						write(outputFile, monitorOut);
					} catch (Exception eee) {
						log.warn("Erreur lors de la génération du fichier "
								+ outputFile);
						throw new RuntimeException(
								"Erreur lors de la génération du fichier "
										+ outputFile, eee);
					}
				}
			} // init && final
		}
	}

	/**
	 * Called for each simple state
	 * 
	 * @param output writer out
	 * @param chart chart
	 * @param state simple state
	 * @throws IOException
	 */
	protected void generateFromSimpleState(Writer output, StateModelStateChart chart, StateModelState state) throws IOException {
	    String copyright = model.getTagValue("copyright");
        if (copyright != null && !copyright.isEmpty()) {
/*{<%=copyright%>
}*/
        }
/*{// Automatically generated by LutinGenerator
package <%=getPackageFromState(chart)%>;

import org.apache.tapestry5.annotations.InjectPage;
import <%=getPackageFromComponents(chart)%>.UseCasePage;

/*
 * State <%=getNameFromState(state)%>
 *)
public abstract class <%=getNameFromState(state)%> extends UseCasePage {
}*/

		// generate events
        generateInjectionAndEventsFromState(output,chart,state);

        // generate use case name
        generateUseCaseNameFromState(output,chart,state);
/*{}}*/
    }

    /**
     * Generate state injections
     * @param output
     * @param chart
     * @param state
     */
    @SuppressWarnings("unused")
    protected void generateInjectionAndEventsFromState(Writer output, StateModelStateChart chart, StateModelState state) throws IOException {

        // liste les états de destination (pour éviter les doublons)
        List<StateModelState> destStates = new ArrayList<StateModelState>();

        for(StateModelTransition transition : state.getTransitions()) {

            StateModelState toState = transition.getDestinationState();

            if(toState.isComplex()) {
                StateModelState toInitState = ((StateModelComplexState)toState).getInitialState();

                // l'attribute doit etre private
                // sinon tapestry ne le traite pas

                // si l'état de destination n'a pas encore été traité
                if(!destStates.contains(toState)){
                    destStates.add(toState);
/*{
    /* linked state "<%=toState.getName()%>" *)
    @InjectPage
    private <%=GeneratorUtil.toUpperCaseFirstLetter(toInitState.getName())%> <%=GeneratorUtil.toLowerCaseFirstLetter(toInitState.getName())%>;
}*/
                // l'attribut etant prive, il faut un getter
/*{
    /* getter for state "<%=toState.getName()%>" *)
    protected <%=GeneratorUtil.toUpperCaseFirstLetter(toInitState.getName())%> get<%=GeneratorUtil.toUpperCaseFirstLetter(toInitState.getName())%>() {
        return <%=GeneratorUtil.toLowerCaseFirstLetter(toInitState.getName())%>;
    }
}*/
                }
/*{
    /* transition on <%=transition.getEvent()%> event *)
    public Object onActionFrom<%=GeneratorUtil.toUpperCaseFirstLetter(transition.getEvent())%>() {
        enterUseCase();
        return <%=GeneratorUtil.toLowerCaseFirstLetter(toInitState.getName())%>;
    }
}*/
            }
            else {

                StateModelSimpleState simpleToState = (StateModelSimpleState)toState;

                // si l'etat a injecter n'est pas final
                if(!simpleToState.isFinal()) {

                    // si l'état de destination n'a pas encore été traité
                    if(!destStates.contains(toState)){
                        destStates.add(toState);
/*{
    /* linked state "<%=simpleToState.getName()%>" *)
    @InjectPage
    private <%=GeneratorUtil.toUpperCaseFirstLetter(simpleToState.getName())%> <%=GeneratorUtil.toLowerCaseFirstLetter(simpleToState.getName())%>;
}*/
                    // l'attribut etant prive, il faut un getter
/*{
    /* getter for state "<%=simpleToState.getName()%>" *)
    protected <%=GeneratorUtil.toUpperCaseFirstLetter(simpleToState.getName())%> get<%=GeneratorUtil.toUpperCaseFirstLetter(simpleToState.getName())%>() {
        return <%=GeneratorUtil.toLowerCaseFirstLetter(simpleToState.getName())%>;
    }
}*/
                    }
                } // isFinal
/*{
    /* transition on "<%=transition.getEvent()%>" event *)
    public Object onActionFrom<%=GeneratorUtil.toUpperCaseFirstLetter(transition.getEvent())%>() {
}*/
                if(simpleToState.isFinal()) {
/*{        return leaveUseCase();
}*/
                } else {
/*{        return <%=GeneratorUtil.toLowerCaseFirstLetter(simpleToState.getName())%>;
}*/
                }
/*{    }
}*/
            }
        }
    }

    /**
     * Generate use case name
     * 
     * @param output
     * @param chart
     * @param state
     */
    protected void generateUseCaseNameFromState(Writer output, StateModelStateChart chart, StateModelState state) throws IOException {

        String stateId = generateStateUseCaseName(state.getName());
/*{
    /* return a unique state id for model *)
    protected final String getUseCaseName() {
        return "<%=stateId%>";
    }
}*/
    }
    
    /**
     * Genere un nom unique d'etat pour le model courant.
     * 
     * Pour des raisons de debugage, on utilise des lettres particuliere du
     * nom de l'etat, suivit d'un numero, s'il y a collision.
     * 
     * @return a unique name
     */
    protected String generateStateUseCaseName(String name) {

        String ucsn = getHashStateName(name);

        // test already assigned
        String uniqueucsn = ucsn;
        int num = 2;
        while(assignedUseCaseStateNames.contains(uniqueucsn)) {
            uniqueucsn = new StringBuffer(ucsn).append(num++).toString();
        }

        assignedUseCaseStateNames.add(uniqueucsn);

        return uniqueucsn;
    }

    /**
     * Hash a state name.
     * 
     * On prend la premiere lettre de chaque mot.
     * 
     * ContactList     -> cl
     * contactList     -> cl
     * ContactListPage -> clp
     * 
     * @param name
     * @return hashed state name
     */
    protected String getHashStateName(String name) {

        // bug si moins de 1 caractere
        if(name == null || name.length()<1) {
            return name;
        }

        StringBuffer sb = new StringBuffer();

        // on prend la premiere lettre
        sb.append(Character.toLowerCase(name.charAt(0)));

        // ensuite, on prend les majuscules
        for(int i = 1 ; i < name.length() ; ++i) {

            // si maj
            if(Character.isUpperCase(name.charAt(i))) {
                sb.append(Character.toLowerCase(name.charAt(i)));
            }
        }

        return sb.toString();
    }

    /**
     * Generate the engine use case class.
     * 
     * Need to be generated inside client webapp directory to be analysed by
     * tapestry (annotations)
     * 
     * @param model model
     * @param chart chart
     */
    protected void generateUseCaseEngineFromModel(StateModel model,StateModelStateChart chart,File destDir) throws IOException {
        String componentPackageName = getPackageFromComponents(chart);

        // UseCasePage
        String filename = (componentPackageName + ".UseCasePage").replace('.', File.separatorChar) + ".java";
        File outputFile = getDestinationFile(destDir, filename);
        if (getOverwrite() || !isNewerThanSource(outputFile)) {
            try {
                StringWriter out = new StringWriter();
                MonitorWriter monitorOut = new MonitorWriter(out);
                // generate UseCasePage class
                generateUseCasePageClass(monitorOut,model,componentPackageName);
                write(outputFile, monitorOut);
            } catch (Exception e) {
                log.warn("Erreur lors de la génération du fichier UseCasePage",e);
                throw new RuntimeException(
                        "Erreur lors de la génération du fichier UseCasePage", e);
            }
        }

        // UseCaseMap
        filename = (componentPackageName + ".UseCaseMap").replace('.', File.separatorChar) + ".java";
        outputFile = getDestinationFile(destDir, filename);
        if (getOverwrite() || !isNewerThanSource(outputFile)) {
            try {
                StringWriter out = new StringWriter();
                MonitorWriter monitorOut = new MonitorWriter(out);
                // generate UseCasePage class
                generateUseCaseMapClass(monitorOut,componentPackageName);
                write(outputFile, monitorOut);
            } catch (Exception e) {
                log.warn("Erreur lors de la génération du fichier UseCaseMap",e);
                throw new RuntimeException(
                        "Erreur lors de la génération du fichier UseCaseMap", e);
            }
        }
    }

    /**
     * Generate base.UseCasePage class
     * @param output
     * @param componentPackageName
     */
    private void generateUseCasePageClass(MonitorWriter output, StateModel model, String componentPackageName) throws IOException {
/*{package <%=componentPackageName%>;

import org.apache.tapestry5.Link;
import org.apache.tapestry5.annotations.SessionState;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.annotations.Service;
import org.apache.tapestry5.services.LinkCreationHub;
import org.apache.tapestry5.services.LinkCreationListener;
import org.apache.tapestry5.services.Request;

public abstract class UseCasePage}*/

        // ajoute un heritage si il a ete specifie dans un fichier de propriete
        Map<String,String> tagValues = model.getTagValues();
        if(tagValues != null) {
            String tagUseCaseEngineExtendedClass = tagValues.get("usecaseengineextendedclass");
            if(tagUseCaseEngineExtendedClass != null) {
                /*{ extends <%=tagUseCaseEngineExtendedClass%>}*/
            }
        }

/*{ implements LinkCreationListener {

    private static final String UC_PARAMETER_NAME = "UC";
    private static final String UC_PARAMETER_SEPARATOR = ":";
    
    @SessionState
    private UseCaseMap useCaseManager;
    private boolean useCaseManagerExists;
    
    @Inject
    @Service("LinkCreationHub")
    private LinkCreationHub linkFactory;
    
    @Inject
    @Service("Request")
    private Request request;

    private String UCName;
    
    public void onActivate() {
        linkFactory.addListener(this);
        
        UCName = request.getParameter(UC_PARAMETER_NAME);
    }
    
    protected abstract String getUseCaseName();
    
    /**
     * Create stack if not exists
     *)
    protected void init() {
        if (!useCaseManagerExists) {
            useCaseManager = new UseCaseMap();
        }
    }
    /**
     * Enter in a new sub use case
     *)
    protected void enterUseCase() {
        init();
        // current
        String currentUCN = UCName;
        // new
        UCName = (currentUCN == null) ? getUseCaseName() : currentUCN + UC_PARAMETER_SEPARATOR + getUseCaseName();
        useCaseManager.enterSubUseCase(UCName, currentUCN, this);
    }
    
    /**
     * Leave a sub use case
     *)
    protected UseCasePage leaveUseCase() {
        init();
        
        UseCasePage nextPage = null;
        
        // current UCN
        String UC = UCName;

        // get state
        Object[] response = useCaseManager.leaveUseCase(UC);
        
        // if null, return null, don't change page
        if(response != null) {
            // get UCN before enter this one
            UCName = (String) response[0];
        
            nextPage = (UseCasePage) response[1];
        }

        return nextPage;
    }

    /*
     * @see org.apache.tapestry5.services.LinkCreationListener#createdComponentEventLink(org.apache.tapestry5.Link)
     *)
    @Override
    public void createdComponentEventLink(Link link) {
        addUCParameter(link);
    }

    /*
     * @see org.apache.tapestry5.services.LinkCreationListener#createdPageRenderLink(org.apache.tapestry5.Link)
     *)
    @Override
    public void createdPageRenderLink(Link link) {
        // PageLink englobe ausssi les redirects envoyés au client apres une
        // action
        addUCParameter(link);
    }
    
    protected void addUCParameter(Link link) {
        if(link.getParameterValue(UC_PARAMETER_NAME) == null) {

            if(UCName != null) {
                link.addParameter(UC_PARAMETER_NAME, UCName);
            }
        }
    }
}
}*/
    }

    /**
     * Generate base.UseCaseMap class 
     * @param output
     * @param componentPackageName
     */
    private void generateUseCaseMapClass(MonitorWriter output, String componentPackageName) throws IOException {
/*{package <%=componentPackageName%>;

import java.util.HashMap;
import java.util.Map;

public class UseCaseMap {
    
    /**
     * Map (state name -> state instance)
     *)
    private Map<String,Object[]> mapUseCase;
    
    /**
     * Constructor
     *)
    public UseCaseMap () {
        mapUseCase = new HashMap<String,Object[]>();
    }
    
    /**
     * New sub use case
     *)
    public void enterSubUseCase(String newUseCaseName, String previousUseCaseName, Object state) {
        mapUseCase.put(newUseCaseName,new Object[]{previousUseCaseName,state});
    }
    
    /**
     * Finish a subusecase
     * 
     * Return the state that init the subuse case
     * @return a state
     *)
    public Object[] leaveUseCase(String useCaseName) {
        return mapUseCase.get(useCaseName);
    }
}
}*/
    }
}
