/* *##% 
 * ToPIA :: Service Migration
 * Copyright (C) 2004 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*/
package org.nuiton.topia.migration.kernel;

import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.migration.common.ProxyClass;
import org.nuiton.topia.migration.common.SimpleProxyClass;
import org.hibernate.SessionFactory;
import org.hibernate.cfg.Configuration;
import org.hibernate.engine.CascadeStyle;
import org.hibernate.engine.SessionFactoryImplementor;
import org.hibernate.mapping.PersistentClass;
import org.hibernate.mapping.Property;
import org.hibernate.mapping.Subclass;
import org.hibernate.persister.entity.Joinable;
import org.hibernate.type.AssociationType;
import org.hibernate.type.CollectionType;
import org.hibernate.type.Type;

/**
 * DependenciesHelper.java
 *
 * Permet de charger un fichier de mapping et de former les dependances entre les classes mappees
 * 
 * @author Chatellier Eric
 * @author Chevallereau Benjamin
 * @author Eon Sébastien
 * @author Trève Vincent
 * @version $Revision: 1459 $
 * 
 * Last update : $Date: 2009-05-16 09:56:47 +0200 (Sat, 16 May 2009) $
 */
public class DependenciesHelper {

    /**
     * La session factory, hibernate en a besoin pour determiner les liens
     */
    protected SessionFactory sessionFactory;
    /**
     * La configuration contenant le schema
     */
    protected Configuration configuration;
    /**
     * Logger (common-logging)
     */
    private Log logger = LogFactory.getLog(DependenciesHelper.class);
    /**
     * Liste des classes dépendantes pour chaque classe
     */
    protected Map<ProxyClass, List<ProxyClass>> mDependencies;
    /**
     * Liste des classes dépendantes d'une classe
     */
    protected Map<ProxyClass, List<ProxyClass>> mInvertDependencies;

    /**
     * Constructeur
     * @param sessionFactory l'usine de sessions
     * @param configuration une Configuration
     */
    public DependenciesHelper(SessionFactory sessionFactory, Configuration configuration) {
        super();
        this.configuration = configuration;
        this.sessionFactory = sessionFactory;

        calculateDependencies();

        if (logger.isDebugEnabled()) {
            for (ProxyClass clazz : mDependencies.keySet()) {
                logger.debug("Dep. for class : " + clazz.getCanonicalName() + "=" + mDependencies.get(clazz));
            }
            for (ProxyClass clazz : mInvertDependencies.keySet()) {
                logger.debug("Rev Dep. for class : " + clazz.getCanonicalName() + "=" + mInvertDependencies.get(clazz));
            }
        }
    }

    /**
     * Calcule les dependances
     */
    protected void calculateDependencies() {

        mDependencies = new Hashtable<ProxyClass, List<ProxyClass>>();
        mInvertDependencies = new Hashtable<ProxyClass, List<ProxyClass>>();

        //FIXED apparement on est oblige d'avoir un sessionFactory
        // pour que hibernate nous fournisse les relations (type)
        SessionFactoryImplementor sfi = (SessionFactoryImplementor) this.sessionFactory;

        //on parcourt les entites
        Iterator<?> im = configuration.getClassMappings();
        while (im.hasNext()) {
            PersistentClass m = (PersistentClass) im.next();

            //on recupere le nom de la classe courante
            String currentClass = m.getEntityName();
            ProxyClass pcCurrentClass = new SimpleProxyClass(currentClass);

            //on initialise les listes correspondantes si ce n'est pas deja fait
            List<ProxyClass> lDependencies = new LinkedList<ProxyClass>();
            List<ProxyClass> lOldDependencies = mDependencies.get(pcCurrentClass);
            if (lOldDependencies != null) {
                lDependencies.addAll(lOldDependencies);
            }
            mDependencies.put(pcCurrentClass, lDependencies);

            if (mInvertDependencies.get(pcCurrentClass) == null) {
                mInvertDependencies.put(pcCurrentClass, new LinkedList<ProxyClass>());
            }

            PersistentClass clazz = configuration.getClassMapping(currentClass);

            // gestion de l'heritage
            if (clazz instanceof Subclass) {
                Subclass usb = (Subclass) clazz;
                String superClassName = usb.getSuperclass().getEntityName();
                ProxyClass pcDependentClass = new SimpleProxyClass(superClassName);
                if (!lDependencies.contains(pcDependentClass)) {
                    lDependencies.add(pcDependentClass);
                }

                //classe dont depend la classe courante -> classe courante
                List<ProxyClass> lOldDependents = mInvertDependencies.get(pcDependentClass);
                if (lOldDependents == null) {
                    lOldDependents = new LinkedList<ProxyClass>();
                    mInvertDependencies.put(pcDependentClass, lOldDependents);
                }
                if (!lOldDependents.contains(pcCurrentClass)) {
                    lOldDependents.add(pcCurrentClass);
                }
            }

            Iterator<?> i = clazz.getPropertyIterator();
            //on parcourt les proprietes de la classe courante
            while (i.hasNext()) {
                Property p = (Property) i.next();
                Type propertyType = p.getType();



                // si c'est une association
                if (propertyType.isAssociationType()) {

                    AssociationType a = (AssociationType) propertyType;
                    Joinable j = a.getAssociatedJoinable(sfi);
                    String s = a.getAssociatedEntityName(sfi);

                    String dependentClassName = null;

                    // on recupere le nom de la classe dont depend la classe courante
                    if (propertyType.isCollectionType()) {
                        //<list name="produits" cascade="all">
                        //  <key column="clients"/>
                        //  <index/>
                        //  <one-to-many class="domaine.Produit"/>
                        //</list>

                        CollectionType collectionType = (CollectionType) propertyType;

                        dependentClassName = collectionType.getAssociatedEntityName(sfi);
                    } else {

                        if (!p.getCascadeStyle().equals(CascadeStyle.NONE)) {
                            //<one-to-one name="produit" class="domaine.Produit" />
                            //<many-to-one name="produit" class="domaine.Produit" />
                            dependentClassName = propertyType.getName();
                        }
                    }

                    // essaye pour gerer les boucles dues aux doubles
                    // visibilites d'associations
                    /*if(j instanceof AbstractCollectionPersister) {
                    AbstractCollectionPersister acp = (AbstractCollectionPersister)j;

                    // aggragate ?
                    if(!acp.isInverse() || acp.hasOrphanDelete()) {
                    dependentClassName = s;

                    //if(mDependencies.get(new SimpleProxyClass(dependentClassName)) != null && mDependencies.get(new SimpleProxyClass(dependentClassName)).contains(pcCurrentClass)) {
                    //    mDependencies.get(new SimpleProxyClass(dependentClassName)).remove(pcCurrentClass);
                    //}
                    }
                    }
                    else {
                    AbstractEntityPersister aep = (AbstractEntityPersister)j;
                    if(aep.getPropertyUpdateability() != null && aep.getPropertyUpdateability()[0]) {
                    dependentClassName = propertyType.getName();
                    }
                    }*/

                    if (dependentClassName != null) {
                        //on l'ajoute dans les listes correspondantes
                        ProxyClass pcDependentClass = new SimpleProxyClass(dependentClassName);
                        //classe courante -> classe dont elle depend
                        if (!lDependencies.contains(pcDependentClass)) {
                            lDependencies.add(pcDependentClass);
                        }

                        //classe dont depend la classe courante -> classe courante
                        List<ProxyClass> lOldDependents = mInvertDependencies.get(pcDependentClass);
                        if (lOldDependents == null) {
                            lOldDependents = new LinkedList<ProxyClass>();
                            mInvertDependencies.put(pcDependentClass, lOldDependents);
                        }
                        if (!lOldDependents.contains(pcCurrentClass)) {
                            lOldDependents.add(pcCurrentClass);
                        }
                    }
                }
            }
        }
    }

    /**
     * Permet de recuperer la liste des classes dont depend la Classe passee en parametre
     * @param pc une Classe
     * @return la liste des classes dont depend la Classe passee en parametre
     */
    public Set<ProxyClass> getDependencies(ProxyClass pc) {
        if (mDependencies.get(pc) == null) {
            return null;
        }
        Set<ProxyClass> lRecursiveDependencies = new HashSet<ProxyClass>(mDependencies.get(pc));
        Set<ProxyClass> result = new HashSet<ProxyClass>(lRecursiveDependencies);
        //IL NE FAUT PAS QU'IL Y AIT DE CYCLE
        for (ProxyClass dependency : lRecursiveDependencies) {
            if (mInvertDependencies.get(dependency) == null) {
                result.addAll(getDependencies(dependency));
            }
        }

        return result;
    }

    /**
     * Permet de recuperer la liste des classes qui dependent de la Classe passee en parametre
     * @param pc une Classe
     * @return la liste des classes qui dependent de la Classe passee en parametre
     */
    public Set<ProxyClass> getInvertDependencies(ProxyClass pc) {
        if (mInvertDependencies.get(pc) == null) {
            return null;
        }
        return new HashSet<ProxyClass>(mInvertDependencies.get(pc));
    }
}
