package org.nuiton.topia.persistence.internal;

/*
 * #%L
 * ToPIA :: Persistence
 * $Id: HibernateProvider.java 3048 2014-04-04 10:34:07Z athimel $
 * $HeadURL: https://svn.nuiton.org/topia/tags/topia-3.0-beta-2/topia-persistence/src/main/java/org/nuiton/topia/persistence/internal/HibernateProvider.java $
 * %%
 * Copyright (C) 2004 - 2014 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import java.io.File;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.SessionFactory;
import org.hibernate.boot.registry.StandardServiceRegistry;
import org.hibernate.boot.registry.StandardServiceRegistryBuilder;
import org.hibernate.cfg.Configuration;
import org.hibernate.engine.jdbc.connections.spi.ConnectionProvider;
import org.hibernate.engine.spi.SessionFactoryImplementor;
import org.hibernate.event.service.spi.EventListenerRegistry;
import org.hibernate.event.spi.EventType;
import org.hibernate.service.Service;
import org.hibernate.service.ServiceRegistry;
import org.hibernate.service.spi.SessionFactoryServiceRegistry;
import org.hibernate.service.spi.Stoppable;
import org.nuiton.topia.persistence.TopiaConfigurationConstants;
import org.nuiton.topia.persistence.TopiaNotFoundException;
import org.nuiton.topia.persistence.TopiaService;
import org.nuiton.topia.persistence.internal.support.TopiaHibernateEventListener;
import org.nuiton.topia.persistence.support.TopiaServiceSupport;
import org.nuiton.topia.persistence.util.TopiaUtil;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;

/**
 * @author Arnaud Thimel (Code Lutin)
 */
public class HibernateProvider {

    private static final Log log = LogFactory.getLog(HibernateProvider.class);

    protected SessionFactory hibernateSessionFactory;
    protected Configuration hibernateConfiguration;

    protected Map<String, String> configuration;
    protected TopiaServiceSupport topiaServiceSupport;
    protected TopiaHibernateSessionRegistry sessionRegistry;

    /**
     * List of persistent classes
     */
    protected List<Class<?>> persistentClasses = Lists.newArrayList();

    public HibernateProvider(Map<String, String> configuration,
                             TopiaServiceSupport topiaServiceSupport,
                             TopiaHibernateSessionRegistry sessionRegistry) {
        this.configuration = configuration;
        this.topiaServiceSupport = topiaServiceSupport;
        this.sessionRegistry = sessionRegistry;
    }

    protected String getProperty(String key) {
        return getProperty(key, null);
    }

    protected String getProperty(String key, String defaultValue) {
        String result = defaultValue;
        if (configuration.containsKey(key)) {
            result = configuration.get(key);
        }

        return result;
    }

    public List<Class<?>> getPersistentClasses() {
        if (persistentClasses.isEmpty()) {
            // Force configuration load
            getHibernateConfiguration();
        }
        return persistentClasses;
    }

    public Configuration getHibernateConfiguration() {
        if (hibernateConfiguration == null) {
            hibernateConfiguration = new Configuration();

            // ajout des repertoires contenant les mappings hibernate
            String[] dirs = getProperty(
                    TopiaConfigurationConstants.CONFIG_PERSISTENCE_DIRECTORIES, "").split(",");
            for (String dir : dirs) {
                dir = dir.trim();
                if (StringUtils.isNotEmpty(dir)) {
                    if (log.isDebugEnabled()) {
                        log.debug("Load persistence from dir : " + dir);
                    }
                    hibernateConfiguration.addDirectory(new File(dir));
                }
            }

            // ajout des classes dites persistentes
            Set<Class<?>> hibernatePersistanceClasses = new HashSet<Class<?>>();
            for (TopiaService service : topiaServiceSupport.getServices().values()) {
                Class<?>[] classes = service.getPersistenceClasses();

                // certains service n'ont pas de classe persistantes
                if (classes != null) {
                    // sletellier 20110411 : http://www.nuiton.org/issues/show/1454
                    hibernatePersistanceClasses.addAll(Arrays.asList(classes));
//                        for (Class<?> clazz : classes) {
//                            hibernateConfiguration.addClass(clazz);
//                        }
                }
            }

            String listPersistenceClasses = getProperty(
                    TopiaConfigurationConstants.CONFIG_PERSISTENCE_CLASSES, "");

            String[] classes = listPersistenceClasses.split(",");
            for (String classname : classes) {
                classname = classname.trim();
                if (StringUtils.isNotEmpty(classname)) {
                    if (log.isDebugEnabled()) {
                        log.debug("Load persistent class : " + classname);
                    }

                    // XXX echatellier 20111007 ce qui est dommage ici, c'est
                    // la definition de cette classe ne sert a rien (apart security)
                    // car pour hibernate hibernateConfiguration.addClass(persistanceClass)
                    // il ne se sert pas de la classe en fait et fait seulement
                    // un  classname.replace( '.', '/' ) + ".hbm.xml";
                    // pour obtenir le mapping et la reinstancier ensuite

                    Class<?> clazz;
                    try {
                        clazz = Class.forName(classname);
                    } catch (ClassNotFoundException eee) {
                        if (log.isDebugEnabled()) {
                            log.debug("Class " + classname + " not found");
                        }
                        throw new TopiaNotFoundException(
                                String.format("Persistence class %1$s not found",
                                        classname));
                    }
                    persistentClasses.add(clazz);

                    // sletellier 20110411 : http://www.nuiton.org/issues/show/1454
//                        hibernateConfiguration.addClass(clazz);
                    hibernatePersistanceClasses.add(clazz);
                }
            }

            // sletellier 20110411 : http://www.nuiton.org/issues/show/1454
            // Add persistence classes in hibernate config
            for (Class<?> persistenceClass : hibernatePersistanceClasses) {
                hibernateConfiguration.addClass(persistenceClass);
            }

            Properties prop = new Properties();
            prop.putAll(hibernateConfiguration.getProperties());
            prop.putAll(configuration);

            // Strange behavior, all properties are already loaded from
            // constructor. Difficult to use this behavior, need to have
            // TOPIA_PERSISTENCE_PROPERTIES_FILE in config.
            Properties propertiesFromClasspath =
                    TopiaUtil.getProperties(getProperty(TopiaConfigurationConstants.CONFIG_PERSISTENCE_PROPERTIES_FILE));

            if (!propertiesFromClasspath.isEmpty()) {
                if (log.isDebugEnabled()) {
                    log.debug("Load properties from file : " +
                            propertiesFromClasspath);
                }
                prop.putAll(propertiesFromClasspath);
            }

            hibernateConfiguration.setProperties(prop);

            // tchemit 2011-05-26 When using hibernate > 3.3, need to make sure all mappings are loaded (the one from directory files are not still done).
            hibernateConfiguration.buildMappings();
        }
        return hibernateConfiguration;
    }

    /**
     * Method to extract from the given Hibernate SessionFactory a working instance of StandardServiceRegistry
     * <p/>
     * IMPORTANT : As much as possible, prefer using the
     * {@link #getSessionFactoryServiceRegistry(org.hibernate.SessionFactory)} mthod instead of the current one because
     * the SessionFactoryServiceRegistry is a child of the StandardServiceRegistry
     * <p/>
     * NB: This method is static to make sure it does not depend on the current instance
     *
     * @param sessionFactory the Hibernate's SessionFactory instance
     * @return the StandardServiceRegistry instance used by the given SessionFactory
     */
    protected static StandardServiceRegistry getStandardServiceRegistry(SessionFactory sessionFactory) {

        // AThimel 03/04/14 The next two lines are the good way to get the StandardServiceRegistry in Hibernate 4.3
        SessionFactory.SessionFactoryOptions sessionFactoryOptions = sessionFactory.getSessionFactoryOptions();
        StandardServiceRegistry result = sessionFactoryOptions.getServiceRegistry();

        return result;
    }

    /**
     * Method to extract from the given Hibernate SessionFactory a working instance of SessionFactoryServiceRegistry
     * <p/>
     * IMPORTANT : If possible, prefer using this method instead of
     * {@link #getStandardServiceRegistry(org.hibernate.SessionFactory)} because the SessionFactoryServiceRegistry is a
     * child of the StandardServiceRegistry
     * <p/>
     * NB: This method is static to make sure it does not depend on the current instance
     *
     * @param sessionFactory the Hibernate's SessionFactory instance
     * @return the SessionFactoryServiceRegistry instance used by the given SessionFactory
     */
    protected static SessionFactoryServiceRegistry getSessionFactoryServiceRegistry(SessionFactory sessionFactory) {

        // AThimel 03/04/14 The next two lines are the good way to get the SessionFactoryServiceRegistry in Hibernate 4.3
        SessionFactoryImplementor sessionFactoryImplementor = (SessionFactoryImplementor) sessionFactory;
        SessionFactoryServiceRegistry result = (SessionFactoryServiceRegistry)sessionFactoryImplementor.getServiceRegistry();

        return result;
    }

    /**
     * Method to get an Hibernate service instance from a given Hibernate SessionFactory
     * <p/>
     * NB: This method is static to make sure it does not depend on the current instance
     *
     * @param sessionFactory the Hibernate's SessionFactory instance
     * @param serviceClass   the expected service class
     * @return the found service instance
     * @throws org.hibernate.service.UnknownServiceException Indicates the service was not known.
     * @see org.hibernate.service.ServiceRegistry#getService(Class)
     */
    public static <S extends Service> S getHibernateService(SessionFactory sessionFactory, Class<S> serviceClass) {

        // Hibernate 4.3.x : prefer using the SessionFactoryServiceRegistry method instead of StandardServiceRegistry
        // because SessionFactoryServiceRegistry is a child of the StandardServiceRegistry
        ServiceRegistry serviceRegistry = getSessionFactoryServiceRegistry(sessionFactory);

        S result = serviceRegistry.getService(serviceClass);
        return result;
    }

    public void close() {
        if (hibernateSessionFactory != null) {
            Preconditions.checkState(!hibernateSessionFactory.isClosed());

            // close connection provider if possible (http://nuiton.org/issues/2757)
            ConnectionProvider service = getHibernateService(hibernateSessionFactory, ConnectionProvider.class);

            if (service instanceof Stoppable) {
                Stoppable stoppable = (Stoppable) service;
                stoppable.stop();
            }

            hibernateSessionFactory.close();
        }
    }

    public SessionFactory getSessionFactory() {

        if (hibernateSessionFactory == null) {

            Properties properties = getHibernateConfiguration().getProperties();

            // Use the next 2 lines if bootstrap customization is needed (classloader, autoclose, ...)
            // {@see org.hibernate.boot.registry.internal.BootstrapServiceRegistryImpl)
//            BootstrapServiceRegistry bootstrap = new BootstrapServiceRegistryBuilder().build();
//            StandardServiceRegistryBuilder builder = new StandardServiceRegistryBuilder(bootstrap);

            StandardServiceRegistryBuilder builder = new StandardServiceRegistryBuilder();
            StandardServiceRegistry standardServiceRegistry = builder.applySettings(properties).build();
            hibernateSessionFactory = getHibernateConfiguration().buildSessionFactory(standardServiceRegistry);

            EventListenerRegistry eventListenerRegistry = getHibernateService(hibernateSessionFactory, EventListenerRegistry.class);

            TopiaHibernateEventListener listener = new TopiaHibernateEventListener(sessionRegistry);
            eventListenerRegistry.appendListeners(EventType.PRE_INSERT, listener);
            eventListenerRegistry.appendListeners(EventType.PRE_LOAD, listener);
            eventListenerRegistry.appendListeners(EventType.PRE_UPDATE, listener);
            eventListenerRegistry.appendListeners(EventType.PRE_DELETE, listener);
            eventListenerRegistry.appendListeners(EventType.POST_INSERT, listener);
            eventListenerRegistry.appendListeners(EventType.POST_LOAD, listener);
            eventListenerRegistry.appendListeners(EventType.POST_UPDATE, listener);
            eventListenerRegistry.appendListeners(EventType.POST_DELETE, listener);

            // following listeners must be called before hibernate
            eventListenerRegistry.prependListeners(EventType.SAVE_UPDATE, listener);
        }
        return hibernateSessionFactory;
    }

}
