/**
 * *##% 
 * ToPIA :: Persistence
 * Copyright (C) 2004 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package org.nuiton.topia.persistence.util;


import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.InvocationTargetException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * A implementation fo the {@link org.nuiton.topia.persistence.util.Loador} contract using
 * {@link PropertyDescriptor}.
 * <p/>
 * Some factory methods are defined to simplify the generic cast, prefer used them
 * instead of the (protected) constructor.
 *
 * @param <E> type of bean to bind
 * @author chemit
 */
public class BeanPropertyLoador<E> implements Loador<E> {

    public static <E> BeanPropertyLoador<E> newLoador(Class<E> klass, String... properties) {
        return new BeanPropertyLoador<E>(klass, properties);
    }
    private static final long serialVersionUID = 1L;
    protected static final Integer ZERO = 0;
    protected static final Float ZEROF = 0f;
    protected static final Long ZEROL = 0l;
    protected static final Double ZEROD = 0.;
    protected static final Byte ZEROB = 0;
    protected List<String> properties;
    protected transient PropertyDescriptor[] descriptors;
    protected Class<E> klass;

    protected BeanPropertyLoador(Class<E> klass, String... properties) {
        this.properties = java.util.Collections.unmodifiableList(Arrays.asList(properties));
        this.klass = klass;
    }

    public List<String> getProperties() {
        return properties;
    }

    public PropertyDescriptor[] getDescriptors() {
        if (descriptors == null) {
            try {
                descriptors = new PropertyDescriptor[properties.size()];
                for (PropertyDescriptor propertydescriptor : Introspector.getBeanInfo(klass).getPropertyDescriptors()) {
                    int index = properties.indexOf(propertydescriptor.getName());
                    if (index != -1) {
                        descriptors[index] = propertydescriptor;
                    }
                }
            } catch (IntrospectionException e) {
                throw new RuntimeException(e);
            }
        }
        return descriptors;
    }

    @Override
    public Map<String, Object> obtainProperties(E from) {
        if (from == null) {
            return java.util.Collections.emptyMap();
        }
        Map<String, Object> result = new HashMap<String, Object>();
        for (PropertyDescriptor descriptor : getDescriptors()) {
            Object read;
            try {
                read = descriptor.getReadMethod().invoke(from);
                if (read != null) {
                    result.put(descriptor.getName(), read);
                }
            } catch (IllegalAccessException e) {
                throw new RuntimeException(e);
            } catch (InvocationTargetException e) {
                throw new RuntimeException(e);
            }

        }
        return result;
    }

    @Override
    @Deprecated
    public Map<String, Object> obtainProgperties(E from) {
        return obtainProperties(from);
    }

    @Override
    public void load(E from, E dst, boolean tech) {
        if (from == null) {
            // reset all fields
            for (PropertyDescriptor descriptor : getDescriptors()) {
                Object read = TopiaEntityHelper.getNullValue(descriptor.getPropertyType());
                setProperty(dst, descriptor, read);
            }
        } else {
            // set all fields from
            for (PropertyDescriptor descriptor : getDescriptors()) {
                try {
                    Object value = descriptor.getReadMethod().invoke(from);
                    setProperty(dst, descriptor, value);
                } catch (IllegalAccessException e) {
                    throw new RuntimeException(e);
                } catch (InvocationTargetException e) {
                    throw new RuntimeException(e);
                }
            }
        }
    }

    protected void setProperty(E dst, PropertyDescriptor descriptor, Object value) {
        try {
            descriptor.getWriteMethod().invoke(dst, value);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * @param type type to test
     * @return the null value for the given type
     * @deprecated use {@link TopiaEntityHelper#getNullValue(Class)}
     */
    @Deprecated
    protected static Object getNullValue(Class<?> type) {
        return TopiaEntityHelper.getNullValue(type);
    }

    /**
     * @param value the value to test
     * @return <code>true</code> if  value is null
     * @deprecated use {@link TopiaEntityHelper#isNullValue(Object)}
     */
    @Deprecated
    public static boolean isNullValue(Object value) {
        return TopiaEntityHelper.isNullValue(value);
    }

    protected void checkProperties() {
        PropertyDescriptor[] d = getDescriptors();
        for (int i = 0; i < d.length; i++) {
            PropertyDescriptor descriptor = d[i];
            if (descriptor == null) {
                throw new IllegalStateException("could not find descriptor for property " + properties.get(i) + " on " + klass);
            }
            if (descriptor.getReadMethod() == null) {
                throw new IllegalStateException("property " + properties.get(i) + " is not readable on " + klass);
            }
            if (descriptor.getWriteMethod() == null) {
                throw new IllegalStateException("property " + properties.get(i) + " is not writable on " + klass);
            }
        }
    }
}
