/*
 * #%L
 * Nuiton Utils :: Nuiton Profiling
 * %%
 * Copyright (C) 2006 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.profiling;

import com.sun.net.httpserver.Headers;
import com.sun.net.httpserver.HttpExchange;
import com.sun.net.httpserver.HttpHandler;
import com.sun.net.httpserver.HttpServer;
import com.sun.net.httpserver.spi.HttpServerProvider;
import java.io.ByteArrayOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.Writer;
import java.lang.reflect.Method;
import java.net.InetSocketAddress;
import java.net.URI;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.Stack;
import java.util.TreeSet;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DurationFormatUtils;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.After;
import org.aspectj.lang.annotation.AfterThrowing;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;

/**
 * Permet de tracer les appels aux methodes.
 *
 * <h1>Mise en place</h1>
 * 
 * Il faut mettre dans les dependances Maven le nuiton-profiling
 * <pre>
    $lt;dependency$gt;
      $lt;groupId$gt;org.nuiton$lt;/groupId$gt;
      $lt;artifactId$gt;nuiton-profiling$lt;/artifactId$gt;
      $lt;version$gt;2.7$lt;/version$gt;
    $lt;/dependency$gt;
 * </pre>
 *
 * <p/>
 * Pour l'utiliser il faut définir un fichier XML aop.xml dans le repertoire
 * META-INF ou WEB-INF qui intercepte les methodes souhaiter.
 * <pre>
 * &lt;!DOCTYPE aspectj PUBLIC "-//AspectJ//DTD//EN" "http://www.eclipse.org/aspectj/dtd/aspectj.dtd"$gt;
$lt;aspectj$gt;
  $lt;weaver options="-verbose"$gt;
    $lt;exclude within="fr.insee..*CGLIB*"/$gt;
  $lt;/weaver$gt;
  $lt;aspects$gt;
    $lt;concrete-aspect name="org.nuiton.profiling.NuitonTraceAppAspect"
                     extends="org.nuiton.profiling.NuitonTrace"$gt;
      $lt;pointcut name="executeMethod"
                expression="
      execution(* org.nuiton..*(..))
   || execution(* org.chorem..*(..))
   || execution(* org.apache..*(..))
   || execution(* ognl..*(..))"/$gt;
    $lt;/concrete-aspect$gt;
  $lt;/aspects$gt;
$lt;/aspectj$gt;
 * </pre>
 * Ensuite il faut lancer la JVM avec l'option
 * <pre>
 * -javaagent:path/to/aspectjweaver-1.7.1.jar
 * </pre>
 * normalement il se trouve dans le repo maven dans le repertoire
 * <li>$MAVEN_REPO/org/aspectj/aspectjweaver/1.7.1/aspectjweaver-1.7.1.jar
 *
 * <h1>Utilisation</h1>
 *
 * Il y a 3 modes: interactif, non interactif et par programmation
 *
 * <h2>Mode non interactif</h2>
 *
 * Il faut mettre le nom d'un fichier dans la variable d'environnnement
 * {@link #AUTO_SAVE_FILENAME_OPTION}. Dans ce cas lors de l'arrêt de la JVM
 * les statistiques sont ecrites dans ce fichier.
 *
 * <h2>Mode interactif</h2>
 *
 * Il est possible de demander a a NuitonTrace lorsqu'il s'initialise de demarrer
 * un serveur Web qui permet de voir et recuperer les donnees statistiques.
 * Pour cela il faut indique sur quel port le serveur doit attendre via la
 * variable d'environnnement {@link #PORT_OPTION}
 *
 * <h2>Par programmation</h2>
 * 
 * Pour afficher les statistiques dans votre programme
 * <li> log.info(NuitonTrace.getStatisticsAndClear());
 * <li> NuitonTrace.printStatistiqueAndClear();
 *
 * <h1>Autre mode de fonctionnement</h1>
 *
 * Il doit être possible, plutot que d'écrire un fichier XML, de sous classer
 * NuitonTrace en ajoutant par exemple
 * <p/>
 * <pre>
 * \@Expression( "execution(* fr.ifremer.isisfish.datastore.ResultStorage.*(..))" +
 * "|| execution(* fr.ifremer.isisfish.aspect.Cache.*(..))" +
 * "|| execution(* fr.ifremer.isisfish.aspect.Trace.*(..))" +
 * "|| execution(* org.nuiton.topia..*(..))" +
 * "|| execution(* org.nuiton.math.matrix..*(..))" +
 * "|| execution(* fr.ifremer.isisfish.types..*(..))" +
 * "|| execution(* org.apache.commons.collections..*(..))"
 * )
 * Pointcut executeMethod;
 * </pre>
 * <p/>
 *
 * @author bpoussin <poussin@codelutin.com>
 * @author tchemit <chemit@codelutin.com>
 */
@Aspect
public abstract class NuitonTrace {

    /**
     * constante determinant le nom de la variable d'environnement a lire
     * pour connaitre le port a utiliser pour le serveur web
     * @since 2.7
     */ // ne doit pas contenir de '.' ou de '-' sinon elle est non utiliable en variable d'env
    final static public String PORT_OPTION = "nuitonprofiling_webport";
    /**
     * constante determinant le nom de la variable d'environnement a lire
     * pour connaitre le nom du fichier de sauvegarde a la sortie de la JVM
     * @since 2.7
     */ // ne doit pas contenir de '.' ou de '-' sinon elle est non utiliable en variable d'env
    final static public String AUTO_SAVE_FILENAME_OPTION = "nuitonprofiling_autosavefile";
    /**
     * Indique si nuiton trace a deja ete initialise ou non
     * @since 2.7
     */
    static protected boolean initilized = false;

    static private final List<NuitonTrace> instances = new ArrayList<NuitonTrace>();

    // poussin 20110824: ne pas mettre de logger, car sinon, ca pose des problemes
    // avec l'AOP. NoClassDefFound :(
    /** to use log facility, just put in your code: log.info("..."); */
//    static private Log log = LogFactory.getLog(NuitonTrace.class);

    /** array : [call's numbers, call do, min time, max time, total time, total time with child] */
    protected Map<Method, Statistic> statistics = new LinkedHashMap<Method, Statistic>();

    protected Map<Method, Caller<Method>> callers = new HashMap<Method, Caller<Method>>();

    /**
     * array : [nest method call, start time, start time with child]
     * 
     * On ne melange pas les stack entre les threads, sinon les resultats
     * ne veulent plus rien dire car toutes les methodes des threads sont
     * melangees
     */
    protected static final ThreadLocal<Stack<Call>> callStack = new ThreadLocal<Stack<Call>> () {
        @Override
        protected Stack<Call> initialValue() {
            Stack<Call> result = new Stack<Call>();
            return result;
        }

    };

    public NuitonTrace() {
        instances.add(this);
    }

    /**
     * Retourne les statistiques d'appele pour la methode passee en parametre
     *
     * @param method
     * @return
     */
    public Statistic getStatistics(Method method) {
        Statistic result = statistics.get(method);
        if (result == null) {
            result = new Statistic(method);
            statistics.put(method, result);
        }
        return result;
    }

    /**
     * Retourne la liste des appelant pour la methode passee en parametre
     * 
     * @param method
     * @return
     */
    public Caller<Method> getCallers(Method method) {
        Caller<Method> result = callers.get(method);
        if (result == null) {
            result = new Caller<Method>();
            callers.put(method, result);
        }
        return result;
    }

    // abstract pointcut: no expression is defined
    @Pointcut
    abstract void executeMethod();

    @Before("executeMethod() && !within(org.nuiton.profiling.NuitonTrace.*)")
    public void traceBeforeExecute(JoinPoint jp) {
        // ajout dans le stack 
        Method method = ((MethodSignature) jp.getSignature()).getMethod();
        long current = System.nanoTime();

        Call stackItem = new Call(method, 0, current, current);
        callStack.get().push(stackItem);
    }

    @AfterThrowing("executeMethod() && !within(org.nuiton.profiling.NuitonTrace.*)")
    public void traceAfterThrowingExecute(JoinPoint jp) {
        // si une exeption est leve, il faut faire la meme chose
        traceAfterExecute(jp);
    }

    @After("executeMethod() && !within(org.nuiton.profiling.NuitonTrace.*)")
    public void traceAfterExecute(JoinPoint jp) {
        Method method = ((MethodSignature) jp.getSignature()).getMethod();

        long current = System.nanoTime();

        Stack<Call> callStack = this.callStack.get();

        if (callStack.isEmpty()) {
//            log.warn("Empty stack in afterExecute for method " + method.getName());
        } else {
            Call stackItem = callStack.pop();
            long timeSpent = current - stackItem.timeStart;
            long timeSpentNestMethod = current - stackItem.timeStartNestMethod;

            Statistic stat = getStatistics(method);
            stat.call++; // add +1 to call number
            stat.callNestMethod += stackItem.callNestMethod;
            stat.timeTotal += timeSpent;
            stat.timeTotalNestMethod += timeSpentNestMethod;
            if (stat.timeMin > timeSpent) {
                stat.timeMin = timeSpent;
            }
            if (stat.timeMax < timeSpent) {
                stat.timeMax = timeSpent;
            }

            if (!callStack.isEmpty()) {
                Call parent = callStack.peek();
                parent.callNestMethod++;  // add +1 to call number nest method
                parent.timeStart += timeSpentNestMethod; // remove to time all time spent in nest method (yes + to remove :)

                getCallers(method).add(parent.method);
            }
        }

        // ajouter le delta de temps dans le temps passé dans la méthod

        // il faud garder le temps passé dans l'appel d'autre methode de la stack
        // --> A
        //   =========
        //   --> B
        //   <-- B
        //   =========
        //   --> C
        //     --> D
        //     <-- D
        //   <-- C  
        //   =========
        // <-- A

        // le temps reellement passé dans A est representé par les =====
    }

    /**
     * Method used to get all keys and prevent java.util.ConcurrentModificationException
     * @param statistics
     * @return 
     */
    static public Method[] getKeys(Map<Method, Statistic> statistics) {
        Method[] result = statistics.keySet().toArray(new Method[statistics.size()]);
        return result;
    }

    /** @return les statistiques in CSV format*/
    static public String getStatisticsCSVAndClear() {
        StringBuilder result = new StringBuilder();
        result.append("method;")
                .append("call;")
                        // TIme is in nano not millis, we must divide by 1000000
                .append("min;")
                .append("mean;")
                .append("max;")
                .append("total;")
                .append("call_nest;")
                .append("total_with_nest;")
                .append("callers;")
                .append("\n");
        
        for (NuitonTrace trace : instances) {
            Method[] keys = getKeys(trace.statistics);
            for (Method method : keys) {
                Statistic stat = trace.getStatistics(method);
                Caller<Method> caller = trace.getCallers(method);
                long meanTime = stat.timeTotal / stat.call;
                result.append(method)
                        .append(";").append(stat.call)
                        // Time is in nano not millis, we must divide by 1000000
                        .append(";").append(DurationFormatUtils.formatDuration(stat.timeMin / 1000000, "s'.'S"))
                        .append(";").append(DurationFormatUtils.formatDuration(meanTime / 1000000, "s'.'S"))
                        .append(";").append(DurationFormatUtils.formatDuration(stat.timeMax / 1000000, "s'.'S"))
                        .append(";").append(DurationFormatUtils.formatDuration(stat.timeTotal / 1000000, "s'.'S"))
                        .append(";").append(stat.callNestMethod)
                        .append(";").append(DurationFormatUtils.formatDuration(stat.timeTotalNestMethod / 1000000, "s'.'S"))
                        .append(";").append(caller)
                        .append("\n");
            }
            trace.statistics.clear();
            trace.callers.clear();
        }
        // Poussin 20110826 on ne vide pas les instances mais les stats
//        instances.clear();
        return result.toString();
    }

    /** 
     * @return les statistiques in json format
     * @since 2.7
     */
    static public String getStatisticsJson() {
        StringBuilder result = new StringBuilder();
        result.append("{");

        String separator = "";
        for (NuitonTrace trace : instances) {
            Method[] keys = getKeys(trace.statistics);
            for (Method method : keys) {
                Statistic stat = trace.getStatistics(method);
                Caller<Method> caller = trace.getCallers(method);
                long meanTime = stat.timeTotal / stat.call;
                result.append(separator);
                separator = ","; // separtor est vide seulement la 1ere fois
                result.append("'").append(method).append("': {");
                result.append("'method':").append("'").append(method).append("'")
                        .append(",")
                        .append("'call':").append(stat.call)
                        .append(",")
                        // Time is in nano not millis, we must divide by 1000000
                        .append("'min':").append(DurationFormatUtils.formatDuration(stat.timeMin / 1000000, "s'.'S"))
                        .append(",")
                        .append("'mean':").append(DurationFormatUtils.formatDuration(meanTime / 1000000, "s'.'S"))
                        .append(",")
                        .append("'max':").append(DurationFormatUtils.formatDuration(stat.timeMax / 1000000, "s'.'S"))
                        .append(",")
                        .append("'total':").append(DurationFormatUtils.formatDuration(stat.timeTotal / 1000000, "s'.'S"))
                        .append(",")
                        .append("'call_nest':").append(stat.callNestMethod)
                        .append(",")
                        .append("'total_with_nest':").append(DurationFormatUtils.formatDuration(stat.timeTotalNestMethod / 1000000, "s'.'S"))
                        .append(",")
                        .append("'callers': [");
                for (Map.Entry<Method, Integer> c : caller) {
                    result.append("{")
                            .append("'caller':").append("'").append(c.getKey()).append("'")
                            .append(",")
                            .append("'call':").append(c.getValue())
                            .append("},");
                }
                result.append("]}").append("\n");
            }
        }
        result.append("}");
        // Poussin 20110826 on ne vide pas les instances mais les stats
//        instances.clear();
        return result.toString();
    }

    /**
     * @return clear all statistiques
     * @since 2.7
     */
    static public void clearStatistics() {
        for (NuitonTrace trace : instances) {
            trace.statistics.clear();
            trace.callers.clear();
        }
    }



    /** @return les statistiques */
    static public String getStatisticsAndClear() {
        StringBuilder result = new StringBuilder();
        for (NuitonTrace trace : instances) {
            result.append("--- Statistics ---\n");
            Method[] keys = getKeys(trace.statistics);
            for (Method method : keys) {
                Statistic stat = trace.getStatistics(method);
                long meanTime = stat.timeTotal / stat.call;
                result.append(method).append("\t")
                        .append(" call: ").append(stat.call)
                        // TIme is in nano not millis, we must divide by 1000000
                        .append(" min: ").append(DurationFormatUtils.formatDuration(stat.timeMin / 1000000, "s'.'S"))
                        .append(" mean: ").append(DurationFormatUtils.formatDuration(meanTime / 1000000, "s'.'S"))
                        .append(" max: ").append(DurationFormatUtils.formatDuration(stat.timeMax / 1000000, "s'.'S"))
                        .append(" total: ").append(DurationFormatUtils.formatDuration(stat.timeTotal / 1000000, "s'.'S"))
                        .append(" call_nest: ").append(stat.callNestMethod)
                        .append(" total_with_nest: ").append(DurationFormatUtils.formatDuration(stat.timeTotalNestMethod / 1000000, "s'.'S"))
                        .append(" callers: ").append(trace.getCallers(method))
                        .append("\n");
            }
            result.append("--------------------\n");
            trace.statistics.clear();
            trace.callers.clear();
        }
        return result.toString();
    }

    static public void printStatisticsAndClear() {
        printStatisticsAndClear(System.out);
    }

    static public void printStatisticsAndClear(PrintStream printer) {
        String stat = getStatisticsAndClear();
        if (stat != null && !"".equals(stat)) {
            printer.println(stat);
        }
    }

    /**
     * Statistique sur un appele de methode
     */
    protected static class Statistic {
        Method method;
        /** nombre d'appel */
        long call;
        /** nombre d'appel vers une autre method depuis cette methode */
        long callNestMethod;
        /** temps mini d'execution de cette methode (sans le temps des autres methodes) */
        long timeMin;
        /** temps max d'execution de cette methode (sans le temps des autres methodes) */
        long timeMax;
        /** temps total d'execution de cette methode (sans le temps des autres methodes) */
        long timeTotal;
        /** temps total d'execution de cette methode (avec le temps des autres methodes) */
        long timeTotalNestMethod;

        public Statistic(Method method) {
            this.method = method;
        }

    }

    /**
     * Represente un call de methode
     */
    protected static class Call {
        Method method;
        /** nombre d'appel vers une autre method depuis cette methode */
        long callNestMethod;
        /** heure de depart de l'appel a la methode (sans le temps des autres methodes) */
        long timeStart;
        /** heure de depart de l'appel a la methode (avec le temps des autres methodes) */
        long timeStartNestMethod;

        public Call(Method method, long callNestMethod, long timeStart, long timeStartNestMethod) {
            this.method = method;
            this.callNestMethod = callNestMethod;
            this.timeStart = timeStart;
            this.timeStartNestMethod = timeStartNestMethod;
        }

    }

    /**
     * Classe qui permet d'ajouter un element et compte le nombre de fois
     * que l'on a ajoute cet element.
     * 
     * exemple d'utilisation: on souhaite compter le nombre de fois qu'une methode
     * est appeler par une autre, on ajoute a chaque fois que quelqu'un fait
     * appele a elle ce quelqu'un.
     * 
     */
    protected static class Caller<E> implements Iterable<Map.Entry<E, Integer>> {

        protected Map<E, Integer> data = new HashMap<E, Integer>();
        
        /**
         * indique que l'on ajoute un element
         * @param e
         */
        public void add(E e) {
            Integer v = data.get(e);
            if (v == null) {
                v = 0;
            }
            data.put(e, v+1);
        }

        /**
         * Retourne la liste des objets ajouter avec le nombre de fois qu'ils
         * ont ete ajoute. Trie par le nombre de fois qu'ils ont ete ajoute,le
         * plus grand nombre en 1er
         * 
         * @return
         */
        @Override
        public String toString() {

            StringBuilder result = new StringBuilder();

            for (Map.Entry<E, Integer> e : this) {
                result.append(e.getValue()).append("=[").append(e.getKey()).append("]").append(",");
            }
            if (result.length() > 0) {
                // on supprime la derniere virgule en trop
                result.deleteCharAt(result.length()-1);
            }

            return result.toString();
        }

        public Iterator<Entry<E, Integer>> iterator() {
            Comparator<Map.Entry<E, Integer>> comparator = new Comparator<Map.Entry<E, Integer>>() {
                public int compare(Entry<E, Integer> o1, Entry<E, Integer> o2) {
                    // on inverse, car on veut le plus grand en 1er
                    return o2.getValue().compareTo(o1.getValue());
                }
            };

            Set<Map.Entry<E, Integer>> entries = data.entrySet();
            Set<Map.Entry<E, Integer>> sorted =
                    new TreeSet<Map.Entry<E, Integer>>(comparator);
            sorted.addAll(entries);
            return sorted.iterator();
        }
    }

    /**
     * Demarre le service web sur le port demande
     * @param port un port valide sur lequel attendre
     * @since 2.7
     */
    static protected void startWebService(int port) {
        try {
            HttpServerProvider provider = HttpServerProvider.provider();
            InetSocketAddress inet = new InetSocketAddress(port);
            HttpServer server = provider.createHttpServer(inet, 0);

            server.createContext("/", new WebHandler());
            server.createContext("/data", new DataHandler());
            server.createContext("/api", new ApiHandler());
            server.start();
            System.out.println("Nuiton profiling Server is listening on " + inet );

        } catch(Throwable eee) {
            System.err.println("Can't start web service on port " + port);
            eee.printStackTrace(System.err);
        }
    }

    /**
     * Handler pour le serveur web qui gere le demande de statistique
     * @since 2.7
     */
    static class WebHandler implements HttpHandler {

        static final public String filepath = "/org/nuiton/profiling/web/";
        
        public void handle(HttpExchange exchange) throws IOException {
            String requestMethod = exchange.getRequestMethod();
            if (requestMethod.equalsIgnoreCase("GET")) {
                String path = exchange.getRequestURI().getPath();
                String file = StringUtils.substringAfterLast(path, "/");
                file = StringUtils.defaultIfBlank(file, "index.html");
                URL resource = getClass().getResource(filepath + file);
                if (resource == null) {
                    exchange.sendResponseHeaders(404, 0);
                    exchange.close();
                } else {
                    String ext = StringUtils.substringAfterLast(file, ".");
                    String type;
                    if (StringUtils.equalsIgnoreCase(ext, "mf")) {
                        type = "text/cache-manifest";
                    } else if (StringUtils.equalsIgnoreCase(ext, "ico")) {
                        type = "image/x-icon";
                    } else if (StringUtils.equalsIgnoreCase(ext, "js")) {
                        type = "text/javascript";
                    } else {
                        // html, css, js
                        type = "text/" + ext;
                    }
                    Headers responseHeaders = exchange.getResponseHeaders();
                    responseHeaders.set("Content-Type", type);
                    exchange.sendResponseHeaders(200, 0);

                    InputStream in = resource.openStream();
                    ByteArrayOutputStream content = new ByteArrayOutputStream();
                    IOUtils.copy(in, content);

                    OutputStream responseBody = exchange.getResponseBody();
                    responseBody.write(content.toByteArray());
                    responseBody.close();
                    IOUtils.closeQuietly(in);
                    IOUtils.closeQuietly(content);
                }
            }

        }
    }

    /**
     * Handler pour le serveur web qui gere le demande de statistique
     * @since 2.7
     */
    static class DataHandler implements HttpHandler {
        /**
         * Permet de recupere le parametre callback de l'URL, utilise pour le jsonp
         * @param exchange
         * @return
         */
        protected String getCallback(HttpExchange exchange) {
            String result = null;
            URI uri = exchange.getRequestURI();
            String query = uri.getQuery();
            if (query != null) {
                String pairs[] = query.split("[&]");

                for (String pair : pairs) {
                    String param[] = pair.split("[=]", 2);
                    if (param.length == 2 && "callback".equalsIgnoreCase(param[0])) {
                        result = param[1];
                        break;
                    }
                 }
            }
            return result;
        }

        public void handle(HttpExchange exchange) throws IOException {
            String requestMethod = exchange.getRequestMethod();
            if (requestMethod.equalsIgnoreCase("GET")) {
                Headers responseHeaders = exchange.getResponseHeaders();
                responseHeaders.set("Content-Type", "text/javascript");
                exchange.sendResponseHeaders(200, 0);

                String jsoncallback = getCallback(exchange);
                String json = NuitonTrace.getStatisticsJson();
                if (jsoncallback != null) {
                    json = jsoncallback + "(" + json + ")";
                }

                OutputStream responseBody = exchange.getResponseBody();
                responseBody.write(json.getBytes());
                responseBody.close();
            }
        }
    }

    /**
     * Handler pour le serveur web qui gere l'api:
     * <li>clear
     * @since 2.7
     */
    static class ApiHandler implements HttpHandler {

        public void handle(HttpExchange exchange) throws IOException {
            String requestMethod = exchange.getRequestMethod();
            if (requestMethod.equalsIgnoreCase("GET")) {
                String path = exchange.getRequestURI().getPath();
                if (StringUtils.endsWithIgnoreCase(path, "clear")) {
                    Headers responseHeaders = exchange.getResponseHeaders();
                    responseHeaders.set("Content-Type", "text/plain");
                    exchange.sendResponseHeaders(200, 0);

                    clearStatistics();

                    OutputStream responseBody = exchange.getResponseBody();
                    responseBody.write("ok".getBytes());
                    responseBody.close();
                } else {
                    exchange.sendResponseHeaders(404, 0);
                    exchange.close();
                }
            }

        }
    }

    /**
     * Permet de lire les options de la ligne de commande Java (-D) et si non
     * trouver de rechercher dans les variables d'environnement
     * @param optionName
     * @return la valeur de l'option ou null si non trouve
     * @since 2.7
     */
    static protected String getOption(String optionName) {
        String result = System.getProperty(optionName);
        if (result == null) {
            result = System.getenv(optionName);
        }
        return result;
    }

    /**
     * Recherche la configuration dans les variables d'environnement et les
     * utilise pour mettre en place ce qui est demande par l'utilisateur
     * @since 2.7
     */
    static protected void init() {
        if (initilized) {
            return;
        }
        initilized = true;
        
        System.out.println("Init Nuiton Profiling ...");
        String portString = getOption(PORT_OPTION);
        System.out.println("NuitonProfiling web port: " + portString);
        if (portString != null) {
            try {
                int port = Integer.parseInt(portString);
                if (port > 0) {
                    startWebService(port);
                }
            } catch (Exception eee) {
                eee.printStackTrace(System.err);
                System.err.println("Can't parse port number: "+ portString);
            }
        }

        final String file = getOption(AUTO_SAVE_FILENAME_OPTION);
        System.out.println("NuitonProfiling auto save file: " + file);
        if (file != null) {
            Runtime.getRuntime().addShutdownHook(new Thread() {
                @Override
                public void run() {
                    Writer out = null;
                    try {
                        String stat = NuitonTrace.getStatisticsCSVAndClear();
                        out = new FileWriter(file);
                        out.write(stat);
                    } catch (IOException eee) {
                        eee.printStackTrace(System.err);
                        System.err.println("Can't write Statistic file: " + file);
                    } finally {
                        IOUtils.closeQuietly(out);
                    }
                }
            });
        }
    }

    /**
     * Constructeur static pour initialise nuiton trace
     * @since 2.7
     */
    static {
        // attention aucune classe utilisee dans le init ne doit utiliser
        // des classes surveille par NuitonTrace, sinon y'a un probleme d'init
        // l'AOP voulant surveillee une method, mais l'AOP n'est pas encore
        // initialise puisqu'on est dans l'init :(
        init();
    }

    /**
     * Force le lancement du serveur web, permet d'avoir le serveur sur une
     * machine pour monitorer une application qui est sur une autre. L'avantage
     * est de pouvoir centraliser les sauvegardes sur le meme domaine web
     * @param args le port a utiliser si vide utilise les variables d'environnement
     * et en dernier recourrs utilise le port 4488
     */
    static public void main(String... args) {
        System.out.println("Starting Nuiton Profiling ...");
        String portString;
        if (args != null && args.length > 0) {
            portString = args[0];
        } else {
            portString = getOption(PORT_OPTION);
        }
        if (portString == null) {
            portString = "4488";
        }
        System.out.println("NuitonProfiling web port: " + portString);
        if (portString != null) {
            try {
                int port = Integer.parseInt(portString);
                if (port > 0) {
                    startWebService(port);
                }
            } catch (Exception eee) {
                eee.printStackTrace(System.err);
                System.err.println("Can't parse port number: "+ portString);
            }
        }
    }
}
