/*
 * *##% 
 * Maven helper plugin
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package org.nuiton.io;

import org.nuiton.io.FileUpdaterHelper.JavaFileUpdater;
import java.io.File;
import org.junit.BeforeClass;
import org.junit.Test;
import org.nuiton.plugin.TestHelper;
import static org.junit.Assert.*;

/**
 * Test of a {@link JavaFileUpdater} on the test sourcepath and test-classpath
 *
 * @author chemit
 */
public class JavaFileUpdaterTest {

    static MirroredFileUpdater fileUpdater;
    static File sourceFile;
    static File mirrorFile;

    @BeforeClass
    public static void initClass() throws Exception {

        File basedir = TestHelper.getBasedir();

        File src = TestHelper.getFile(basedir, "src", "test", "resources");
        File dir = TestHelper.getFile(basedir, "target", "test-classes");

        fileUpdater = FileUpdaterHelper.newJavaFileUpdater(src, dir);

        // test source dir exists
        File sourceDirectory = fileUpdater.getSourceDirectory();
        assertTrue("sourcedir is null", sourceDirectory != null);
        assertTrue("sourcedir does not exists " + sourceDirectory, sourceDirectory.exists());

        // test destination dir exists
        File destinationDirectory = fileUpdater.getDestinationDirectory();
        assertTrue("destinationDirectory  is null", destinationDirectory != null);
        assertTrue("destinationDirectory  does not exists " + destinationDirectory, destinationDirectory.exists());

        sourceFile = TestHelper.getFile(fileUpdater.getSourceDirectory(), "org", "nuiton", "io", JavaDummy.class.getSimpleName() + ".java");

    }

    @Test
    public void testSourceFileIsUptoDate() throws Exception {

        assertTrue("fileUpdater was not init", fileUpdater != null);
        assertTrue("sourceFile was not init ", sourceFile != null);
        assertTrue("could not find source of this test " + sourceFile, sourceFile.exists());

        // java file is up to date since it was compiled to launch this test
        assertTrue(sourceFile + " should be uptodate", fileUpdater.isFileUpToDate(sourceFile));

        mirrorFile = fileUpdater.getMirrorFile(sourceFile);
    }

    @Test
    public void testSourceFileDoesNotHaveMirrorFile() throws Exception {

        assertTrue("fileUpdater was not init", fileUpdater != null);
        assertTrue("sourceFile was not init ", sourceFile != null);
        assertTrue("could not find source of this test " + sourceFile, sourceFile.exists());
        assertTrue("mirrorFile was not init ", mirrorFile != null);
        assertTrue("could not find mirrorFile " + mirrorFile, mirrorFile.exists());

        // now rename mirror Class, so source file is no more up to date
        File mirrorClassRenamed = new File(fileUpdater.getMirrorFile(sourceFile).getAbsolutePath() + "2");
        mirrorFile.renameTo(mirrorClassRenamed);

        // java file is no more up to date
        assertFalse(sourceFile + " should not be uptodate", fileUpdater.isFileUpToDate(sourceFile));

        // renmae mirror Class to his original name
        mirrorClassRenamed.renameTo(mirrorFile);

        // java file is now up to date (rename method does not affect lasmodified property on a file)
        assertTrue(sourceFile + " should be uptodate", fileUpdater.isFileUpToDate(sourceFile));
    }

    @Test
    public void testSourceFileIsNewerThanMirrorFile() throws Exception {

        assertTrue("fileUpdater was not init", fileUpdater != null);
        assertTrue("sourceFile was not init ", sourceFile != null);
        assertTrue("could not find source of this test " + sourceFile, sourceFile.exists());
        assertTrue("mirrorFile was not init ", mirrorFile != null);
        assertTrue("could not find mirrorFile " + mirrorFile, mirrorFile.exists());

        long sourceTime = sourceFile.lastModified();
        long mirrorTime = mirrorFile.lastModified();

        // make source file newer than mirror file
        sourceFile.setLastModified(mirrorTime + 10);

        // java file is no more up to date
        assertFalse(sourceFile + " should not be uptodate", fileUpdater.isFileUpToDate(sourceFile));

        // put back old time to source file
        sourceFile.setLastModified(sourceTime);

        // java file is now up to date (rename method does not affect lasmodified property on a file)
        assertTrue(sourceFile + " should be uptodate", fileUpdater.isFileUpToDate(sourceFile));
    }
}
